#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# === This file is part of Calamares - <http://github.com/calamares> ===
#
#   Copyright 2016, Artoo <artoo@manjaro.org>
#   Copyright 2016-2019, Philip Müller <philm@manjaro.org>
#
#   Calamares is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Calamares is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Calamares. If not, see <http://www.gnu.org/licenses/>.

import libcalamares

from libcalamares.utils import target_env_call, debug
from os.path import join
from subprocess import call


class GhtController:
    def __init__(self):
        self.__root = libcalamares.globalstorage.value("rootMountPoint")
        self.__bus = libcalamares.job.configuration.get('bus', [])
        self.__identifier = libcalamares.job.configuration.get(
            'identifier',
            []
            )
        self.__local = libcalamares.job.configuration['local']
        self.__repo = libcalamares.job.configuration['repo']
        self._driver = libcalamares.job.configuration['driver']

    @property
    def driver(self):
        return self._driver

    @driver.setter
    def driver(self, value):
        self._driver = value

    @property
    def root(self):
        return self.__root

    @property
    def local(self):
        return self.__local

    @property
    def repo(self):
        return self.__repo

    @property
    def identifier(self):
        return self.__identifier

    @property
    def bus(self):
        return self.__bus

    def umount(self, mp):
        call(["umount", "-l", join(self.root, mp)])

    def mount(self, mp):
        call(["mount", "-B", "/" + mp, join(self.root, mp)])

    def rmdir(self, dir):
        call(["rm", "-Rf", join(self.root, dir)])

    def mkdir(self, dir):
        call(["mkdir", "-p", join(self.root, dir)])

    def configure(self):
        cmd = ["garuda-hardware-tool", "--sync", "--noconfirm", "--" + str(self.driver)]
        if self.local:
            self.mkdir("opt/ght")
            self.mount("opt/ght")
            self.mount("var/lib/ght/db")
            cmd.extend(["--pacmanconfig", self.repo])

        self.mount("etc/resolv.conf")
        target_env_call(cmd)

        if self.local:
            self.umount("opt/ght")
            self.rmdir("opt/ght")
            self.umount("var/lib/ght/db")
        self.umount("etc/resolv.conf")

    def run(self):
        self.configure()

        return None


def run():
    """ Configuring the hardware """

    ght = GhtController()

    return ght.run()
