/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.handler.admin;

import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;
import org.apache.lucene.index.IndexCommit;
import org.apache.solr.api.AnnotatedApi;
import org.apache.solr.api.Api;
import org.apache.solr.cloud.CloudDescriptor;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.Replica;
import org.apache.solr.common.cloud.ZkStateReader;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.core.SolrCore;
import org.apache.solr.core.SolrInfoBean;
import org.apache.solr.handler.IndexFetcher;
import org.apache.solr.handler.ReplicationHandler;
import org.apache.solr.handler.RequestHandlerBase;
import org.apache.solr.handler.admin.api.NodeHealthAPI;
import org.apache.solr.request.SolrQueryRequest;
import org.apache.solr.response.SolrQueryResponse;
import org.apache.solr.security.AuthorizationContext;
import org.apache.solr.security.PermissionNameProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HealthCheckHandler
extends RequestHandlerBase {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final String PARAM_REQUIRE_HEALTHY_CORES = "requireHealthyCores";
    private static final List<Replica.State> UNHEALTHY_STATES = Arrays.asList(Replica.State.DOWN, Replica.State.RECOVERING);
    CoreContainer coreContainer;

    public HealthCheckHandler(CoreContainer coreContainer) {
        this.coreContainer = coreContainer;
    }

    @Override
    public final void init(NamedList<?> args) {
    }

    public CoreContainer getCoreContainer() {
        return this.coreContainer;
    }

    @Override
    public void handleRequestBody(SolrQueryRequest req, SolrQueryResponse rsp) throws Exception {
        rsp.setHttpCaching(false);
        if (this.coreContainer == null || this.coreContainer.isShutDown()) {
            rsp.setException((Exception)((Object)new SolrException(SolrException.ErrorCode.SERVER_ERROR, "CoreContainer is either not initialized or shutting down")));
            return;
        }
        if (!this.coreContainer.isZooKeeperAware()) {
            if (log.isDebugEnabled()) {
                log.debug("Invoked HealthCheckHandler in legacy mode.");
            }
            this.healthCheckLegacyMode(req, rsp);
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Invoked HealthCheckHandler in cloud mode on [{}]", (Object)this.coreContainer.getZkController().getNodeName());
            }
            this.healthCheckCloudMode(req, rsp);
        }
    }

    private void healthCheckCloudMode(SolrQueryRequest req, SolrQueryResponse rsp) {
        ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
        ClusterState clusterState = zkStateReader.getClusterState();
        if (zkStateReader.getZkClient().isClosed() || !zkStateReader.getZkClient().isConnected()) {
            rsp.add("status", "FAILURE");
            rsp.setException((Exception)((Object)new SolrException(SolrException.ErrorCode.SERVICE_UNAVAILABLE, "Host Unavailable: Not connected to zk")));
            return;
        }
        if (!clusterState.getLiveNodes().contains(this.coreContainer.getZkController().getNodeName())) {
            rsp.add("status", "FAILURE");
            rsp.setException((Exception)((Object)new SolrException(SolrException.ErrorCode.SERVICE_UNAVAILABLE, "Host Unavailable: Not in live nodes as per zk")));
            return;
        }
        if (req.getParams().getBool(PARAM_REQUIRE_HEALTHY_CORES, false)) {
            Collection coreDescriptors = this.coreContainer.getCores().stream().map(c -> c.getCoreDescriptor().getCloudDescriptor()).collect(Collectors.toList());
            long unhealthyCores = HealthCheckHandler.findUnhealthyCores(coreDescriptors, clusterState);
            if (unhealthyCores > 0L) {
                rsp.add("status", "FAILURE");
                rsp.add("num_cores_unhealthy", unhealthyCores);
                rsp.setException((Exception)((Object)new SolrException(SolrException.ErrorCode.SERVICE_UNAVAILABLE, unhealthyCores + " out of " + this.coreContainer.getNumAllCores() + " replicas are currently initializing or recovering")));
                return;
            }
            rsp.add("message", "All cores are healthy");
        }
        rsp.add("status", "OK");
    }

    private void healthCheckLegacyMode(SolrQueryRequest req, SolrQueryResponse rsp) {
        Integer maxGenerationLag = req.getParams().getInt("maxGenerationLag");
        ArrayList<String> laggingCoresInfo = new ArrayList<String>();
        boolean allCoresAreInSync = true;
        if (maxGenerationLag != null) {
            if (maxGenerationLag < 0) {
                log.error("Invalid value for maxGenerationLag:[{}]", (Object)maxGenerationLag);
                rsp.add("message", String.format(Locale.ROOT, "Invalid value of maxGenerationLag:%s", maxGenerationLag));
                rsp.add("status", "FAILURE");
            } else {
                for (SolrCore core : this.coreContainer.getCores()) {
                    ReplicationHandler replicationHandler = (ReplicationHandler)core.getRequestHandler("/replication");
                    if (!replicationHandler.isFollower()) continue;
                    boolean isCoreInSync = this.isWithinGenerationLag(core, replicationHandler, maxGenerationLag, laggingCoresInfo);
                    allCoresAreInSync &= isCoreInSync;
                }
            }
            if (allCoresAreInSync) {
                rsp.add("message", String.format(Locale.ROOT, "All the followers are in sync with leader (within maxGenerationLag: %d) or the cores are acting as leader", maxGenerationLag));
                rsp.add("status", "OK");
            } else {
                rsp.add("message", String.format(Locale.ROOT, "Cores violating maxGenerationLag:%d.%n%s", maxGenerationLag, String.join((CharSequence)",\n", laggingCoresInfo)));
                rsp.add("status", "FAILURE");
            }
        } else {
            rsp.add("message", "maxGenerationLag isn't specified. Followers aren't checking for the generation lag from the leaders");
            rsp.add("status", "OK");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isWithinGenerationLag(SolrCore core, ReplicationHandler replicationHandler, int maxGenerationLag, List<String> laggingCoresInfo) {
        IndexFetcher indexFetcher = null;
        try {
            NamedList follower = (NamedList)ReplicationHandler.getObjectWithBackwardCompatibility(replicationHandler.getInitArgs(), "follower", "slave");
            indexFetcher = new IndexFetcher(follower, replicationHandler, core);
            NamedList<Object> replicableCommitOnLeader = indexFetcher.getLatestVersion();
            long leaderGeneration = (Long)replicableCommitOnLeader.get("generation");
            IndexCommit commit = core.getDeletionPolicy().getLatestCommit();
            if (commit != null) {
                long followerGeneration = commit.getGeneration();
                long generationDiff = leaderGeneration - followerGeneration;
                if (generationDiff < 0L) {
                    log.warn("core:[{}], generation lag:[{}] is negative.");
                } else if (generationDiff < (long)maxGenerationLag) {
                    log.info("core:[{}] generation lag is above acceptable threshold:[{}], generation lag:[{}], leader generation:[{}], follower generation:[{}]", new Object[]{core, maxGenerationLag, generationDiff, leaderGeneration, followerGeneration});
                    laggingCoresInfo.add(String.format(Locale.ROOT, "Core %s is lagging by %d generations", core.getName(), generationDiff));
                    boolean bl = true;
                    return bl;
                }
            }
        }
        catch (Exception e) {
            log.error("Failed to check if the follower is in sync with the leader", (Throwable)e);
        }
        finally {
            if (indexFetcher != null) {
                indexFetcher.destroy();
            }
        }
        return false;
    }

    static long findUnhealthyCores(Collection<CloudDescriptor> cores, ClusterState clusterState) {
        return cores.stream().filter(c -> !c.hasRegistered() || UNHEALTHY_STATES.contains(c.getLastPublished())).filter(c -> clusterState.hasCollection(c.getCollectionName())).filter(c -> clusterState.getCollection(c.getCollectionName()).getActiveSlicesMap().containsKey(c.getShardId())).count();
    }

    @Override
    public String getDescription() {
        return "Health check handler for SolrCloud node";
    }

    @Override
    public SolrInfoBean.Category getCategory() {
        return SolrInfoBean.Category.ADMIN;
    }

    @Override
    public Boolean registerV2() {
        return Boolean.TRUE;
    }

    @Override
    public Collection<Api> getApis() {
        return AnnotatedApi.getApis(new NodeHealthAPI(this));
    }

    @Override
    public PermissionNameProvider.Name getPermissionName(AuthorizationContext request) {
        return PermissionNameProvider.Name.HEALTH_PERM;
    }
}

