// Copyright (c) blackshirt. All rights reserved.
// Use of this source code is governed by an MIT license
// that can be found in the LICENSE file.
// vtest build: has_modern_openssl?
module slhdsa

import encoding.hex

fn test_basic_slhdsa_signing_verifying() ! {
	mut pv := PrivateKey.new()!
	msg := 'msg'.bytes()

	sig := pv.sign(msg)!

	mut pb := pv.public_key()!
	verified := pb.verify(sig, msg)!
	assert verified

	pv.free()
	pb.free()
}

// Test for SLH-DSA signature generation, for SLH_DSA_SHA2_128s key
fn test_sha2_128_signature_generation() ! {
	for idx, item in samples_slh128s_generated_signatures {
		priv := hex.decode(item.priv)!
		entropy := hex.decode(item.entropy)!
		msg := hex.decode(item.msg)!
		sig := hex.decode(item.sig)!

		// key options
		k_opt := KeyOpts{
			kind: .sha2_128s
			flag: 2
			priv: priv
		}
		mut pv := PrivateKey.new(k_opt)!

		// signing (verifying) options
		s_opt := SignerOpts{
			encoding: 0
			entropy:  entropy
		}
		out_sig := pv.sign(msg, s_opt)!
		assert out_sig == sig

		mut pb := pv.public_key()!
		verified := pb.verify(out_sig, msg, s_opt)!
		assert verified

		pv.free()
		pb.free()
	}
}

struct TestSignatureGenerationSLHSHA2128s {
	priv    string
	entropy string
	msg     string
	sig     string
}

// This samples of data was copied and adapted into V opaque from
// https://boringssl.googlesource.com/boringssl.git/+/chromium-stable/crypto/slhdsa/slhdsa_siggen.txt
//
const samples_slh128s_generated_signatures = [
	TestSignatureGenerationSLHSHA2128s{
		priv:    '4329f96b412b5d5ca9c730c9d54e52595265f6fc3ea67c183aa7043d90e949b8fd320c8fc0fa6a23462483535f131b9ea265848e7bb5f4d3cfd99de2595382b3'
		entropy: '65ac888a4cbd14aa5533e8685b0cc0e1'
		msg:     '707951a4c5040af79e4dbb1d5614ea77ee58a1b16d8c17a40abb01499d1b65f5c55b517b603711e5db67f758cf6717aa10cee53b4417b4c1f4b1dcd6e9e65e96a68b856ebefb45d394e6c9214d0c55e692eb2ffbb10b804bc91076a50aa5cba4884564254ea99540c99647c4637ed26b57e77cb58be6db0ab0d418543b0f69db'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    '5c34d15e5931c27c976f7044d18f24e11a268267de70e8c86e1e90f888dd692fb534c67bf12f66252e772e66675c46de3aecaf7da2a98e7971b1a455eb4e1300'
		entropy: 'f93091ff5b082105513400726dfbd17b'
		msg:     '2d884fd72a2daf869220adb4abfd959ebbfff58b3e74676afdad958ef77b'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    '4f0b94c0e1d8abec0d22ad3f180e514115272ce1ab8252b0608da02659d84732a4bf46d8c3329e2e48594ab7970d1ff0e644e8ac35994f9308049ca1e108ff1e'
		entropy: 'c02b762bed67adc3a058487df6fabf42'
		msg:     '7ea1563d2d0e4d398028b06f1fab51f69962c6e84f9e916b655aeff9bcd2d811fcbfd1aacfe09515f4e45f73441046fc1b6912f1204d01504c251dc323421edf9bf57f65a86d60ab53e73650efdd91821d133bdab274067c257a182f8be9fc2b354e51'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    'f99d60c4ba723d4a81277aa8bdeec37b6fc27624faafef99f6e82256f351fae23d422fcf462f0f4a0df974a7ebcf5b85217fc357d9db5e5fa864d848636b5511'
		entropy: 'd45c2dcbb823f8db583a4038ca7af794'
		msg:     '90'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    '2174e8f49dab2ab631e641a5406fd710f772a575d2746c9c9b7a2b4217441f88bf830dd8d578bd710316bf443cca3f48e4e056030eaad73bdef022c2a0291fd3'
		entropy: '10bc13cc39a3d35de53c309a6ad1902c'
		msg:     '7f47df895ab1a0a777ab84002eef5844bcffb413dabe8fd6fc0027721be348207a59316f89dc88d587c3b22d8227b659595d898890516ab0b66966e387286a2ad90627e448b1a246884465456f463f63e3ae9b860e2127a23e7a202b221019ed082173c28a9907f4e398a9b7f4a82ad79488f4a116734c4cc9b2'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    'be4bfeaaf8e4fbe9bdc93b44d621380a48090ab3599d40d3bcbcfb50c0564d7b48f8846d7ba5fa5ef49151563e84bce614b947b2be7565f4fcf7ad7a5ee899f9'
		entropy: 'df096f9d0302657c0f9b6974b80c59eb'
		msg:     'eb29cb0687183747a27d51778eb7282fd52778778ea810131092cc90e4e3500d348908e88622a237c28d5309f70c7b895a2027262f6d94f4125129294ff9d4f6'
		sig:     '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'
	},
	TestSignatureGenerationSLHSHA2128s{
		priv:    '03d3b1c80c61d8f13ffa8ab1f2515d2071c4afd7d2b91d979cf8f4b6dd907ef2a0985f04c61e90ef3af41d6fbba798d070b5f35f53599f764ad53121ae24da8d'
		entropy: '426827d1935bb4642b404799bafc62d6'
		msg:     'f724'
		sig:     '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'
	},
]
