// SPDX-License-Identifier: GPL-3.0-or-later
// SPDX-FileCopyrightText: Andy Holmes <andrew.g.r.holmes@gmail.com>

#pragma once

#if !defined (VALENT_INSIDE) && !defined (VALENT_COMPILATION)
# error "Only <valent.h> can be included directly."
#endif

#include <stdint.h>

#include <glib.h>

G_BEGIN_DECLS

/**
 * VALENT_MAJOR_VERSION:
 *
 * The major version component of libvalent (e.g. `1` if for `1.2.3`)
 */
#define VALENT_MAJOR_VERSION (1)

/**
 * VALENT_MINOR_VERSION:
 *
 * The minor version component of libvalent (e.g. `2` if for `1.2.3`)
 */
#define VALENT_MINOR_VERSION (0)

/**
 * VALENT_MICRO_VERSION:
 *
 * The micro version component of libvalent (e.g. `3` if for `1.2.3`)
 */
#define VALENT_MICRO_VERSION (0)

/**
 * VALENT_VERSION:
 *
 * The full version of libvalent, encoded as a string (eg. `"1.0.0.alpha"`).
 */
#define VALENT_VERSION "1.0.0.alpha.48"

/**
 * VALENT_API_VERSION:
 *
 * The API version of libvalent, encoded as a string (eg. `"1.0"`).
 */
#define VALENT_API_VERSION "1"

/**
 * VALENT_CHECK_VERSION:
 * @major: required major version
 * @minor: required minor version
 *
 * Compile-time version check.
 *
 * Evaluates to %TRUE if the API version of libvalent is greater than or equal
 * to the required one.
 *
 * Returns: %TRUE if the requirement is met, or %FALSE if not
 */
#define VALENT_CHECK_VERSION(major,minor)  \
        (VALENT_MAJOR_VERSION > (major) || \
         (VALENT_MAJOR_VERSION == (major) && VALENT_MINOR_VERSION >= (minor)))


#ifndef _VALENT_EXTERN
#define _VALENT_EXTERN extern
#endif

/**
 * VALENT_DISABLE_DEPRECATION_WARNINGS:
 *
 * A macro that should be defined before including any libvalent headers.
 *
 * If defined, no compiler warnings will be produced for deprecated API usage.
 */
#ifdef VALENT_DISABLE_DEPRECATION_WARNINGS
# define VALENT_DEPRECATED           _VALENT_EXTERN
# define VALENT_DEPRECATED_FOR(f)    _VALENT_EXTERN
# define VALENT_UNAVAILABLE(maj,min) _VALENT_EXTERN
#else
# define VALENT_DEPRECATED           G_DEPRECATED           _VALENT_EXTERN
# define VALENT_DEPRECATED_FOR(f)    G_DEPRECATED_FOR(f)    _VALENT_EXTERN
# define VALENT_UNAVAILABLE(maj,min) G_UNAVAILABLE(maj,min) _VALENT_EXTERN
#endif

/* XXX: Every new stable minor release bump should add a macro here */

/**
 * VALENT_VERSION_1_0:
 *
 * The encoded representation of Valent version "1.0".
 *
 * Since: 1.0
 */
#define VALENT_VERSION_1_0 (G_ENCODE_VERSION (1, 0))

/**
 * VALENT_VERSION_MIN_REQUIRED:
 *
 * A macro that defines the lower bound for the libvalent API to use.
 *
 * This should be defined prior to including any libvalent headers and must be
 * one of the predefined version macros, such as %VALENT_VERSION_1_0.
 *
 * If a function has been deprecated in a newer version, it is possible to use
 * this symbol to avoid the compiler warnings without disabling warnings for
 * every deprecated function.
 */
#ifndef VALENT_VERSION_MIN_REQUIRED
# define VALENT_VERSION_MIN_REQUIRED (G_ENCODE_VERSION (VALENT_MAJOR_VERSION, VALENT_MINOR_VERSION))
#endif

/**
 * VALENT_VERSION_MAX_ALLOWED:
 *
 * A macro that defines the upper bound for the libvalent API to use.
 *
 * This should be defined prior to including any libvalent headers and must be
 * one of the predefined version macros, such as %VALENT_VERSION_1_0.
 *
 * If a function has been introduced in a newer version, it is possible to use
 * this symbol to get compiler warnings when trying to use that function.
 */
#ifndef VALENT_VERSION_MAX_ALLOWED
# define VALENT_VERSION_MAX_ALLOWED (VALENT_VERSION_MIN_REQUIRED)
#endif

/* Correctness checks */
#if VALENT_VERSION_MAX_ALLOWED < VALENT_VERSION_MIN_REQUIRED
#error "VALENT_VERSION_MAX_ALLOWED must be >= VALENT_VERSION_MIN_REQUIRED"
#endif
#if VALENT_VERSION_MIN_REQUIRED < VALENT_VERSION_1_0
#error "VALENT_VERSION_MIN_REQUIRED must be >= VALENT_VERSION_1_0"
#endif

#define VALENT_AVAILABLE_IN_ALL _VALENT_EXTERN

/* XXX: Every new stable minor release should add a set of macros here */

/* 1.0 */
#if VALENT_VERSION_MIN_REQUIRED >= VALENT_VERSION_1_0
# define VALENT_DEPRECATED_IN_1_0        VALENT_DEPRECATED
# define VALENT_DEPRECATED_IN_1_0_FOR(f) VALENT_DEPRECATED_FOR(f)
#else
# define VALENT_DEPRECATED_IN_1_0        _VALENT_EXTERN
# define VALENT_DEPRECATED_IN_1_0_FOR(f) _VALENT_EXTERN
#endif

#if VALENT_VERSION_MAX_ALLOWED < VALENT_VERSION_1_0
# define VALENT_AVAILABLE_IN_1_0         VALENT_UNAVAILABLE(1, 0)
#else
# define VALENT_AVAILABLE_IN_1_0         _VALENT_EXTERN
#endif


VALENT_AVAILABLE_IN_ALL
gboolean       valent_check_version     (unsigned int major,
                                         unsigned int minor);
VALENT_AVAILABLE_IN_ALL
unsigned int   valent_get_major_version (void) G_GNUC_CONST;
VALENT_AVAILABLE_IN_ALL
unsigned int   valent_get_minor_version (void) G_GNUC_CONST;
VALENT_AVAILABLE_IN_ALL
unsigned int   valent_get_micro_version (void) G_GNUC_CONST;

G_END_DECLS

