/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.eclipse.jetty.client.Connection;
import org.eclipse.jetty.client.ConnectionPool;
import org.eclipse.jetty.client.Destination;
import org.eclipse.jetty.client.transport.HttpDestination;
import org.eclipse.jetty.util.Attachable;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.Pool;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.thread.Sweeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public abstract class AbstractConnectionPool
extends ContainerLifeCycle
implements ConnectionPool,
Dumpable,
Sweeper.Sweepable {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractConnectionPool.class);
    private final AtomicInteger pending = new AtomicInteger();
    private final HttpDestination destination;
    private final Pool.Factory<Connection> poolFactory;
    private Pool<Connection> pool;
    private boolean maximizeConnections;
    private volatile long maxDurationNanos;
    private volatile int maxUsage;
    private volatile int initialMaxMultiplex;

    protected AbstractConnectionPool(Destination destination, Pool.Factory<Connection> poolFactory, int initialMaxMultiplex) {
        this.destination = (HttpDestination)destination;
        this.poolFactory = poolFactory;
        this.initialMaxMultiplex = initialMaxMultiplex;
    }

    @Override
    protected void doStart() throws Exception {
        this.pool = this.poolFactory.wrap(this.poolFactory.newPool());
        this.addBean(this.pool);
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        super.doStop();
        this.removeBean(this.pool);
        this.pool.terminate().forEach(this::close);
        this.pool = null;
    }

    @Override
    public CompletableFuture<Void> preCreateConnections(int connectionCount) {
        Pool.Entry<Connection> entry;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Pre-creating connections {}/{}", (Object)connectionCount, (Object)this.getMaxConnectionCount());
        }
        ArrayList<FutureConnection> futures = new ArrayList<FutureConnection>();
        for (int i = 0; i < connectionCount && (entry = this.pool.reserve()) != null; ++i) {
            this.pending.incrementAndGet();
            FutureConnection future = new FutureConnection(entry);
            futures.add(future);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Pre-creating connection {}/{} at {}", futures.size(), this.getMaxConnectionCount(), entry);
            }
            this.destination.newConnection(future);
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    @ManagedAttribute(value="The maximum duration in milliseconds a connection can be used for before it gets closed")
    public long getMaxDuration() {
        return TimeUnit.NANOSECONDS.toMillis(this.maxDurationNanos);
    }

    public void setMaxDuration(long maxDurationInMs) {
        this.maxDurationNanos = TimeUnit.MILLISECONDS.toNanos(maxDurationInMs);
    }

    protected int getInitialMaxMultiplex() {
        return this.initialMaxMultiplex;
    }

    protected void setInitialMaxMultiplex(int initialMaxMultiplex) {
        this.initialMaxMultiplex = initialMaxMultiplex;
    }

    @ManagedAttribute(value="The maximum amount of times a connection is used before it gets closed")
    public int getMaxUsage() {
        return this.maxUsage;
    }

    public void setMaxUsage(int maxUsage) {
        this.maxUsage = maxUsage;
    }

    @ManagedAttribute(value="The number of active connections", readonly=true)
    public int getActiveConnectionCount() {
        Pool<Connection> pool = this.pool;
        return pool == null ? 0 : pool.getInUseCount();
    }

    @ManagedAttribute(value="The number of idle connections", readonly=true)
    public int getIdleConnectionCount() {
        Pool<Connection> pool = this.pool;
        return pool == null ? 0 : pool.getIdleCount();
    }

    @ManagedAttribute(value="The max number of connections", readonly=true)
    public int getMaxConnectionCount() {
        Pool<Connection> pool = this.pool;
        return pool == null ? 0 : pool.getMaxSize();
    }

    @ManagedAttribute(value="The number of connections", readonly=true)
    public int getConnectionCount() {
        Pool<Connection> pool = this.pool;
        return pool == null ? 0 : pool.size();
    }

    @ManagedAttribute(value="The number of pending connections", readonly=true)
    public int getPendingConnectionCount() {
        return this.pending.get();
    }

    @Override
    public boolean isEmpty() {
        Pool<Connection> pool = this.pool;
        return pool == null || pool.size() == 0;
    }

    @ManagedAttribute(value="Whether the pool tries to maximize the number of connections used")
    public boolean isMaximizeConnections() {
        return this.maximizeConnections;
    }

    public void setMaximizeConnections(boolean maximizeConnections) {
        this.maximizeConnections = maximizeConnections;
    }

    @Override
    public Connection acquire(boolean create) {
        Connection connection;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquiring create={} on {}", (Object)create, (Object)this);
        }
        if ((connection = this.activate()) == null) {
            this.tryCreate(create);
            connection = this.activate();
        }
        return connection;
    }

    protected void tryCreate(boolean create) {
        int pending;
        int connectionCount = this.getConnectionCount();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Try creating connection {}/{} with {} pending", connectionCount, this.getMaxConnectionCount(), this.getPendingConnectionCount());
        }
        int multiplexed = this.getInitialMaxMultiplex();
        do {
            boolean tryCreate;
            pending = this.pending.get();
            int supply = pending * multiplexed;
            int demand = this.destination.getQueuedRequestCount() + (create ? 1 : 0);
            boolean bl = tryCreate = this.isMaximizeConnections() || supply < demand;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Try creating({}) connection, pending/demand/supply: {}/{}/{}, result={}", create, pending, demand, supply, tryCreate);
            }
            if (tryCreate) continue;
            return;
        } while (!this.pending.compareAndSet(pending, pending + 1));
        Pool.Entry<Connection> entry = this.pool.reserve();
        if (entry == null) {
            this.pending.decrementAndGet();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Not creating connection as pool {} is full, pending: {}", (Object)this.pool, (Object)this.pending);
            }
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Creating connection {}/{} at {}", connectionCount, this.getMaxConnectionCount(), entry);
        }
        FutureConnection future = new FutureConnection(entry);
        this.destination.newConnection(future);
    }

    @Override
    public boolean accept(Connection connection) {
        if (!(connection instanceof Attachable)) {
            throw new IllegalArgumentException("Invalid connection object: " + String.valueOf(connection));
        }
        Attachable attachable = (Attachable)((Object)connection);
        Pool.Entry<Connection> entry = this.pool.reserve();
        if (entry == null) {
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("onCreating {} {}", (Object)entry, (Object)connection);
        }
        attachable.setAttachment(new EntryHolder(entry));
        this.onCreated(connection);
        entry.enable(connection, false);
        this.idle(connection, false);
        return true;
    }

    protected void proceed() {
        this.destination.succeeded();
    }

    protected Connection activate() {
        Pool.Entry<Connection> entry;
        while ((entry = this.pool.acquire()) != null) {
            EntryHolder holder;
            EntryHolder holder2;
            Connection connection = entry.getPooled();
            if (connection.isClosed()) {
                this.remove(connection);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Connection removed: closed {} {}", (Object)entry, (Object)this.pool);
                continue;
            }
            long maxDurationNanos = this.maxDurationNanos;
            if (maxDurationNanos > 0L && (holder2 = (EntryHolder)((Attachable)((Object)connection)).getAttachment()).isExpired(maxDurationNanos)) {
                boolean canClose = this.remove(connection);
                if (canClose) {
                    IO.close(connection);
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Connection removed{}: expired {} {}", canClose ? " and closed" : "", entry, this.pool);
                continue;
            }
            int maxUsage = this.getMaxUsage();
            if (connection instanceof ConnectionPool.MaxUsable) {
                ConnectionPool.MaxUsable maxUsable = (ConnectionPool.MaxUsable)((Object)connection);
                maxUsage = Math.min(maxUsage, maxUsable.getMaxUsage());
            }
            if (maxUsage > 0 && !(holder = (EntryHolder)((Attachable)((Object)connection)).getAttachment()).use(maxUsage)) {
                boolean canClose = this.remove(connection);
                if (canClose) {
                    IO.close(connection);
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Connection removed{}: over used {} {}", canClose ? " and closed" : "", entry, this.pool);
                continue;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Activated {} {}", (Object)entry, (Object)this.pool);
            }
            this.acquired(connection);
            return connection;
        }
        return null;
    }

    @Override
    public boolean isActive(Connection connection) {
        if (!(connection instanceof Attachable)) {
            throw new IllegalArgumentException("Invalid connection object: " + String.valueOf(connection));
        }
        Attachable attachable = (Attachable)((Object)connection);
        EntryHolder holder = (EntryHolder)attachable.getAttachment();
        if (holder == null) {
            return false;
        }
        return !holder.entry.isIdle();
    }

    @Override
    public boolean release(Connection connection) {
        if (!this.deactivate(connection)) {
            return false;
        }
        this.released(connection);
        return this.idle(connection, this.isStopped());
    }

    protected boolean deactivate(Connection connection) {
        if (!(connection instanceof Attachable)) {
            throw new IllegalArgumentException("Invalid connection object: " + String.valueOf(connection));
        }
        Attachable attachable = (Attachable)((Object)connection);
        EntryHolder holder = (EntryHolder)attachable.getAttachment();
        if (holder == null) {
            return true;
        }
        long maxDurationNanos = this.maxDurationNanos;
        if (maxDurationNanos > 0L && holder.isExpired(maxDurationNanos)) {
            return !this.remove(connection);
        }
        int maxUsage = this.getMaxUsage();
        if (connection instanceof ConnectionPool.MaxUsable) {
            ConnectionPool.MaxUsable maxUsable = (ConnectionPool.MaxUsable)((Object)connection);
            maxUsage = maxUsable.getMaxUsage();
        }
        if (maxUsage > 0 && holder.isOverUsed(maxUsage)) {
            return !this.remove(connection);
        }
        boolean reusable = holder.entry.release();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Released ({}) {} {}", reusable, holder.entry, this.pool);
        }
        if (reusable) {
            return true;
        }
        return !this.remove(connection);
    }

    @Override
    public boolean remove(Connection connection) {
        if (!(connection instanceof Attachable)) {
            throw new IllegalArgumentException("Invalid connection object: " + String.valueOf(connection));
        }
        Attachable attachable = (Attachable)((Object)connection);
        EntryHolder holder = (EntryHolder)attachable.getAttachment();
        if (holder == null) {
            return false;
        }
        boolean removed = holder.entry.remove();
        if (removed) {
            attachable.setAttachment(null);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Removed ({}) {} {}", removed, holder.entry, this.pool);
        }
        if (removed) {
            this.released(connection);
            this.onRemoved(connection);
        }
        return removed;
    }

    protected void onCreated(Connection connection) {
    }

    @Deprecated(since="12.0.8", forRemoval=true)
    protected boolean idle(Connection connection, boolean close) {
        return !close;
    }

    @Deprecated(since="12.0.8", forRemoval=true)
    protected void acquired(Connection connection) {
    }

    @Deprecated(since="12.0.8", forRemoval=true)
    protected void released(Connection connection) {
    }

    @Deprecated(since="12.0.8", forRemoval=true)
    protected void removed(Connection connection) {
    }

    protected void onRemoved(Connection connection) {
        this.removed(connection);
    }

    Collection<Connection> getIdleConnections() {
        return this.pool.stream().filter(Pool.Entry::isIdle).filter(entry -> !entry.isTerminated()).map(Pool.Entry::getPooled).collect(Collectors.toCollection(ArrayDeque::new));
    }

    Collection<Connection> getActiveConnections() {
        return this.pool.stream().filter(entry -> !entry.isIdle()).filter(entry -> !entry.isTerminated()).map(Pool.Entry::getPooled).collect(Collectors.toList());
    }

    private void close(Pool.Entry<Connection> entry) {
        assert (this.pool.isTerminated());
        Connection connection = entry.getPooled();
        while (true) {
            if (entry.remove()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Removed terminated entry {}", (Object)entry);
                }
                this.onRemoved(connection);
                IO.close(connection);
            }
            if (!entry.isInUse()) break;
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Entry {} still in use, removing it again", (Object)entry);
        }
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, new Object[0]);
    }

    @Override
    public boolean sweep() {
        Pool<Connection> pool = this.pool;
        if (pool != null) {
            pool.stream().map(Pool.Entry::getPooled).filter(connection -> connection instanceof Sweeper.Sweepable).forEach(connection -> {
                if (((Sweeper.Sweepable)((Object)connection)).sweep()) {
                    boolean removed = this.remove((Connection)connection);
                    LOG.warn("Connection swept: {}{}{} from active connections{}{}", connection, System.lineSeparator(), removed ? "Removed" : "Not removed", System.lineSeparator(), this.dump());
                }
            });
        }
        return false;
    }

    @Override
    public String toString() {
        return String.format("%s@%x[s=%s,c=%d/%d/%d,a=%d,i=%d,q=%d,p=%s]", TypeUtil.toShortName(this.getClass()), this.hashCode(), this.getState(), this.getPendingConnectionCount(), this.getConnectionCount(), this.getMaxConnectionCount(), this.getActiveConnectionCount(), this.getIdleConnectionCount(), this.destination.getQueuedRequestCount(), this.pool);
    }

    private class FutureConnection
    extends Promise.Completable<Connection> {
        private final Pool.Entry<Connection> reserved;

        public FutureConnection(Pool.Entry<Connection> reserved) {
            this.reserved = reserved;
        }

        @Override
        public void succeeded(Connection connection) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection creation succeeded {}: {}", (Object)this.reserved, (Object)connection);
            }
            if (connection instanceof Attachable) {
                ((Attachable)((Object)connection)).setAttachment(new EntryHolder(this.reserved));
                AbstractConnectionPool.this.onCreated(connection);
                AbstractConnectionPool.this.pending.decrementAndGet();
                this.reserved.enable(connection, false);
                AbstractConnectionPool.this.idle(connection, false);
                super.succeeded(connection);
                AbstractConnectionPool.this.proceed();
            } else {
                this.failed(new IllegalArgumentException("Invalid connection object: " + String.valueOf(connection)));
            }
        }

        @Override
        public void failed(Throwable x) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection creation failed {}", (Object)this.reserved, (Object)x);
            }
            AbstractConnectionPool.this.pending.decrementAndGet();
            this.reserved.remove();
            super.failed(x);
            AbstractConnectionPool.this.destination.failed(x);
        }
    }

    private static class EntryHolder {
        private final Pool.Entry<Connection> entry;
        private final long creationNanoTime = NanoTime.now();
        private final AtomicInteger usage = new AtomicInteger();

        private EntryHolder(Pool.Entry<Connection> entry) {
            this.entry = Objects.requireNonNull(entry);
        }

        private boolean isExpired(long timeoutNanos) {
            return NanoTime.since(this.creationNanoTime) >= timeoutNanos;
        }

        private boolean use(int maxUsage) {
            int current;
            do {
                if ((current = this.usage.get()) < maxUsage) continue;
                return false;
            } while (!this.usage.compareAndSet(current, current + 1));
            return true;
        }

        public boolean isOverUsed(int maxUsage) {
            return this.usage.get() >= maxUsage;
        }
    }
}

