/*
 * Decompiled with CFR 0.152.
 */
package org.digitalmediaserver.cuelib;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.digitalmediaserver.cuelib.CueSheet;
import org.digitalmediaserver.cuelib.CueSheetSerializer;
import org.digitalmediaserver.cuelib.CueSheetToXmlSerializer;
import org.digitalmediaserver.cuelib.FileData;
import org.digitalmediaserver.cuelib.Index;
import org.digitalmediaserver.cuelib.LineOfInput;
import org.digitalmediaserver.cuelib.Message;
import org.digitalmediaserver.cuelib.Position;
import org.digitalmediaserver.cuelib.TrackData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CueParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(CueParser.class);
    protected static final String WARNING_EMPTY_LINES = "Empty lines not allowed. Will ignore";
    protected static final String WARNING_UNPARSEABLE_INPUT = "Unparseable line. Will ignore";
    protected static final String WARNING_INVALID_CATALOG_NUMBER = "Invalid catalog number";
    protected static final String WARNING_NONCOMPLIANT_FILE_TYPE = "Noncompliant file type";
    protected static final String WARNING_NO_FLAGS = "No flags specified";
    protected static final String WARNING_NONCOMPLIANT_FLAG = "Noncompliant flag(s) specified";
    protected static final String WARNING_WRONG_NUMBER_OF_DIGITS = "Wrong number of digits in number";
    protected static final String WARNING_NONCOMPLIANT_ISRC_CODE = "ISRC code has noncompliant format";
    protected static final String WARNING_FIELD_LENGTH_OVER_80 = "The field is too long to burn as CD-TEXT. The maximum length is 80";
    protected static final String WARNING_NONCOMPLIANT_DATA_TYPE = "Noncompliant data type specified";
    protected static final String WARNING_TOKEN_NOT_UPPERCASE = "Token has wrong case. Uppercase was expected";
    protected static final String WARNING_INVALID_FRAMES_VALUE = "Position has invalid frame value, should be 00-74";
    protected static final String WARNING_INVALID_SECONDS_VALUE = "Position has invalid seconds value, should be 00-59";
    protected static final String WARNING_DATUM_APPEARS_TOO_OFTEN = "Datum appears too often";
    protected static final String WARNING_FILE_IN_WRONG_PLACE = "A FILE datum must come before everything else except REM and CATALOG";
    protected static final String WARNING_FLAGS_IN_WRONG_PLACE = "A FLAGS datum must come after a TRACK, but before any INDEX of that TRACK";
    protected static final String WARNING_NO_FILE_SPECIFIED = "Datum must appear in FILE, but no FILE specified";
    protected static final String WARNING_NO_TRACK_SPECIFIED = "Datum must appear in TRACK, but no TRACK specified";
    protected static final String WARNING_INVALID_INDEX_NUMBER = "Invalid index number. First number must be 0 or 1; all next ones sequential";
    protected static final String WARNING_INVALID_FIRST_POSITION = "Invalid position. First index must have position 00:00:00";
    protected static final String WARNING_ISRC_IN_WRONG_PLACE = "An ISRC datum must come after TRACK, but before any INDEX of TRACK";
    protected static final String WARNING_PREGAP_IN_WRONG_PLACE = "A PREGAP datum must come after TRACK, but before any INDEX of that TRACK";
    protected static final String WARNING_INDEX_AFTER_POSTGAP = "A POSTGAP datum must come after all INDEX data of a TRACK";
    protected static final String WARNING_INVALID_DISCNUMBER = "Invalid disc number. Should be a number from 1";
    protected static final String WARNING_INVALID_TOTALDISCS = "Invalid total discs. Should be a number from 1";
    protected static final String WARNING_INVALID_TRACK_NUMBER = "Invalid track number. First number must be 1; all next ones sequential";
    protected static final String WARNING_INVALID_YEAR = "Invalid year. Should be a number from 1 to 9999 (inclusive)";
    protected static final Pattern PATTERN_POSITION = Pattern.compile("^(\\d*):(\\d*):(\\d*)$");
    protected static final Pattern PATTERN_CATALOG_NUMBER = Pattern.compile("^\\d{13}$");
    protected static final Pattern PATTERN_FILE = Pattern.compile("^FILE\\s+((?:\"[^\"]*\")|\\S+)\\s+(\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_CDTEXTFILE = Pattern.compile("^CDTEXTFILE\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_FLAGS = Pattern.compile("^FLAGS(\\s+\\w+)*\\s*$", 2);
    protected static final Pattern PATTERN_INDEX = Pattern.compile("^INDEX\\s+(\\d+)\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    protected static final Pattern PATTERN_ISRC_CODE = Pattern.compile("^\\w{5}\\d{7}$");
    protected static final Pattern PATTERN_PERFORMER = Pattern.compile("^PERFORMER\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_POSTGAP = Pattern.compile("^POSTGAP\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    protected static final Pattern PATTERN_PREGAP = Pattern.compile("^PREGAP\\s+(\\d*:\\d*:\\d*)\\s*$", 2);
    protected static final Pattern PATTERN_REM_COMMENT = Pattern.compile("^(REM\\s+COMMENT)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_REM_DATE = Pattern.compile("^(REM\\s+DATE)\\s+(\\d+)\\s*$", 2);
    protected static final Pattern PATTERN_REM_DISCID = Pattern.compile("^(REM\\s+DISCID)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_REM_DISCNUMBER = Pattern.compile("^(REM\\s+DISCNUMBER)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_REM_GENRE = Pattern.compile("^(REM\\s+GENRE)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_REM_TOTALDISCS = Pattern.compile("^(REM\\s+TOTALDISCS)\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_SONGWRITER = Pattern.compile("^SONGWRITER\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_TITLE = Pattern.compile("^TITLE\\s+((?:\"[^\"]*\")|\\S+)\\s*$", 2);
    protected static final Pattern PATTERN_TRACK = Pattern.compile("TRACK\\s+(\\d+)\\s+(\\S+)\\s*$", 2);
    protected static final Set<String> COMPLIANT_FILE_TYPES = Collections.unmodifiableSet(new TreeSet<String>(Arrays.asList("BINARY", "MOTOROLA", "AIFF", "WAVE", "MP3")));
    protected static final Set<String> COMPLIANT_FLAGS = Collections.unmodifiableSet(new TreeSet<String>(Arrays.asList("DCP", "4CH", "PRE", "SCMS", "DATA")));
    protected static final Set<String> COMPLIANT_DATA_TYPES = Collections.unmodifiableSet(new TreeSet<String>(Arrays.asList("AUDIO", "CDG", "MODE1/2048", "MODE1/2352", "MODE2/2336", "MODE2/2352", "CDI/2336", "CDI/2352")));

    private CueParser() {
    }

    @Deprecated
    public static CueSheet parse(InputStream inputStream) throws IOException {
        return CueParser.parse(inputStream, null);
    }

    public static CueSheet parse(InputStream inputStream, Charset charset) throws IOException {
        if (charset == null) {
            charset = Charset.defaultCharset();
        }
        try (LineNumberReader lnReader = new LineNumberReader(new InputStreamReader(inputStream, charset));){
            CueSheet cueSheet = CueParser.parse(lnReader);
            return cueSheet;
        }
    }

    public static CueSheet parse(Path file, Charset charset) throws IOException {
        if (charset == null) {
            charset = Charset.defaultCharset();
        }
        try (LineNumberReader lnReader = new LineNumberReader(Files.newBufferedReader(file, charset));){
            CueSheet cueSheet = CueParser.parse(lnReader, file);
            return cueSheet;
        }
    }

    @Deprecated
    public static CueSheet parse(File file) throws IOException {
        return CueParser.parse(file, null);
    }

    public static CueSheet parse(File file, Charset charset) throws IOException {
        if (charset == null) {
            charset = Charset.defaultCharset();
        }
        try (LineNumberReader lnReader = new LineNumberReader(new InputStreamReader((InputStream)new FileInputStream(file), charset));){
            CueSheet cueSheet = CueParser.parse(lnReader, file.toPath());
            return cueSheet;
        }
    }

    public static CueSheet parse(LineNumberReader reader) throws IOException {
        return CueParser.parse(reader, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static CueSheet parse(LineNumberReader reader, Path file) throws IOException {
        if (file == null) {
            LOGGER.debug("Parsing cue sheet.");
        } else {
            LOGGER.debug("Parsing cue sheet \"{}\".", (Object)file);
        }
        CueSheet result = new CueSheet(file);
        try {
            String inputLine = reader.readLine();
            while (inputLine != null) {
                LOGGER.trace("Processing input line \"{}\".", (Object)inputLine);
                inputLine = inputLine.trim();
                LineOfInput input = new LineOfInput(reader.getLineNumber(), inputLine, result);
                if (inputLine.length() == 0) {
                    CueParser.addWarning(input, WARNING_EMPTY_LINES);
                } else if (inputLine.length() < 2) {
                    CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                } else {
                    block1 : switch (inputLine.charAt(0)) {
                        case 'C': 
                        case 'c': {
                            switch (inputLine.charAt(1)) {
                                case 'A': 
                                case 'a': {
                                    CueParser.parseCatalog(input);
                                    break block1;
                                }
                                case 'D': 
                                case 'd': {
                                    CueParser.parseCdTextFile(input);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'F': 
                        case 'f': {
                            switch (inputLine.charAt(1)) {
                                case 'I': 
                                case 'i': {
                                    CueParser.parseFile(input);
                                    break block1;
                                }
                                case 'L': 
                                case 'l': {
                                    CueParser.parseFlags(input);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'I': 
                        case 'i': {
                            switch (inputLine.charAt(1)) {
                                case 'N': 
                                case 'n': {
                                    CueParser.parseIndex(input);
                                    break block1;
                                }
                                case 'S': 
                                case 's': {
                                    CueParser.parseIsrc(input);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'P': 
                        case 'p': {
                            switch (inputLine.charAt(1)) {
                                case 'E': 
                                case 'e': {
                                    CueParser.parsePerformer(input);
                                    break block1;
                                }
                                case 'O': 
                                case 'o': {
                                    CueParser.parsePostgap(input);
                                    break block1;
                                }
                                case 'R': 
                                case 'r': {
                                    CueParser.parsePregap(input);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        case 'R': 
                        case 'r': {
                            CueParser.parseRem(input);
                            break;
                        }
                        case 'S': 
                        case 's': {
                            CueParser.parseSongwriter(input);
                            break;
                        }
                        case 'T': 
                        case 't': {
                            switch (inputLine.charAt(1)) {
                                case 'I': 
                                case 'i': {
                                    CueParser.parseTitle(input);
                                    break block1;
                                }
                                case 'R': 
                                case 'r': {
                                    CueParser.parseTrack(input);
                                    break block1;
                                }
                            }
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                            break;
                        }
                        default: {
                            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                        }
                    }
                }
                inputLine = reader.readLine();
            }
        }
        finally {
            LOGGER.trace("Closing input reader.");
            reader.close();
        }
        return result;
    }

    protected static boolean startsWith(LineOfInput input, String start) {
        if (input.getInput().startsWith(start)) {
            return true;
        }
        if (input.getInput().substring(0, start.length()).equalsIgnoreCase(start)) {
            CueParser.addWarning(input, WARNING_TOKEN_NOT_UPPERCASE);
            return true;
        }
        return false;
    }

    protected static boolean contains(LineOfInput input, Pattern pattern) {
        Matcher matcher = pattern.matcher(input.getInput());
        if (matcher.find()) {
            if (matcher.groupCount() > 0 && !matcher.group(1).equals(matcher.group(1).toUpperCase())) {
                CueParser.addWarning(input, WARNING_TOKEN_NOT_UPPERCASE);
            }
            return true;
        }
        return false;
    }

    protected static void parseCatalog(LineOfInput input) {
        if (CueParser.startsWith(input, "CATALOG")) {
            String catalogNumber = input.getInput().substring("CATALOG".length()).trim();
            if (!PATTERN_CATALOG_NUMBER.matcher(catalogNumber).matches()) {
                CueParser.addWarning(input, WARNING_INVALID_CATALOG_NUMBER);
            }
            if (input.getAssociatedSheet().getCatalog() != null) {
                CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            input.getAssociatedSheet().setCatalog(catalogNumber);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseFile(LineOfInput input) {
        Matcher fileMatcher = PATTERN_FILE.matcher(input.getInput());
        if (CueParser.startsWith(input, "FILE") && fileMatcher.matches()) {
            String file;
            if (!COMPLIANT_FILE_TYPES.contains(fileMatcher.group(2))) {
                if (COMPLIANT_FILE_TYPES.contains(fileMatcher.group(2).toUpperCase())) {
                    CueParser.addWarning(input, WARNING_TOKEN_NOT_UPPERCASE);
                } else {
                    CueParser.addWarning(input, WARNING_NONCOMPLIANT_FILE_TYPE);
                }
            }
            if ((file = fileMatcher.group(1)).length() > 0 && file.charAt(0) == '\"' && file.charAt(file.length() - 1) == '\"') {
                file = file.substring(1, file.length() - 1);
            }
            input.getAssociatedSheet().getFileData().add(new FileData(input.getAssociatedSheet(), file, fileMatcher.group(2).toUpperCase()));
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseCdTextFile(LineOfInput input) {
        Matcher cdTextFileMatcher = PATTERN_CDTEXTFILE.matcher(input.getInput());
        if (CueParser.startsWith(input, "CDTEXTFILE") && cdTextFileMatcher.matches()) {
            String file;
            if (input.getAssociatedSheet().getCdTextFile() != null) {
                LOGGER.warn(WARNING_DATUM_APPEARS_TOO_OFTEN);
                input.getAssociatedSheet().addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            if ((file = cdTextFileMatcher.group(1)).length() > 0 && file.charAt(0) == '\"' && file.charAt(file.length() - 1) == '\"') {
                file = file.substring(1, file.length() - 1);
            }
            input.getAssociatedSheet().setCdTextFile(file);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseFlags(LineOfInput input) {
        Matcher flagsMatcher = PATTERN_FLAGS.matcher(input.getInput());
        if (CueParser.startsWith(input, "FLAGS") && flagsMatcher.matches()) {
            if (null == flagsMatcher.group(1)) {
                CueParser.addWarning(input, WARNING_NO_FLAGS);
            } else {
                Set<String> flagCollection;
                TrackData trackData = CueParser.getLastTrackData(input);
                if (trackData.getIndices().size() > 0) {
                    CueParser.addWarning(input, WARNING_FLAGS_IN_WRONG_PLACE);
                }
                if (!(flagCollection = trackData.getFlags()).isEmpty()) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                try (Scanner flagScanner = new Scanner(flagsMatcher.group(1));){
                    while (flagScanner.hasNext()) {
                        String flag = flagScanner.next();
                        if (!COMPLIANT_FLAGS.contains(flag)) {
                            CueParser.addWarning(input, WARNING_NONCOMPLIANT_FLAG);
                        }
                        flagCollection.add(flag);
                    }
                }
            }
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseIndex(LineOfInput input) {
        Matcher indexMatcher = PATTERN_INDEX.matcher(input.getInput());
        if (CueParser.startsWith(input, "INDEX") && indexMatcher.matches()) {
            TrackData trackData;
            List<Index> trackIndices;
            if (indexMatcher.group(1).length() != 2) {
                CueParser.addWarning(input, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            if ((trackIndices = (trackData = CueParser.getLastTrackData(input)).getIndices()).isEmpty() && trackData.getPostgap() != null) {
                CueParser.addWarning(input, WARNING_INDEX_AFTER_POSTGAP);
            }
            int indexNumber = Integer.parseInt(indexMatcher.group(1));
            if (trackIndices.isEmpty() && indexNumber > 1 || !trackIndices.isEmpty() && trackIndices.get(trackIndices.size() - 1).getNumber() != indexNumber - 1) {
                CueParser.addWarning(input, WARNING_INVALID_INDEX_NUMBER);
            }
            List<Index> fileIndices = CueParser.getLastFileData(input).getAllIndices();
            Position position = CueParser.parsePosition(input, indexMatcher.group(2));
            if (fileIndices.isEmpty() && (position.getMinutes() != 0 || position.getSeconds() != 0 || position.getFrames() != 0)) {
                CueParser.addWarning(input, WARNING_INVALID_FIRST_POSITION);
            }
            trackIndices.add(new Index(indexNumber, position));
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseIsrc(LineOfInput input) {
        if (CueParser.startsWith(input, "ISRC")) {
            TrackData trackData;
            String isrcCode = input.getInput().substring("ISRC".length()).trim();
            if (!PATTERN_ISRC_CODE.matcher(isrcCode).matches()) {
                CueParser.addWarning(input, WARNING_NONCOMPLIANT_ISRC_CODE);
            }
            if ((trackData = CueParser.getLastTrackData(input)).getIndices().size() > 0) {
                CueParser.addWarning(input, WARNING_ISRC_IN_WRONG_PLACE);
            }
            if (trackData.getIsrcCode() != null) {
                CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            trackData.setIsrcCode(isrcCode);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parsePerformer(LineOfInput input) {
        Matcher performerMatcher = PATTERN_PERFORMER.matcher(input.getInput());
        if (CueParser.startsWith(input, "PERFORMER") && performerMatcher.matches()) {
            String performer = performerMatcher.group(1);
            if (performer.charAt(0) == '\"') {
                performer = performer.substring(1, performer.length() - 1);
            }
            if (performer.length() > 80) {
                CueParser.addWarning(input, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (input.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(input).getTrackData().size() == 0) {
                if (input.getAssociatedSheet().getPerformer() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                input.getAssociatedSheet().setPerformer(performer);
            } else {
                TrackData trackData = CueParser.getLastTrackData(input);
                if (trackData.getPerformer() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setPerformer(performer);
            }
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parsePostgap(LineOfInput input) {
        Matcher postgapMatcher = PATTERN_POSTGAP.matcher(input.getInput());
        if (CueParser.startsWith(input, "POSTGAP") && postgapMatcher.matches()) {
            TrackData trackData = CueParser.getLastTrackData(input);
            if (trackData.getPostgap() != null) {
                CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            trackData.setPostgap(CueParser.parsePosition(input, postgapMatcher.group(1)));
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parsePregap(LineOfInput input) {
        Matcher pregapMatcher = PATTERN_PREGAP.matcher(input.getInput());
        if (CueParser.startsWith(input, "PREGAP") && pregapMatcher.matches()) {
            TrackData trackData = CueParser.getLastTrackData(input);
            if (trackData.getPregap() != null) {
                CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
            }
            if (trackData.getIndices().size() > 0) {
                CueParser.addWarning(input, WARNING_PREGAP_IN_WRONG_PLACE);
            }
            trackData.setPregap(CueParser.parsePosition(input, pregapMatcher.group(1)));
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemComment(LineOfInput input) {
        Matcher matcher = PATTERN_REM_COMMENT.matcher(input.getInput());
        if (matcher.find()) {
            String comment = matcher.group(2);
            if (comment.charAt(0) == '\"' && comment.charAt(comment.length() - 1) == '\"') {
                comment = comment.substring(1, comment.length() - 1);
            }
            input.getAssociatedSheet().setComment(comment);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemDate(LineOfInput input) {
        Matcher matcher = PATTERN_REM_DATE.matcher(input.getInput());
        if (matcher.find()) {
            int year = Integer.parseInt(matcher.group(2));
            if (year < 1 || year > 9999) {
                CueParser.addWarning(input, WARNING_INVALID_YEAR);
            }
            input.getAssociatedSheet().setYear(year);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemDiscid(LineOfInput input) {
        Matcher matcher = PATTERN_REM_DISCID.matcher(input.getInput());
        if (matcher.find()) {
            String discid = matcher.group(2);
            if (discid.charAt(0) == '\"' && discid.charAt(discid.length() - 1) == '\"') {
                discid = discid.substring(1, discid.length() - 1);
            }
            input.getAssociatedSheet().setDiscId(discid);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemDiscNumber(LineOfInput input) {
        Matcher matcher = PATTERN_REM_DISCNUMBER.matcher(input.getInput());
        if (matcher.find()) {
            int discNumber = Integer.parseInt(matcher.group(2));
            if (discNumber < 1) {
                CueParser.addWarning(input, WARNING_INVALID_DISCNUMBER);
            }
            input.getAssociatedSheet().setDiscNumber(discNumber);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemGenre(LineOfInput input) {
        Matcher matcher = PATTERN_REM_GENRE.matcher(input.getInput());
        if (matcher.find()) {
            String genre = matcher.group(2);
            if (genre.charAt(0) == '\"' && genre.charAt(genre.length() - 1) == '\"') {
                genre = genre.substring(1, genre.length() - 1);
            }
            input.getAssociatedSheet().setGenre(genre);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRemTotalDiscs(LineOfInput input) {
        Matcher matcher = PATTERN_REM_TOTALDISCS.matcher(input.getInput());
        if (matcher.find()) {
            int totalDiscs = Integer.parseInt(matcher.group(2));
            if (totalDiscs < 1) {
                CueParser.addWarning(input, WARNING_INVALID_TOTALDISCS);
            }
            input.getAssociatedSheet().setTotalDiscs(totalDiscs);
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseRem(LineOfInput input) {
        block13: {
            block12: {
                if (!CueParser.startsWith(input, "REM")) break block12;
                String comment = input.getInput().substring("REM".length()).trim();
                switch (comment.charAt(0)) {
                    case 'C': 
                    case 'c': {
                        if (CueParser.contains(input, PATTERN_REM_COMMENT)) {
                            CueParser.parseRemComment(input);
                            break;
                        }
                        break block13;
                    }
                    case 'D': 
                    case 'd': {
                        if (CueParser.contains(input, PATTERN_REM_DATE)) {
                            CueParser.parseRemDate(input);
                            break;
                        }
                        if (CueParser.contains(input, PATTERN_REM_DISCID)) {
                            CueParser.parseRemDiscid(input);
                            break;
                        }
                        if (CueParser.contains(input, PATTERN_REM_DISCNUMBER)) {
                            CueParser.parseRemDiscNumber(input);
                            break;
                        }
                        break block13;
                    }
                    case 'G': 
                    case 'g': {
                        if (CueParser.contains(input, PATTERN_REM_GENRE)) {
                            CueParser.parseRemGenre(input);
                            break;
                        }
                        break block13;
                    }
                    case 'T': 
                    case 't': {
                        if (CueParser.contains(input, PATTERN_REM_TOTALDISCS)) {
                            CueParser.parseRemTotalDiscs(input);
                            break;
                        }
                        break block13;
                    }
                    default: {
                        CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
                        break;
                    }
                }
                break block13;
            }
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseSongwriter(LineOfInput input) {
        Matcher songwriterMatcher = PATTERN_SONGWRITER.matcher(input.getInput());
        if (CueParser.startsWith(input, "SONGWRITER") && songwriterMatcher.matches()) {
            String songwriter = songwriterMatcher.group(1);
            if (songwriter.charAt(0) == '\"') {
                songwriter = songwriter.substring(1, songwriter.length() - 1);
            }
            if (songwriter.length() > 80) {
                CueParser.addWarning(input, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (input.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(input).getTrackData().size() == 0) {
                if (input.getAssociatedSheet().getSongwriter() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                input.getAssociatedSheet().setSongwriter(songwriter);
            } else {
                TrackData trackData = CueParser.getLastTrackData(input);
                if (trackData.getSongwriter() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setSongwriter(songwriter);
            }
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseTitle(LineOfInput input) {
        Matcher titleMatcher = PATTERN_TITLE.matcher(input.getInput());
        if (CueParser.startsWith(input, "TITLE") && titleMatcher.matches()) {
            String title = titleMatcher.group(1);
            if (title.charAt(0) == '\"') {
                title = title.substring(1, title.length() - 1);
            }
            if (title.length() > 80) {
                CueParser.addWarning(input, WARNING_FIELD_LENGTH_OVER_80);
            }
            if (input.getAssociatedSheet().getFileData().size() == 0 || CueParser.getLastFileData(input).getTrackData().size() == 0) {
                if (input.getAssociatedSheet().getTitle() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                input.getAssociatedSheet().setTitle(title);
            } else {
                TrackData trackData = CueParser.getLastTrackData(input);
                if (trackData.getTitle() != null) {
                    CueParser.addWarning(input, WARNING_DATUM_APPEARS_TOO_OFTEN);
                }
                trackData.setTitle(title);
            }
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static void parseTrack(LineOfInput input) {
        Matcher trackMatcher = PATTERN_TRACK.matcher(input.getInput());
        if (CueParser.startsWith(input, "TRACK") && trackMatcher.matches()) {
            List<TrackData> trackDataList;
            if (trackMatcher.group(1).length() != 2) {
                CueParser.addWarning(input, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            int trackNumber = Integer.parseInt(trackMatcher.group(1));
            String dataType = trackMatcher.group(2);
            if (!COMPLIANT_DATA_TYPES.contains(dataType)) {
                CueParser.addWarning(input, WARNING_NONCOMPLIANT_DATA_TYPE);
            }
            if ((trackDataList = input.getAssociatedSheet().getAllTrackData()).isEmpty() && trackNumber != 1 || !trackDataList.isEmpty() && trackDataList.get(trackDataList.size() - 1).getNumber() != trackNumber - 1) {
                CueParser.addWarning(input, WARNING_INVALID_TRACK_NUMBER);
            }
            FileData lastFileData = CueParser.getLastFileData(input);
            lastFileData.getTrackData().add(new TrackData(lastFileData, trackNumber, dataType));
        } else {
            CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        }
    }

    protected static Position parsePosition(LineOfInput input, String position) {
        Matcher positionMatcher = PATTERN_POSITION.matcher(position);
        if (positionMatcher.matches()) {
            String minutesString = positionMatcher.group(1);
            String secondsString = positionMatcher.group(2);
            String framesString = positionMatcher.group(3);
            int minutes = Integer.parseInt(minutesString);
            int seconds = Integer.parseInt(secondsString);
            int frames = Integer.parseInt(framesString);
            if (minutesString.length() != 2 || secondsString.length() != 2 || framesString.length() != 2) {
                CueParser.addWarning(input, WARNING_WRONG_NUMBER_OF_DIGITS);
            }
            if (seconds > 59) {
                CueParser.addWarning(input, WARNING_INVALID_SECONDS_VALUE);
            }
            if (frames > 74) {
                CueParser.addWarning(input, WARNING_INVALID_FRAMES_VALUE);
            }
            return new Position(minutes, seconds, frames);
        }
        CueParser.addWarning(input, WARNING_UNPARSEABLE_INPUT);
        return new Position();
    }

    protected static TrackData getLastTrackData(LineOfInput input) {
        FileData lastFileData = CueParser.getLastFileData(input);
        List<TrackData> trackDataList = lastFileData.getTrackData();
        if (trackDataList.size() == 0) {
            trackDataList.add(new TrackData(lastFileData));
            CueParser.addWarning(input, WARNING_NO_TRACK_SPECIFIED);
        }
        return trackDataList.get(trackDataList.size() - 1);
    }

    protected static FileData getLastFileData(LineOfInput input) {
        List<FileData> fileDataList = input.getAssociatedSheet().getFileData();
        if (fileDataList.size() == 0) {
            fileDataList.add(new FileData(input.getAssociatedSheet()));
            CueParser.addWarning(input, WARNING_NO_FILE_SPECIFIED);
        }
        return fileDataList.get(fileDataList.size() - 1);
    }

    protected static void addWarning(LineOfInput input, String warning) {
        LOGGER.warn("Cue sheet parsing line {}: {}", (Object)input.getLineNumber(), (Object)warning);
        input.getAssociatedSheet().addWarning(input, warning);
    }

    public static void main(String[] args) {
        CueSheet sheet = null;
        try {
            CueSheetToXmlSerializer xmlSerializer = new CueSheetToXmlSerializer();
            FileFilter cueFilter = new FileFilter(){

                @Override
                public boolean accept(File file) {
                    return file.getName().length() >= 4 && file.getName().substring(file.getName().length() - 4).equalsIgnoreCase(".cue");
                }
            };
            ArrayList<File> files = new ArrayList<File>();
            File[] filesFound = null;
            File workingDir = new File(System.getProperty("user.dir"));
            filesFound = workingDir.listFiles(cueFilter);
            if (filesFound != null) {
                files.addAll(Arrays.asList(filesFound));
            }
            for (File file : files) {
                LOGGER.info("Processing file: \"{}\"", (Object)file);
                sheet = CueParser.parse(file);
                for (Message message : sheet.getMessages()) {
                    System.out.println(message);
                }
                System.out.println(new CueSheetSerializer().serializeCueSheet(sheet));
                xmlSerializer.serializeCueSheet(sheet, System.out);
            }
        }
        catch (Exception e) {
            LOGGER.error("", e);
        }
    }
}

