/*
 * Decompiled with CFR 0.152.
 */
package net.pms.parsers;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Locale;
import net.pms.PMS;
import net.pms.dlna.DLNAThumbnail;
import net.pms.external.musicbrainz.coverart.CoverUtil;
import net.pms.image.ImageFormat;
import net.pms.image.ImagesUtil;
import net.pms.media.MediaInfo;
import net.pms.media.audio.MediaAudio;
import net.pms.media.audio.metadata.MediaAudioMetadata;
import net.pms.parsers.JaudiotaggerParser;
import net.pms.parsers.Parser;
import net.pms.store.ThumbnailSource;
import net.pms.store.ThumbnailStore;
import net.pms.util.CoverSupplier;
import net.pms.util.InputFile;
import org.apache.commons.lang3.StringUtils;
import org.jaudiotagger.tag.id3.ID3v1Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RealAudioParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(JaudiotaggerParser.class);
    private static final String PARSER_NAME = "RealAudio";

    private RealAudioParser() {
    }

    public static boolean parse(MediaInfo mediaInfo, InputFile file, int type) {
        try {
            FileChannel channel = FileChannel.open(file.getFile().toPath(), StandardOpenOption.READ);
            if (RealAudioParser.parseRealAudio(channel, mediaInfo)) {
                Parser.postParse(mediaInfo, type);
                return true;
            }
        }
        catch (IOException e) {
            LOGGER.warn("An error occurred when trying to open \"{}\" for reading: {}", (Object)file, (Object)e.getMessage());
            LOGGER.trace("", e);
        }
        return false;
    }

    private static boolean parseRealAudio(ReadableByteChannel channel, MediaInfo mediaInfo) {
        byte[] magicBytes = new byte[]{46, 114, 97, -3};
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.order(ByteOrder.BIG_ENDIAN);
        MediaAudio audio = new MediaAudio();
        MediaAudioMetadata audioMetadata = new MediaAudioMetadata();
        try {
            int reportedDataSize;
            int count = channel.read(buffer);
            if (count < 4) {
                LOGGER.trace("Input is too short to be RealAudio");
                return false;
            }
            buffer.flip();
            byte[] signature = new byte[4];
            buffer.get(signature);
            if (!Arrays.equals(magicBytes, signature)) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Input signature ({}) mismatches RealAudio version 1.0 or 2.0", (Object)new String(signature, StandardCharsets.US_ASCII));
                }
                return false;
            }
            mediaInfo.setContainer("ra");
            short version = buffer.getShort();
            int reportedHeaderSize = 0;
            switch (version) {
                case 3: {
                    audio.setCodec("ra14.4");
                    audio.setNumberOfChannels(1);
                    audio.setSampleRate(8000);
                    short headerSize = buffer.getShort();
                    buffer = ByteBuffer.allocate(headerSize);
                    channel.read(buffer);
                    buffer.flip();
                    buffer.position(8);
                    int bytesPerMinute = buffer.getShort() & 0xFFFF;
                    reportedDataSize = buffer.getInt();
                    byte b = buffer.get();
                    if (b != 0) {
                        byte[] title = new byte[b & 0xFF];
                        buffer.get(title);
                        String titleString = new String(title, StandardCharsets.US_ASCII);
                        audioMetadata.setSongname(titleString);
                        audio.setTitle(titleString);
                    }
                    if ((b = buffer.get()) != 0) {
                        byte[] artist = new byte[b & 0xFF];
                        buffer.get(artist);
                        audioMetadata.setArtist(new String(artist, StandardCharsets.US_ASCII));
                    }
                    audio.setBitRate(bytesPerMinute * 8 / 60);
                    mediaInfo.setBitRate(bytesPerMinute * 8 / 60);
                    break;
                }
                case 4: 
                case 5: {
                    String fourCCString;
                    byte[] fourCC;
                    buffer = ByteBuffer.allocate(14);
                    channel.read(buffer);
                    buffer.flip();
                    buffer.get(signature);
                    if (!".ra4".equals(new String(signature, StandardCharsets.US_ASCII))) {
                        LOGGER.debug("Invalid RealAudio 2.0 signature \"{}\"", (Object)new String(signature, StandardCharsets.US_ASCII));
                        return false;
                    }
                    reportedDataSize = buffer.getInt();
                    buffer.getShort();
                    reportedHeaderSize = buffer.getInt();
                    buffer = ByteBuffer.allocate(reportedHeaderSize);
                    channel.read(buffer);
                    buffer.flip();
                    buffer.getShort();
                    buffer.getInt();
                    buffer.getInt();
                    long bytesPerMinute = (long)buffer.getInt() & 0xFFFFFFFFL;
                    buffer.getInt();
                    buffer.getShort();
                    buffer.getShort();
                    buffer.getShort();
                    buffer.getShort();
                    if (version == 5) {
                        buffer.position(buffer.position() + 6);
                    }
                    short sampleRate = buffer.getShort();
                    buffer.getShort();
                    short bitDepth = buffer.getShort();
                    short nrChannels = buffer.getShort();
                    if (version == 4) {
                        buffer.position(buffer.get() + buffer.position());
                        fourCC = new byte[buffer.get()];
                        buffer.get(fourCC);
                    } else {
                        buffer.getFloat();
                        fourCC = new byte[4];
                        buffer.get(fourCC);
                    }
                    switch (fourCCString = new String(fourCC, StandardCharsets.US_ASCII).toLowerCase(Locale.ROOT)) {
                        case "lpcJ": {
                            audio.setCodec("ra14.4");
                            break;
                        }
                        case "28_8": {
                            audio.setCodec("ra28.8");
                            break;
                        }
                        case "dnet": {
                            audio.setCodec("ac3");
                            break;
                        }
                        case "sipr": {
                            audio.setCodec("sipro");
                            break;
                        }
                        case "cook": {
                            audio.setCodec("cook");
                            break;
                        }
                        case "atrc": {
                            audio.setCodec("atrac");
                            break;
                        }
                        case "ralf": {
                            audio.setCodec("ralf");
                            break;
                        }
                        case "raac": {
                            audio.setCodec("aac-lc");
                            break;
                        }
                        case "racp": {
                            audio.setCodec("he-aac");
                            break;
                        }
                        default: {
                            LOGGER.debug("Unknown RealMedia codec FourCC \"{}\" - parsing failed", (Object)fourCCString);
                            return false;
                        }
                    }
                    if (buffer.hasRemaining()) {
                        RealAudioParser.parseRealAudioMetaData(buffer, audioMetadata, version);
                    }
                    audio.setBitRate((int)(bytesPerMinute * 8L / 60L));
                    mediaInfo.setBitRate((int)(bytesPerMinute * 8L / 60L));
                    audio.setBitDepth(bitDepth);
                    audio.setNumberOfChannels(nrChannels);
                    audio.setSampleRate(sampleRate);
                    break;
                }
                default: {
                    LOGGER.error("Could not parse RealAudio format - unknown format version {}", (Object)version);
                    return false;
                }
            }
            mediaInfo.getAudioTracks().add(audio);
            long fileSize = 0L;
            if (channel instanceof FileChannel) {
                FileChannel fileChannel = (FileChannel)channel;
                fileSize = fileChannel.size();
                mediaInfo.setSize(fileSize);
            }
            if (audio.getBitRate() > 0) {
                int dataSize;
                if (fileSize > 0L && reportedHeaderSize > 0) {
                    int fullHeaderSize = reportedHeaderSize + (version == 3 ? 8 : 16);
                    dataSize = reportedDataSize > 0 ? (int)Math.min((long)reportedDataSize, fileSize - (long)fullHeaderSize) : (int)(fileSize - (long)fullHeaderSize);
                } else {
                    dataSize = reportedDataSize;
                }
                mediaInfo.setDuration((double)dataSize / (double)audio.getBitRate() * 8.0);
            }
        }
        catch (IOException e) {
            LOGGER.debug("Error while trying to parse RealAudio version 1 or 2: {}", (Object)e.getMessage());
            LOGGER.trace("", e);
            return false;
        }
        if (PMS.getConfiguration() != null && !PMS.getConfiguration().getAudioThumbnailMethod().equals(CoverSupplier.NONE) && (StringUtils.isNotBlank(audioMetadata.getSongname()) || StringUtils.isNotBlank(audioMetadata.getArtist()))) {
            ID3v1Tag tag = new ID3v1Tag();
            if (StringUtils.isNotBlank(audioMetadata.getSongname())) {
                tag.setTitle(audioMetadata.getSongname());
            }
            if (StringUtils.isNotBlank(audioMetadata.getArtist())) {
                tag.setArtist(audioMetadata.getArtist());
            }
            try {
                DLNAThumbnail thumbnail = DLNAThumbnail.toThumbnail(CoverUtil.get().getThumbnail(tag), 640, 480, ImagesUtil.ScaleType.MAX, ImageFormat.SOURCE, false);
                if (thumbnail != null) {
                    Long thumbId = ThumbnailStore.getId(thumbnail);
                    mediaInfo.setThumbnailId(thumbId);
                    mediaInfo.setThumbnailSource(ThumbnailSource.MUSICBRAINZ);
                }
            }
            catch (IOException e) {
                LOGGER.error("An error occurred while generating thumbnail for RealAudio source: [\"{}\", \"{}\"]", (Object)tag.getFirstTitle(), (Object)tag.getFirstArtist());
            }
        }
        mediaInfo.setMediaParser(PARSER_NAME);
        return true;
    }

    private static void parseRealAudioMetaData(ByteBuffer buffer, MediaAudioMetadata audioMetadata, short version) {
        buffer.position(buffer.position() + (version == 4 ? 3 : 4));
        byte b = buffer.get();
        if (b != 0) {
            byte[] title = new byte[Math.min(b & 0xFF, buffer.remaining())];
            buffer.get(title);
            String titleString = new String(title, StandardCharsets.US_ASCII);
            audioMetadata.setSongname(titleString);
        }
        if (buffer.hasRemaining() && (b = buffer.get()) != 0) {
            byte[] artist = new byte[Math.min(b & 0xFF, buffer.remaining())];
            buffer.get(artist);
            audioMetadata.setArtist(new String(artist, StandardCharsets.US_ASCII));
        }
    }
}

