/*
 * Decompiled with CFR 0.152.
 */
package net.pms.database;

import com.zaxxer.hikari.HikariDataSource;
import java.sql.Connection;
import java.sql.SQLException;
import net.pms.Messages;
import net.pms.database.DatabaseEmbedded;
import net.pms.database.DatabaseHelper;
import net.pms.gui.GuiManager;
import net.pms.util.UMSUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Database
extends DatabaseHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(Database.class);
    private final boolean embedded;
    private final HikariDataSource ds;
    private final String dbName;
    protected DatabaseStatus status;

    protected Database(String name) {
        this.dbName = name;
        this.status = DatabaseStatus.CLOSED;
        this.embedded = true;
        String jdbcUrl = DatabaseEmbedded.getJdbcUrl(name);
        this.ds = new HikariDataSource();
        this.ds.setJdbcUrl(jdbcUrl);
        this.ds.setUsername(DatabaseEmbedded.getDbUser());
    }

    public Connection getConnection() throws SQLException {
        return this.ds.getConnection();
    }

    public int getActiveConnections() throws SQLException {
        return this.ds.isRegisterMbeans() ? this.ds.getHikariPoolMXBean().getActiveConnections() : 0;
    }

    public boolean isOpened() {
        return this.status == DatabaseStatus.OPENED;
    }

    public boolean isEmbedded() {
        return this.embedded;
    }

    public synchronized void init(boolean force) {
        if (this.embedded) {
            DatabaseEmbedded.deleteDatabaseLock(this.dbName);
        }
        this.open();
        switch (this.status) {
            case OPENED: {
                this.onOpening(force);
                break;
            }
            case OPENFAILED: {
                this.onOpeningFail(force);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void open() {
        this.status = DatabaseStatus.OPENING;
        Connection conn = null;
        try {
            conn = this.getConnection();
            this.status = DatabaseStatus.OPENED;
        }
        catch (SQLException se) {
            boolean needRetry;
            this.status = DatabaseStatus.OPENFAILED;
            if (this.embedded) {
                needRetry = DatabaseEmbedded.openFailed(this.dbName, se);
            } else {
                LOGGER.debug("Database connection error, retrying in 10 seconds");
                UMSUtils.sleep(10000);
                needRetry = true;
            }
            if (needRetry) {
                try {
                    conn = this.getConnection();
                    this.status = DatabaseStatus.OPENED;
                }
                catch (SQLException se2) {
                    Database.showMessageDialog("TheLocalCacheCouldNotStarted", null);
                    LOGGER.debug("", se2);
                }
            }
        }
        finally {
            if (this.embedded && conn != null) {
                DatabaseEmbedded.checkTableStorageType(conn, this.dbName);
            }
            Database.close(conn);
        }
    }

    abstract void onOpening(boolean var1);

    abstract void onOpeningFail(boolean var1);

    public void close() {
        this.status = DatabaseStatus.CLOSING;
        try {
            Thread.sleep(50L);
            int activeConnections = this.getActiveConnections();
            int maxAttempts = 10;
            for (int curAttempts = 1; activeConnections > 0 && curAttempts <= maxAttempts; ++curAttempts) {
                LOGGER.trace("Database shutdown waiting 500 ms ({}/{}) for {} connections to close", curAttempts, maxAttempts, activeConnections);
                Thread.sleep(500L);
                activeConnections = this.getActiveConnections();
            }
            if (activeConnections > 0) {
                LOGGER.debug("Database shutdown will kill remaining connections ({}), db errors may occurs", (Object)activeConnections);
            }
        }
        catch (SQLException e) {
            LOGGER.error("Waiting DB connections", e);
        }
        catch (InterruptedException e) {
            LOGGER.debug("Interrupted while shutting down database..");
            LOGGER.trace("", e);
        }
        if (this.embedded) {
            try {
                Connection con = this.getConnection();
                DatabaseEmbedded.shutdown(con);
            }
            catch (SQLException ex) {
                LOGGER.error("shutdown DB ", ex);
            }
        }
        this.ds.close();
        this.status = DatabaseStatus.CLOSED;
    }

    public String getDatabaseFilename() {
        return this.embedded ? DatabaseEmbedded.getDatabaseFilename(this.dbName) : "";
    }

    public void createDatabaseReport() {
        if (this.embedded && CONFIGURATION.getDatabaseLogging()) {
            DatabaseEmbedded.createDatabaseReport(this.dbName);
        }
    }

    public static void showMessageDialog(String message, String dbDir) {
        GuiManager.showErrorMessage(String.format(Messages.getString(message), dbDir), Messages.getString("Error"));
    }

    public static enum DatabaseStatus {
        OPENING,
        OPENED,
        OPENFAILED,
        CLOSING,
        CLOSED;

    }
}

