# frozen_string_literal: true

# Arrays are ordered, integer-indexed collections of any object.
# Array indexing starts at 0, as in C or Java.  A negative index is
# assumed to be relative to the end of the array---that is, an index of -1
# indicates the last element of the array, -2 is the next to last
# element in the array, and so on.
class Array
  include Enumerable

  # Returns a new array populated with the given objects.
  #
  #   Array.[]( 1, 'a', /^A/ )
  #   Array[ 1, 'a', /^A/ ]
  #   [ 1, 'a', /^A/ ]
  def self.[](*args) end

  # Tries to convert +obj+ into an array, using +to_ary+ method.  Returns the
  # converted array or +nil+ if +obj+ cannot be converted for any reason.
  # This method can be used to check if an argument is an array.
  #
  #    Array.try_convert([1])   #=> [1]
  #    Array.try_convert("1")   #=> nil
  #
  #    if tmp = Array.try_convert(arg)
  #      # the argument is an array
  #    elsif tmp = String.try_convert(arg)
  #      # the argument is a string
  #    end
  def self.try_convert(obj) end

  # Returns a new array.
  #
  # In the first form, if no arguments are sent, the new array will be empty.
  # When a +size+ and an optional +obj+ are sent, an array is created with
  # +size+ copies of +obj+.  Take notice that all elements will reference the
  # same object +obj+.
  #
  # The second form creates a copy of the array passed as a parameter (the
  # array is generated by calling to_ary on the parameter).
  #
  #   first_array = ["Matz", "Guido"]
  #
  #   second_array = Array.new(first_array) #=> ["Matz", "Guido"]
  #
  #   first_array.equal? second_array       #=> false
  #
  # In the last form, an array of the given size is created.  Each element in
  # this array is created by passing the element's index to the given block
  # and storing the return value.
  #
  #   Array.new(3){ |index| index ** 2 }
  #   # => [0, 1, 4]
  #
  # == Common gotchas
  #
  # When sending the second parameter, the same object will be used as the
  # value for all the array elements:
  #
  #    a = Array.new(2, Hash.new)
  #    # => [{}, {}]
  #
  #    a[0]['cat'] = 'feline'
  #    a # => [{"cat"=>"feline"}, {"cat"=>"feline"}]
  #
  #    a[1]['cat'] = 'Felix'
  #    a # => [{"cat"=>"Felix"}, {"cat"=>"Felix"}]
  #
  # Since all the Array elements store the same hash, changes to one of them
  # will affect them all.
  #
  # If multiple copies are what you want, you should use the block
  # version which uses the result of that block each time an element
  # of the array needs to be initialized:
  #
  #    a = Array.new(2) { Hash.new }
  #    a[0]['cat'] = 'feline'
  #    a # => [{"cat"=>"feline"}, {}]
  def initialize(*several_variants) end

  # Set Intersection---Returns a new array
  # containing elements common to the two arrays, with no duplicates.
  #
  #    [ 1, 1, 3, 5 ] & [ 1, 2, 3 ]   #=> [ 1, 3 ]
  def &(other) end

  # Repetition---With a String argument, equivalent to
  # self.join(str). Otherwise, returns a new array
  # built by concatenating the _int_ copies of +self+.
  #
  #    [ 1, 2, 3 ] * 3    #=> [ 1, 2, 3, 1, 2, 3, 1, 2, 3 ]
  #    [ 1, 2, 3 ] * ","  #=> "1,2,3"
  def *(*several_variants) end

  # Concatenation---Returns a new array built by concatenating the
  # two arrays together to produce a third array.
  #
  #    [ 1, 2, 3 ] + [ 4, 5 ]    #=> [ 1, 2, 3, 4, 5 ]
  def +(other) end

  # Array Difference---Returns a new array that is a copy of
  # the original array, removing any items that also appear in
  # <i>other_ary</i>. (If you need set-like behavior, see the
  # library class Set.)
  #
  #    [ 1, 1, 2, 2, 3, 3, 4, 5 ] - [ 1, 2, 4 ]  #=>  [ 3, 3, 5 ]
  def -(other) end

  # Append---Pushes the given object on to the end of this array. This
  # expression returns the array itself, so several appends
  # may be chained together.
  #
  #    [ 1, 2 ] << "c" << "d" << [ 3, 4 ]
  #            #=>  [ 1, 2, "c", "d", [ 3, 4 ] ]
  def <<(obj) end

  # Comparison---Returns an integer (-1, 0,
  # or +1) if this array is less than, equal to, or greater than
  # <i>other_ary</i>.  Each object in each array is compared
  # (using <=>). If any value isn't
  # equal, then that inequality is the return value. If all the
  # values found are equal, then the return is based on a
  # comparison of the array lengths.  Thus, two arrays are
  # ``equal'' according to <code>Array#<=></code> if and only if they have
  # the same length and the value of each element is equal to the
  # value of the corresponding element in the other array.
  #
  #    [ "a", "a", "c" ]    <=> [ "a", "b", "c" ]   #=> -1
  #    [ 1, 2, 3, 4, 5, 6 ] <=> [ 1, 2 ]            #=> +1
  def <=>(other) end

  # Equality---Two arrays are equal if they contain the same number
  # of elements and if each element is equal to (according to
  # Object.==) the corresponding element in the other array.
  #
  #    [ "a", "c" ]    == [ "a", "c", 7 ]     #=> false
  #    [ "a", "c", 7 ] == [ "a", "c", 7 ]     #=> true
  #    [ "a", "c", 7 ] == [ "a", "d", "f" ]   #=> false
  def ==(other) end

  # Element Reference---Returns the element at _index_,
  # or returns a subarray starting at _start_ and
  # continuing for _length_ elements, or returns a subarray
  # specified by _range_.
  # Negative indices count backward from the end of the
  # array (-1 is the last element). Returns +nil+ if the index
  # (or starting index) are out of range.
  #
  #    a = [ "a", "b", "c", "d", "e" ]
  #    a[2] +  a[0] + a[1]    #=> "cab"
  #    a[6]                   #=> nil
  #    a[1, 2]                #=> [ "b", "c" ]
  #    a[1..3]                #=> [ "b", "c", "d" ]
  #    a[4..7]                #=> [ "e" ]
  #    a[6..10]               #=> nil
  #    a[-3, 3]               #=> [ "c", "d", "e" ]
  #    # special cases
  #    a[5]                   #=> nil
  #    a[5, 1]                #=> []
  #    a[5..10]               #=> []
  def [](*several_variants) end
  alias slice []

  # Element Assignment---Sets the element at _index_,
  # or replaces a subarray starting at _start_ and
  # continuing for _length_ elements, or replaces a subarray
  # specified by _range_.  If indices are greater than
  # the current capacity of the array, the array grows
  # automatically. A negative indices will count backward
  # from the end of the array. Inserts elements if _length_ is
  # zero. An +IndexError+ is raised if a negative index points
  # past the beginning of the array. See also
  # <code>Array#push</code>, and <code>Array#unshift</code>.
  #
  #    a = Array.new
  #    a[4] = "4";                 #=> [nil, nil, nil, nil, "4"]
  #    a[0, 3] = [ 'a', 'b', 'c' ] #=> ["a", "b", "c", nil, "4"]
  #    a[1..2] = [ 1, 2 ]          #=> ["a", 1, 2, nil, "4"]
  #    a[0, 2] = "?"               #=> ["?", 2, nil, "4"]
  #    a[0..2] = "A"               #=> ["A", "4"]
  #    a[-1]   = "Z"               #=> ["A", "Z"]
  #    a[1..-1] = nil              #=> ["A", nil]
  #    a[1..-1] = []               #=> ["A"]
  def []=(index, obj) end

  # Set Union---Returns a new array by joining this array with
  # <i>other_ary</i>, removing duplicates.
  #
  #    [ "a", "b", "c" ] | [ "c", "d", "a" ]
  #           #=> [ "a", "b", "c", "d" ]
  def |(other) end

  # Searches through an array whose elements are also arrays
  # comparing _obj_ with the first element of each contained array
  # using obj.==.
  # Returns the first contained array that matches (that
  # is, the first associated array),
  # or +nil+ if no match is found.
  # See also <code>Array#rassoc</code>.
  #
  #    s1 = [ "colors", "red", "blue", "green" ]
  #    s2 = [ "letters", "a", "b", "c" ]
  #    s3 = "foo"
  #    a  = [ s1, s2, s3 ]
  #    a.assoc("letters")  #=> [ "letters", "a", "b", "c" ]
  #    a.assoc("foo")      #=> nil
  def assoc(obj) end

  # Returns the element at _index_. A
  # negative index counts from the end of +self+.  Returns +nil+
  # if the index is out of range. See also <code>Array#[]</code>.
  #
  #    a = [ "a", "b", "c", "d", "e" ]
  #    a.at(0)     #=> "a"
  #    a.at(-1)    #=> "e"
  def at(index) end

  # Removes all elements from +self+.
  #
  #    a = [ "a", "b", "c", "d", "e" ]
  #    a.clear    #=> [ ]
  def clear; end

  # Invokes <i>block</i> once for each element of +self+. Creates a
  # new array containing the values returned by the block.
  # See also <code>Enumerable#collect</code>.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.collect {|x| x + "!" }   #=> ["a!", "b!", "c!", "d!"]
  #    a                          #=> ["a", "b", "c", "d"]
  def collect; end
  alias map collect

  # Invokes the block once for each element of +self+, replacing the
  # element with the value returned by _block_.
  # See also <code>Enumerable#collect</code>.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.collect! {|x| x + "!" }
  #    a             #=>  [ "a!", "b!", "c!", "d!" ]
  def collect!; end
  alias map! collect!

  # When invoked with a block, yields all combinations of length <i>n</i>
  # of elements from <i>ary</i> and then returns <i>ary</i> itself.
  # The implementation makes no guarantees about the order in which
  # the combinations are yielded.
  #
  # If no block is given, an enumerator is returned instead.
  #
  # Examples:
  #
  #     a = [1, 2, 3, 4]
  #     a.combination(1).to_a  #=> [[1],[2],[3],[4]]
  #     a.combination(2).to_a  #=> [[1,2],[1,3],[1,4],[2,3],[2,4],[3,4]]
  #     a.combination(3).to_a  #=> [[1,2,3],[1,2,4],[1,3,4],[2,3,4]]
  #     a.combination(4).to_a  #=> [[1,2,3,4]]
  #     a.combination(0).to_a  #=> [[]] # one combination of length 0
  #     a.combination(5).to_a  #=> []   # no combinations of length 5
  def combination(n) end

  # Returns a copy of +self+ with all +nil+ elements removed.
  #
  #    [ "a", nil, "b", nil, "c", nil ].compact
  #                      #=> [ "a", "b", "c" ]
  def compact; end

  # Removes +nil+ elements from the array.
  # Returns +nil+ if no changes were made, otherwise returns
  # <i>ary</i>.
  #
  #    [ "a", nil, "b", nil, "c" ].compact! #=> [ "a", "b", "c" ]
  #    [ "a", "b", "c" ].compact!           #=> nil
  def compact!; end

  # Appends the elements of <i>other_ary</i> to +self+.
  #
  #    [ "a", "b" ].concat( ["c", "d"] ) #=> [ "a", "b", "c", "d" ]
  def concat(other_ary) end

  # Returns the number of elements.  If an argument is given, counts
  # the number of elements which equals to <i>obj</i>.  If a block is
  # given, counts the number of elements yielding a true value.
  #
  #    ary = [1, 2, 4, 2]
  #    ary.count             #=> 4
  #    ary.count(2)          #=> 2
  #    ary.count{|x|x%2==0}  #=> 3
  def count(*several_variants) end

  # Calls <i>block</i> for each element repeatedly _n_ times or
  # forever if none or +nil+ is given.  If a non-positive number is
  # given or the array is empty, does nothing.  Returns +nil+ if the
  # loop has finished without getting interrupted.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = ["a", "b", "c"]
  #    a.cycle {|x| puts x }  # print, a, b, c, a, b, c,.. forever.
  #    a.cycle(2) {|x| puts x }  # print, a, b, c, a, b, c.
  def cycle(n = nil) end

  # Deletes items from +self+ that are equal to <i>obj</i>.
  # If any items are found, returns <i>obj</i>.   If
  # the item is not found, returns <code>nil</code>. If the optional
  # code block is given, returns the result of <i>block</i> if the item
  # is not found.  (To remove <code>nil</code> elements and
  # get an informative return value, use #compact!)
  #
  #    a = [ "a", "b", "b", "b", "c" ]
  #    a.delete("b")                   #=> "b"
  #    a                               #=> ["a", "c"]
  #    a.delete("z")                   #=> nil
  #    a.delete("z") { "not found" }   #=> "not found"
  def delete(obj) end

  # Deletes the element at the specified index, returning that element,
  # or <code>nil</code> if the index is out of range. See also
  # <code>Array#slice!</code>.
  #
  #    a = %w( ant bat cat dog )
  #    a.delete_at(2)    #=> "cat"
  #    a                 #=> ["ant", "bat", "dog"]
  #    a.delete_at(99)   #=> nil
  def delete_at(index) end

  # Deletes every element of +self+ for which <i>block</i> evaluates
  # to true.
  # The array is changed instantly every time the block is called and
  # not after the iteration is over.
  # See also <code>Array#reject!</code>
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c" ]
  #    a.delete_if {|x| x >= "b" }   #=> ["a"]
  def delete_if; end

  # Drops first n elements from +ary+ and returns the rest of
  # the elements in an array.
  #
  #    a = [1, 2, 3, 4, 5, 0]
  #    a.drop(3)             #=> [4, 5, 0]
  def drop(n) end

  # Drops elements up to, but not including, the first element for
  # which the block returns +nil+ or +false+ and returns an array
  # containing the remaining elements.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [1, 2, 3, 4, 5, 0]
  #    a.drop_while {|i| i < 3 }   #=> [3, 4, 5, 0]
  def drop_while; end

  # Calls <i>block</i> once for each element in +self+, passing that
  # element as a parameter.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c" ]
  #    a.each {|x| print x, " -- " }
  #
  # produces:
  #
  #    a -- b -- c --
  def each; end

  # Same as <code>Array#each</code>, but passes the index of the element
  # instead of the element itself.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c" ]
  #    a.each_index {|x| print x, " -- " }
  #
  # produces:
  #
  #    0 -- 1 -- 2 --
  def each_index; end

  # Returns <code>true</code> if +self+ contains no elements.
  #
  #    [].empty?   #=> true
  def empty?; end

  # Returns <code>true</code> if +self+ and _other_ are the same object,
  # or are both arrays with the same content.
  def eql?(other) end

  # Tries to return the element at position <i>index</i>. If the index
  # lies outside the array, the first form throws an
  # <code>IndexError</code> exception, the second form returns
  # <i>default</i>, and the third form returns the value of invoking
  # the block, passing in the index. Negative values of <i>index</i>
  # count from the end of the array.
  #
  #    a = [ 11, 22, 33, 44 ]
  #    a.fetch(1)               #=> 22
  #    a.fetch(-1)              #=> 44
  #    a.fetch(4, 'cat')        #=> "cat"
  #    a.fetch(4) { |i| i*i }   #=> 16
  def fetch(*several_variants) end

  # The first three forms set the selected elements of +self+ (which
  # may be the entire array) to <i>obj</i>. A <i>start</i> of
  # <code>nil</code> is equivalent to zero. A <i>length</i> of
  # <code>nil</code> is equivalent to <i>self.length</i>. The last three
  # forms fill the array with the value of the block. The block is
  # passed the absolute index of each element to be filled.
  # Negative values of <i>start</i> count from the end of the array.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.fill("x")              #=> ["x", "x", "x", "x"]
  #    a.fill("z", 2, 2)        #=> ["x", "x", "z", "z"]
  #    a.fill("y", 0..1)        #=> ["y", "y", "z", "z"]
  #    a.fill {|i| i*i}         #=> [0, 1, 4, 9]
  #    a.fill(-2) {|i| i*i*i}   #=> [0, 1, 8, 27]
  def fill(*several_variants) end

  # Returns the index of the first object in +self+ such that the object is
  # <code>==</code> to <i>obj</i>. If a block is given instead of an
  # argument, returns index of first object for which <em>block</em> is true.
  # Returns <code>nil</code> if no match is found.
  # See also <code>Array#rindex</code>.
  #
  # If neither block nor argument is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "c" ]
  #    a.index("b")        #=> 1
  #    a.index("z")        #=> nil
  #    a.index{|x|x=="b"}  #=> 1
  #
  # This is an alias of <code>#find_index</code>.
  def find_index(p1) end
  alias index find_index

  # Returns the first element, or the first +n+ elements, of the array.
  # If the array is empty, the first form returns <code>nil</code>, and the
  # second form returns an empty array.
  #
  #    a = [ "q", "r", "s", "t" ]
  #    a.first     #=> "q"
  #    a.first(2)  #=> ["q", "r"]
  def first(*several_variants) end

  # Returns a new array that is a one-dimensional flattening of this
  # array (recursively). That is, for every element that is an array,
  # extract its elements into the new array.  If the optional
  # <i>level</i> argument determines the level of recursion to flatten.
  #
  #    s = [ 1, 2, 3 ]           #=> [1, 2, 3]
  #    t = [ 4, 5, 6, [7, 8] ]   #=> [4, 5, 6, [7, 8]]
  #    a = [ s, t, 9, 10 ]       #=> [[1, 2, 3], [4, 5, 6, [7, 8]], 9, 10]
  #    a.flatten                 #=> [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]
  #    a = [ 1, 2, [3, [4, 5] ] ]
  #    a.flatten(1)              #=> [1, 2, 3, [4, 5]]
  def flatten(*several_variants) end

  # Flattens +self+ in place.
  # Returns <code>nil</code> if no modifications were made (i.e.,
  # <i>ary</i> contains no subarrays.)  If the optional <i>level</i>
  # argument determines the level of recursion to flatten.
  #
  #    a = [ 1, 2, [3, [4, 5] ] ]
  #    a.flatten!   #=> [1, 2, 3, 4, 5]
  #    a.flatten!   #=> nil
  #    a            #=> [1, 2, 3, 4, 5]
  #    a = [ 1, 2, [3, [4, 5] ] ]
  #    a.flatten!(1) #=> [1, 2, 3, [4, 5]]
  def flatten!(*several_variants) end

  # Return <code>true</code> if this array is frozen (or temporarily frozen
  # while being sorted).
  def frozen?; end

  # Compute a hash-code for this array. Two arrays with the same content
  # will have the same hash code (and will compare using <code>eql?</code>).
  def hash; end

  # Returns <code>true</code> if the given object is present in
  # +self+ (that is, if any object <code>==</code> <i>anObject</i>),
  # <code>false</code> otherwise.
  #
  #    a = [ "a", "b", "c" ]
  #    a.include?("b")   #=> true
  #    a.include?("z")   #=> false
  def include?(obj) end

  # Replaces the contents of +self+ with the contents of
  # <i>other_ary</i>, truncating or expanding if necessary.
  #
  #    a = [ "a", "b", "c", "d", "e" ]
  #    a.replace([ "x", "y", "z" ])   #=> ["x", "y", "z"]
  #    a                              #=> ["x", "y", "z"]
  def initialize_copy(p1) end
  alias replace initialize_copy

  # Inserts the given values before the element with the given index
  # (which may be negative).
  #
  #    a = %w{ a b c d }
  #    a.insert(2, 99)         #=> ["a", "b", 99, "c", "d"]
  #    a.insert(-2, 1, 2, 3)   #=> ["a", "b", 99, "c", 1, 2, 3, "d"]
  def insert(*args) end

  # Creates a string representation of +self+.
  def inspect; end
  alias to_s inspect

  # Returns a string created by converting each element of the array to
  # a string, separated by <i>sep</i>.
  #
  #    [ "a", "b", "c" ].join        #=> "abc"
  #    [ "a", "b", "c" ].join("-")   #=> "a-b-c"
  def join(sep = $,) end

  # Deletes every element of +self+ for which <i>block</i> evaluates
  # to false.
  # See also <code>Array#select!</code>
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = %w{ a b c d e f }
  #    a.keep_if {|v| v =~ /[aeiou]/}   #=> ["a", "e"]
  def keep_if; end

  # Returns the last element(s) of +self+. If the array is empty,
  # the first form returns <code>nil</code>.
  #
  #    a = [ "w", "x", "y", "z" ]
  #    a.last     #=> "z"
  #    a.last(2)  #=> ["y", "z"]
  def last(*several_variants) end

  # Returns the number of elements in +self+. May be zero.
  #
  #    [ 1, 2, 3, 4, 5 ].length   #=> 5
  def length; end
  alias size length

  # Packs the contents of <i>arr</i> into a binary sequence according to
  # the directives in <i>aTemplateString</i> (see the table below)
  # Directives ``A,'' ``a,'' and ``Z'' may be followed by a count,
  # which gives the width of the resulting field. The remaining
  # directives also may take a count, indicating the number of array
  # elements to convert. If the count is an asterisk
  # (``<code>*</code>''), all remaining array elements will be
  # converted. Any of the directives ``<code>sSiIlL</code>'' may be
  # followed by an underscore (``<code>_</code>'') or
  # exclamation mark (``<code>!</code>'') to use the underlying
  # platform's native size for the specified type; otherwise, they use a
  # platform-independent size. Spaces are ignored in the template
  # string. See also <code>String#unpack</code>.
  #
  #    a = [ "a", "b", "c" ]
  #    n = [ 65, 66, 67 ]
  #    a.pack("A3A3A3")   #=> "a  b  c  "
  #    a.pack("a3a3a3")   #=> "a\000\000b\000\000c\000\000"
  #    n.pack("ccc")      #=> "ABC"
  #
  # Directives for +pack+.
  #
  #  Integer      | Array   |
  #  Directive    | Element | Meaning
  #  ---------------------------------------------------------------------------
  #     C         | Integer | 8-bit unsigned (unsigned char)
  #     S         | Integer | 16-bit unsigned, native endian (uint16_t)
  #     L         | Integer | 32-bit unsigned, native endian (uint32_t)
  #     Q         | Integer | 64-bit unsigned, native endian (uint64_t)
  #               |         |
  #     c         | Integer | 8-bit signed (signed char)
  #     s         | Integer | 16-bit signed, native endian (int16_t)
  #     l         | Integer | 32-bit signed, native endian (int32_t)
  #     q         | Integer | 64-bit signed, native endian (int64_t)
  #               |         |
  #     S_, S!    | Integer | unsigned short, native endian
  #     I, I_, I! | Integer | unsigned int, native endian
  #     L_, L!    | Integer | unsigned long, native endian
  #               |         |
  #     s_, s!    | Integer | signed short, native endian
  #     i, i_, i! | Integer | signed int, native endian
  #     l_, l!    | Integer | signed long, native endian
  #               |         |
  #     S> L> Q>  | Integer | same as the directives without ">" except
  #     s> l> q>  |         | big endian
  #     S!> I!>   |         | (available since Ruby 1.9.3)
  #     L!>       |         | "S>" is same as "n"
  #     s!> i!>   |         | "L>" is same as "N"
  #     l!>       |         |
  #               |         |
  #     S< L< Q<  | Integer | same as the directives without "<" except
  #     s< l< q<  |         | little endian
  #     S!< I!<   |         | (available since Ruby 1.9.3)
  #     L!<       |         | "S<" is same as "v"
  #     s!< i!<   |         | "L<" is same as "V"
  #     l!<       |         |
  #               |         |
  #     n         | Integer | 16-bit unsigned, network (big-endian) byte order
  #     N         | Integer | 32-bit unsigned, network (big-endian) byte order
  #     v         | Integer | 16-bit unsigned, VAX (little-endian) byte order
  #     V         | Integer | 32-bit unsigned, VAX (little-endian) byte order
  #               |         |
  #     U         | Integer | UTF-8 character
  #     w         | Integer | BER-compressed integer
  #
  #  Float        |         |
  #  Directive    |         | Meaning
  #  ---------------------------------------------------------------------------
  #     D, d      | Float   | double-precision, native format
  #     F, f      | Float   | single-precision, native format
  #     E         | Float   | double-precision, little-endian byte order
  #     e         | Float   | single-precision, little-endian byte order
  #     G         | Float   | double-precision, network (big-endian) byte order
  #     g         | Float   | single-precision, network (big-endian) byte order
  #
  #  String       |         |
  #  Directive    |         | Meaning
  #  ---------------------------------------------------------------------------
  #     A         | String  | arbitrary binary string (space padded, count is width)
  #     a         | String  | arbitrary binary string (null padded, count is width)
  #     Z         | String  | same as ``a'', except that null is added with *
  #     B         | String  | bit string (MSB first)
  #     b         | String  | bit string (LSB first)
  #     H         | String  | hex string (high nibble first)
  #     h         | String  | hex string (low nibble first)
  #     u         | String  | UU-encoded string
  #     M         | String  | quoted printable, MIME encoding (see RFC2045)
  #     m         | String  | base64 encoded string (see RFC 2045, count is width)
  #               |         | (if count is 0, no line feed are added, see RFC 4648)
  #     P         | String  | pointer to a structure (fixed-length string)
  #     p         | String  | pointer to a null-terminated string
  #
  #  Misc.        |         |
  #  Directive    |         | Meaning
  #  ---------------------------------------------------------------------------
  #     @         | ---     | moves to absolute position
  #     X         | ---     | back up a byte
  #     x         | ---     | null byte
  def pack(template_string) end

  # When invoked with a block, yield all permutations of length <i>n</i>
  # of the elements of <i>ary</i>, then return the array itself.
  # If <i>n</i> is not specified, yield all permutations of all elements.
  # The implementation makes no guarantees about the order in which
  # the permutations are yielded.
  #
  # If no block is given, an enumerator is returned instead.
  #
  # Examples:
  #
  #     a = [1, 2, 3]
  #     a.permutation.to_a     #=> [[1,2,3],[1,3,2],[2,1,3],[2,3,1],[3,1,2],[3,2,1]]
  #     a.permutation(1).to_a  #=> [[1],[2],[3]]
  #     a.permutation(2).to_a  #=> [[1,2],[1,3],[2,1],[2,3],[3,1],[3,2]]
  #     a.permutation(3).to_a  #=> [[1,2,3],[1,3,2],[2,1,3],[2,3,1],[3,1,2],[3,2,1]]
  #     a.permutation(0).to_a  #=> [[]] # one permutation of length 0
  #     a.permutation(4).to_a  #=> []   # no permutations of length 4
  def permutation(*several_variants) end

  # Removes the last element from +self+ and returns it, or
  # <code>nil</code> if the array is empty.
  #
  # If a number _n_ is given, returns an array of the last n elements
  # (or less) just like <code>array.slice!(-n, n)</code> does.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.pop     #=> "d"
  #    a.pop(2)  #=> ["b", "c"]
  #    a         #=> ["a"]
  def pop(*several_variants) end

  # Returns an array of all combinations of elements from all arrays.
  # The length of the returned array is the product of the length
  # of +self+ and the argument arrays.
  # If given a block, <i>product</i> will yield all combinations
  # and return +self+ instead.
  #
  #    [1,2,3].product([4,5])     #=> [[1,4],[1,5],[2,4],[2,5],[3,4],[3,5]]
  #    [1,2].product([1,2])       #=> [[1,1],[1,2],[2,1],[2,2]]
  #    [1,2].product([3,4],[5,6]) #=> [[1,3,5],[1,3,6],[1,4,5],[1,4,6],
  #                               #     [2,3,5],[2,3,6],[2,4,5],[2,4,6]]
  #    [1,2].product()            #=> [[1],[2]]
  #    [1,2].product([])          #=> []
  def product(other_ary, *args) end

  # Append---Pushes the given object(s) on to the end of this array. This
  # expression returns the array itself, so several appends
  # may be chained together.
  #
  #    a = [ "a", "b", "c" ]
  #    a.push("d", "e", "f")
  #            #=> ["a", "b", "c", "d", "e", "f"]
  def push(obj, *args) end

  # Searches through the array whose elements are also arrays. Compares
  # _obj_ with the second element of each contained array using
  # <code>==</code>. Returns the first contained array that matches. See
  # also <code>Array#assoc</code>.
  #
  #    a = [ [ 1, "one"], [2, "two"], [3, "three"], ["ii", "two"] ]
  #    a.rassoc("two")    #=> [2, "two"]
  #    a.rassoc("four")   #=> nil
  def rassoc(obj) end

  # Returns a new array containing the items in +self+
  # for which the block is not true.
  # See also <code>Array#delete_if</code>
  #
  # If no block is given, an enumerator is returned instead.
  def reject; end

  # Equivalent to <code>Array#delete_if</code>, deleting elements from
  # +self+ for which the block evaluates to true, but returns
  # <code>nil</code> if no changes were made.
  # The array is changed instantly every time the block is called and
  # not after the iteration is over.
  # See also <code>Enumerable#reject</code> and <code>Array#delete_if</code>.
  #
  # If no block is given, an enumerator is returned instead.
  def reject!; end

  # When invoked with a block, yields all repeated combinations of
  # length <i>n</i> of elements from <i>ary</i> and then returns
  # <i>ary</i> itself.
  # The implementation makes no guarantees about the order in which
  # the repeated combinations are yielded.
  #
  # If no block is given, an enumerator is returned instead.
  #
  # Examples:
  #
  #     a = [1, 2, 3]
  #     a.repeated_combination(1).to_a  #=> [[1], [2], [3]]
  #     a.repeated_combination(2).to_a  #=> [[1,1],[1,2],[1,3],[2,2],[2,3],[3,3]]
  #     a.repeated_combination(3).to_a  #=> [[1,1,1],[1,1,2],[1,1,3],[1,2,2],[1,2,3],
  #                                     #    [1,3,3],[2,2,2],[2,2,3],[2,3,3],[3,3,3]]
  #     a.repeated_combination(4).to_a  #=> [[1,1,1,1],[1,1,1,2],[1,1,1,3],[1,1,2,2],[1,1,2,3],
  #                                     #    [1,1,3,3],[1,2,2,2],[1,2,2,3],[1,2,3,3],[1,3,3,3],
  #                                     #    [2,2,2,2],[2,2,2,3],[2,2,3,3],[2,3,3,3],[3,3,3,3]]
  #     a.repeated_combination(0).to_a  #=> [[]] # one combination of length 0
  def repeated_combination(n) end

  # When invoked with a block, yield all repeated permutations of length
  # <i>n</i> of the elements of <i>ary</i>, then return the array itself.
  # The implementation makes no guarantees about the order in which
  # the repeated permutations are yielded.
  #
  # If no block is given, an enumerator is returned instead.
  #
  # Examples:
  #
  #     a = [1, 2]
  #     a.repeated_permutation(1).to_a  #=> [[1], [2]]
  #     a.repeated_permutation(2).to_a  #=> [[1,1],[1,2],[2,1],[2,2]]
  #     a.repeated_permutation(3).to_a  #=> [[1,1,1],[1,1,2],[1,2,1],[1,2,2],
  #                                     #    [2,1,1],[2,1,2],[2,2,1],[2,2,2]]
  #     a.repeated_permutation(0).to_a  #=> [[]] # one permutation of length 0
  def repeated_permutation(n) end

  # Returns a new array containing +self+'s elements in reverse order.
  #
  #    [ "a", "b", "c" ].reverse   #=> ["c", "b", "a"]
  #    [ 1 ].reverse               #=> [1]
  def reverse; end

  # Reverses +self+ in place.
  #
  #    a = [ "a", "b", "c" ]
  #    a.reverse!       #=> ["c", "b", "a"]
  #    a                #=> ["c", "b", "a"]
  def reverse!; end

  # Same as <code>Array#each</code>, but traverses +self+ in reverse
  # order.
  #
  #    a = [ "a", "b", "c" ]
  #    a.reverse_each {|x| print x, " " }
  #
  # produces:
  #
  #    c b a
  def reverse_each; end

  # Returns the index of the last object in +self+
  # <code>==</code> to <i>obj</i>. If a block is given instead of an
  # argument, returns index of first object for which <em>block</em> is
  # true, starting from the last object.
  # Returns <code>nil</code> if no match is found.
  # See also <code>Array#index</code>.
  #
  # If neither block nor argument is given, an enumerator is returned instead.
  #
  #    a = [ "a", "b", "b", "b", "c" ]
  #    a.rindex("b")             #=> 3
  #    a.rindex("z")             #=> nil
  #    a.rindex { |x| x == "b" } #=> 3
  def rindex(*several_variants) end

  # Returns new array by rotating +self+ so that the element at
  # +cnt+ in +self+ is the first element of the new array. If +cnt+
  # is negative then it rotates in the opposite direction.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.rotate         #=> ["b", "c", "d", "a"]
  #    a                #=> ["a", "b", "c", "d"]
  #    a.rotate(2)      #=> ["c", "d", "a", "b"]
  #    a.rotate(-3)     #=> ["b", "c", "d", "a"]
  def rotate(cnt = 1) end

  # Rotates +self+ in place so that the element at +cnt+ comes first,
  # and returns +self+.  If +cnt+ is negative then it rotates in
  # the opposite direction.
  #
  #    a = [ "a", "b", "c", "d" ]
  #    a.rotate!        #=> ["b", "c", "d", "a"]
  #    a                #=> ["b", "c", "d", "a"]
  #    a.rotate!(2)     #=> ["d", "a", "b", "c"]
  #    a.rotate!(-3)    #=> ["a", "b", "c", "d"]
  def rotate!(cnt = 1) end

  # Choose a random element or +n+ random elements from the array. The elements
  # are chosen by using random and unique indices into the array in order to
  # ensure that an element doesn't repeat itself unless the array already
  # contained duplicate elements. If the array is empty the first form returns
  # <code>nil</code> and the second form returns an empty array.
  #
  # If +rng+ is given, it will be used as the random number generator.
  def sample(*several_variants) end

  # Invokes the block passing in successive elements from +self+,
  # returning an array containing those elements for which the block
  # returns a true value (equivalent to <code>Enumerable#select</code>).
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = %w{ a b c d e f }
  #    a.select {|v| v =~ /[aeiou]/}   #=> ["a", "e"]
  def select; end

  # Invokes the block passing in successive elements from
  # +self+, deleting elements for which the block returns a
  # false value. It returns +self+ if changes were made,
  # otherwise it returns <code>nil</code>.
  # See also <code>Array#keep_if</code>
  #
  # If no block is given, an enumerator is returned instead.
  def select!; end

  # Returns the first element of +self+ and removes it (shifting all
  # other elements down by one). Returns <code>nil</code> if the array
  # is empty.
  #
  # If a number _n_ is given, returns an array of the first n elements
  # (or less) just like <code>array.slice!(0, n)</code> does.
  #
  #    args = [ "-m", "-q", "filename" ]
  #    args.shift     #=> "-m"
  #    args           #=> ["-q", "filename"]
  #
  #    args = [ "-m", "-q", "filename" ]
  #    args.shift(2)  #=> ["-m", "-q"]
  #    args           #=> ["filename"]
  def shift(*several_variants) end

  # Returns a new array with elements of this array shuffled.
  #
  #    a = [ 1, 2, 3 ]           #=> [1, 2, 3]
  #    a.shuffle                 #=> [2, 3, 1]
  #
  # If +rng+ is given, it will be used as the random number generator.
  #
  #    a.shuffle(random: Random.new(1))  #=> [1, 3, 2]
  def shuffle(*several_variants) end

  # Shuffles elements in +self+ in place.
  # If +rng+ is given, it will be used as the random number generator.
  def shuffle!(*several_variants) end

  # Deletes the element(s) given by an index (optionally with a length)
  # or by a range. Returns the deleted object (or objects), or
  # <code>nil</code> if the index is out of range.
  #
  #    a = [ "a", "b", "c" ]
  #    a.slice!(1)     #=> "b"
  #    a               #=> ["a", "c"]
  #    a.slice!(-1)    #=> "c"
  #    a               #=> ["a"]
  #    a.slice!(100)   #=> nil
  #    a               #=> ["a"]
  def slice!(*several_variants) end

  # Returns a new array created by sorting +self+. Comparisons for
  # the sort will be done using the <code><=></code> operator or using
  # an optional code block. The block implements a comparison between
  # <i>a</i> and <i>b</i>, returning -1, 0, or +1. See also
  # <code>Enumerable#sort_by</code>.
  #
  #    a = [ "d", "a", "e", "c", "b" ]
  #    a.sort                    #=> ["a", "b", "c", "d", "e"]
  #    a.sort {|x,y| y <=> x }   #=> ["e", "d", "c", "b", "a"]
  def sort; end

  # Sorts +self+. Comparisons for
  # the sort will be done using the <code><=></code> operator or using
  # an optional code block. The block implements a comparison between
  # <i>a</i> and <i>b</i>, returning -1, 0, or +1. See also
  # <code>Enumerable#sort_by</code>.
  #
  #    a = [ "d", "a", "e", "c", "b" ]
  #    a.sort!                    #=> ["a", "b", "c", "d", "e"]
  #    a.sort! {|x,y| y <=> x }   #=> ["e", "d", "c", "b", "a"]
  def sort!; end

  # Sorts +self+ in place using a set of keys generated by mapping the
  # values in +self+ through the given block.
  #
  # If no block is given, an enumerator is returned instead.
  def sort_by!; end

  # Returns first n elements from <i>ary</i>.
  #
  #    a = [1, 2, 3, 4, 5, 0]
  #    a.take(3)             #=> [1, 2, 3]
  def take(n) end

  # Passes elements to the block until the block returns +nil+ or +false+,
  # then stops iterating and returns an array of all prior elements.
  #
  # If no block is given, an enumerator is returned instead.
  #
  #    a = [1, 2, 3, 4, 5, 0]
  #    a.take_while {|i| i < 3 }   #=> [1, 2]
  def take_while; end

  # Returns +self+. If called on a subclass of Array, converts
  # the receiver to an Array object.
  def to_a; end

  # Returns +self+.
  def to_ary; end

  # Assumes that +self+ is an array of arrays and transposes the
  # rows and columns.
  #
  #    a = [[1,2], [3,4], [5,6]]
  #    a.transpose   #=> [[1, 3, 5], [2, 4, 6]]
  def transpose; end

  # Returns a new array by removing duplicate values in +self+. If a block
  # is given, it will use the return value of the block for comparison.
  #
  #    a = [ "a", "a", "b", "b", "c" ]
  #    a.uniq   # => ["a", "b", "c"]
  #
  #    b = [["student","sam"], ["student","george"], ["teacher","matz"]]
  #    b.uniq { |s| s.first } # => [["student", "sam"], ["teacher", "matz"]]
  def uniq; end

  # Removes duplicate elements from +self+. If a block is given,
  # it will use the return value of the block for comparison.
  # Returns <code>nil</code> if no changes are made (that is, no
  # duplicates are found).
  #
  #    a = [ "a", "a", "b", "b", "c" ]
  #    a.uniq!   # => ["a", "b", "c"]
  #
  #    b = [ "a", "b", "c" ]
  #    b.uniq!   # => nil
  #
  #    c = [["student","sam"], ["student","george"], ["teacher","matz"]]
  #    c.uniq! { |s| s.first } # => [["student", "sam"], ["teacher", "matz"]]
  def uniq!; end

  # Prepends objects to the front of +self+,
  # moving other elements upwards.
  #
  #    a = [ "b", "c", "d" ]
  #    a.unshift("a")   #=> ["a", "b", "c", "d"]
  #    a.unshift(1, 2)  #=> [ 1, 2, "a", "b", "c", "d"]
  def unshift(*several_variants) end

  # Returns an array containing the elements in
  # +self+ corresponding to the given selector(s). The selectors
  # may be either integer indices or ranges.
  # See also <code>Array#select</code>.
  #
  #    a = %w{ a b c d e f }
  #    a.values_at(1, 3, 5)
  #    a.values_at(1, 3, 5, 7)
  #    a.values_at(-1, -3, -5, -7)
  #    a.values_at(1..3, 2...5)
  def values_at(selector, *args) end

  # Converts any arguments to arrays, then merges elements of
  # +self+ with corresponding elements from each argument. This
  # generates a sequence of <code>self.size</code> <em>n</em>-element
  # arrays, where <em>n</em> is one more that the count of arguments. If
  # the size of any argument is less than <code>enumObj.size</code>,
  # <code>nil</code> values are supplied. If a block is given, it is
  # invoked for each output array, otherwise an array of arrays is
  # returned.
  #
  #    a = [ 4, 5, 6 ]
  #    b = [ 7, 8, 9 ]
  #    [1,2,3].zip(a, b)      #=> [[1, 4, 7], [2, 5, 8], [3, 6, 9]]
  #    [1,2].zip(a,b)         #=> [[1, 4, 7], [2, 5, 8]]
  #    a.zip([1,2],[8])       #=> [[4,1,8], [5,2,nil], [6,nil,nil]]
  def zip(arg, *args) end
end
