/*-------------------------------------------------|
| :: Barbatos SSR_T (Screen-Space Reflections)  :: |
'--------------------------------------------------|
| Version: 0.0.13                                  |
| Author: Barbatos                                 |
| License: MIT                                     |
| Description:Barbatos SSR, but focused for testing|
'--------------------------------------------------*/

#include "ReShade.fxh"
#include "ReShadeUI.fxh"

//--------------------|
// :: Preprocessor :: |
//--------------------|

#if __RESHADE__ < 40000
static const int STEPS_PER_RAY_WALLS_DX9 = 20;
static const int STEPS_PER_RAY_FLOOR_CEILING_QUALITY_DX9 = 64;
static const int STEPS_PER_RAY_FLOOR_CEILING_PERF_DX9 = 32;
#else
    static const int STEPS_PER_RAY_WALLS = 32;
#endif

#define fReflectFloorsIntensity 1
#define fReflectWallsIntensity 0
#define fReflectCeilingsIntensity 0

static const float2 LOD_MASK = float2(0.0, 1.0);
static const float2 ZERO_LOD = float2(0.0, 0.0);
#define PI 3.1415927
#define GetDepth(coords) (ReShade::GetLinearizedDepth(coords))
#define GetColor(c) tex2Dlod(ReShade::BackBuffer, float4((c).xy, 0, 0))
#define GetLod(s,c) tex2Dlod(s, ((c).xyyy * LOD_MASK.yyxx + ZERO_LOD.xxxy))

#define MVErrorTolerance 0.96

//----------|
// :: UI :: |
//----------|

// -- Main Settings --
uniform float SPIntensity <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.0; ui_max = 3.0; ui_step = 0.01;
    ui_category = "Main Settings";
    ui_label = "Reflection Intensity";
> = 1.1;

uniform float FadeEnd <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.0; ui_max = 5.0; ui_step = 0.010;
    ui_category = "Main Settings";
    ui_label = "Fade Distance";
    ui_tooltip = "Distance at which reflections begin to fade out.";
> = 4.999;

uniform float THICKNESS_THRESHOLD <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.001; ui_max = 0.02; ui_step = 0.001;
    ui_category = "Main Settings";
    ui_label = "Thickness Threshold";
    ui_tooltip = "Determines how thick a surface is before a ray passes through it. Prevents self-reflection artifacts.";
> = 0.01;

// -- Surface & Material --
uniform float Metallic <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.0; ui_max = 1.0; ui_step = 0.01;
    ui_category = "Surface & Material";
    ui_label = "Metallic";
    ui_tooltip = "Controls how metallic a surface is. 0.0 for dielectrics (plastic, wood), 1.0 for metals.";
> = 0.2;

uniform float Roughness <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.0; ui_max = 1.0; ui_step = 0.01;
    ui_category = "Surface & Material";
    ui_label = "Roughness";
    ui_tooltip = "Controls the surface roughness. 0.0 is a perfect mirror, 1.0 is very rough (diffuse).";
> = 0.25;

uniform float BumpIntensity <
    ui_label = "Bump Mapping Intensity";
    __UNIFORM_DRAG_FLOAT1
    ui_category = "Surface & Material";
    ui_min = 0.0; ui_max = 5.0; ui_step = 0.01;
> = 0.1;

uniform float SobelEdgeThreshold <
    ui_label = "Sobel Edge Threshold";
    ui_tooltip = "Sets a minimum edge strength for bump mapping to occur. Helps reduce noise on flat surfaces.";
    __UNIFORM_DRAG_FLOAT1
    ui_category = "Surface & Material";
    ui_min = 0.0; ui_max = 1.0; ui_step = 0.01;
> = 0.03;

// -- Performance & Quality --
uniform int Quality <
    __UNIFORM_COMBO_INT1
    ui_items = "Quality\0Performance\0";
    ui_category = "Performance & Quality";
    ui_label = "Quality Preset";
    ui_tooltip = "Choose between higher quality ray tracing or a faster preset.";
> = 0;

uniform float RenderScale <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.1; ui_max = 0.99; ui_step = 0.01;
    ui_category = "Performance & Quality";
    ui_label = "Render Scale";
    ui_tooltip = "Renders reflections at a lower resolution for performance";
> = 0.8;

// -- Glossy Reflections --
uniform bool EnableGlossy <
    ui_category = "Glossy Reflections";
    ui_label = "Enable Glossy Reflections";
    ui_tooltip = "Simulates reflections on non-perfectly smooth surfaces, creating a blurry/glossy effect.";
> = true;

uniform int GlossySamples <
    __UNIFORM_SLIDER_INT1
    ui_min = 1; ui_max = 20;
    ui_category = "Glossy Reflections";
    ui_label = "Glossy Samples";
    ui_tooltip = "Number of samples for the blur effect. Higher values are better quality but slower.";
> = 10;

// -- Temporal Filtering --
uniform bool EnableTAA <
    ui_category = "Temporal Filtering";
    ui_label = "Enable Temporal Reprojection";
    ui_tooltip = "Blends the current frame's reflection with previous frames to reduce noise and flickering using temporal reprojection.";
> = true;

uniform float FeedbackFactor <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.0; ui_max = 0.99; ui_step = 0.01;
    ui_category = "Temporal Filtering";
    ui_label = "Temporal Feedback";
    ui_tooltip = "Controls how much of the previous frame is blended in. Higher values are smoother but can cause more ghosting.";
> = 0.99;

// -- Advanced --
uniform float OrientationThreshold <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 0.01; ui_max = 1.0; ui_step = 0.01;
    ui_category = "Advanced";
    ui_label = "Orientation Threshold";
    ui_tooltip = "Controls sensitivity for detecting floors/walls/ceilings based on their normal vector. Lower is stricter.";
> = 0.90;

uniform float GeoCorrectionIntensity <
    __UNIFORM_DRAG_FLOAT1
    ui_min = -0.1; ui_max = 0.01;
    ui_step = 0.01;
    ui_category = "Advanced";
    ui_label = "Geometry Correction Intensity";
    ui_tooltip = "Subtly adjusts surface normals based on color data to correct minor geometry inaccuracies.";
> = -0.01;

uniform float VERTICAL_FOV <
    __UNIFORM_DRAG_FLOAT1
    ui_min = 15.0; ui_max = 120.0;
    ui_step = 0.1;
    ui_category = "Advanced";
    ui_label = "Vertical FOV";
> = 37.0;

// -- Debug --
uniform int ViewMode <
    __UNIFORM_COMBO_INT1
    ui_items = "None\0Motion Vectors\0Final Reflection\0Normals\0Depth\0Raw Low-Res Reflection\0Reflection Mask\0";
    ui_category = "Debug";
    ui_label = "Debug View Mode";
> = 0;

uniform int FRAME_COUNT < source = "framecount"; >;

//----------------|
// :: Textures :: |
//----------------|

#ifndef USE_MARTY_LAUNCHPAD_MOTION
#define USE_MARTY_LAUNCHPAD_MOTION 0
#endif

#ifndef USE_VORT_MOTION
#define USE_VORT_MOTION 0
#endif

#if USE_MARTY_LAUNCHPAD_MOTION
    namespace Deferred {
        texture MotionVectorsTex { Width = BUFFER_WIDTH; Height = BUFFER_HEIGHT; Format = RG16F; };
        sampler sMotionVectorsTex { Texture = MotionVectorsTex; };
    }
#elif USE_VORT_MOTION
    texture2D MotVectTexVort { Width = BUFFER_WIDTH; Height = BUFFER_HEIGHT; Format = RG16F; };
    sampler2D sMotVectTexVort { Texture = MotVectTexVort; MagFilter=POINT;MinFilter=POINT;MipFilter=POINT;AddressU=Clamp;AddressV=Clamp; };
#else
texture texMotionVectors
{
    Width = BUFFER_WIDTH;
    Height = BUFFER_HEIGHT;
    Format = RG16F;
};
sampler sTexMotionVectorsSampler
{
    Texture = texMotionVectors;
    MagFilter = POINT;
    MinFilter = POINT;
    MipFilter = POINT;
    AddressU = Clamp;
    AddressV = Clamp;
};
#endif

#if !USE_MARTY_LAUNCHPAD_MOTION && !USE_VORT_MOTION
float2 SampleMotionVectors(float2 texcoord)
{
    return GetLod(sTexMotionVectorsSampler, texcoord).rg;
}
#elif USE_MARTY_LAUNCHPAD_MOTION
    float2 SampleMotionVectors(float2 texcoord) {
        return GetLod(Deferred::sMotionVectorsTex, texcoord).rg;
    }
#elif USE_VORT_MOTION
    float2 SampleMotionVectors(float2 texcoord) {
        return GetLod(sMotVectTexVort, texcoord).rg;
    }
#endif

namespace Barbatos_SSR_TEST2
{
    // Stores the raw
    texture Reflection
    {
        Width = BUFFER_WIDTH;
        Height = BUFFER_HEIGHT;
        Format = RGBA16F;
    };
    sampler sReflection
    {
        Texture = Reflection;
    };

    // Intermediate
    texture Temp
    {
        Width = BUFFER_WIDTH;
        Height = BUFFER_HEIGHT;
        Format = RGBA16F;
    };
    sampler sTemp
    {
        Texture = Temp;
    };

    // Previous frame.
    texture History
    {
        Width = BUFFER_WIDTH;
        Height = BUFFER_HEIGHT;
        Format = RGBA16F;
    };
    sampler sHistory
    {
        Texture = History;
    };

    // Stores the final
    texture Upscaled
    {
        Width = BUFFER_WIDTH;
        Height = BUFFER_HEIGHT;
        Format = RGBA16F;
    };
    sampler sUpscaled
    {
        Texture = Upscaled;
    };

//---------------|
// :: Structs :: |
//---------------|

    struct Ray
    {
        float3 origin;
        float3 direction;
    };

    struct HitResult
    {
        bool found;
        float3 viewPos;
        float2 uv;
    };

//-------------|
// :: Utility::|
//-------------|

    static const float DIELECTRIC_REFLECTANCE = 0.04;

    float3 F_Schlick(float VdotH, float3 f0)
    {
        return f0 + (1.0.xxx - f0) * pow(1.0 - VdotH, 5.0);
    }
    
    float3 HSVToRGB(float3 c)
    {
        const float4 K = float4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
        float3 p = abs(frac(c.xxx + K.xyz) * 6.0 - K.www);
        return c.z * lerp(K.xxx, saturate(p - K.xxx), c.y);
    }

    float GetLuminance(float3 linearColor)
    {
        return dot(linearColor, float3(0.2126, 0.7152, 0.0722));
    }
    
    float2 GetMotionVectors(float2 texcoord)
    {
        float2 p = ReShade::PixelSize;
        float2 MV = SampleMotionVectors(texcoord);

        if (MVErrorTolerance < 1.0)
        {
            if (abs(MV.x) < p.x && abs(MV.y) < p.y)
                return float2(0.0, 0.0);
        }

        return MV;
    }
    
    // From http://psgraphics.blogspot.com/2011/01/improved-code-for-concentric-map.html
    float2 ConcentricSquareMapping(float2 u)
    {
        float a = 2.0 * u.x - 1.0;
        float b = 2.0 * u.y - 1.0;

        float r, phi;

        if ((a * a) > (b * b))
        {
            r = a;
            phi = (PI / 4.0) * (b / a);
        }
        else
        {
            r = b;
            if (b != 0.0)
            {
                phi = (PI / 2.0) - (PI / 4.0) * (a / b);
            }
            else
            {
                phi = 0.0;
            }
        }
        
        float X = r * cos(phi);
        float Y = r * sin(phi);

        return float2(X, Y);
    }
    
//------------------------------------|
// :: View Space & Normal Functions ::|
//------------------------------------|

    // Converts a screen-space UV coordinate and a view-space Z depth into a 3D view-space position.
    float3 UVToViewPos(float2 uv, float view_z)
    {
        float fov_rad = VERTICAL_FOV * (PI / 180.0);
        float proj_scale_y = 1.0 / tan(fov_rad * 0.5);
        float proj_scale_x = proj_scale_y / ReShade::AspectRatio;

        float2 clip_pos = uv * 2.0 - 1.0;

        float3 view_pos;
        view_pos.x = clip_pos.x / proj_scale_x * view_z;
        view_pos.y = -clip_pos.y / proj_scale_y * view_z;
        view_pos.z = view_z;

        return view_pos;
    }

    // Converts a 3D view-space position back into a 2D screen-space UV coordinate.
    float2 ViewPosToUV(float3 view_pos)
    {
        float fov_rad = VERTICAL_FOV * (PI / 180.0);
        float proj_scale_y = 1.0 / tan(fov_rad * 0.5);
        float proj_scale_x = proj_scale_y / ReShade::AspectRatio;

        float2 clip_pos;
        clip_pos.x = view_pos.x * proj_scale_x / view_pos.z;
        clip_pos.y = -view_pos.y * proj_scale_y / view_pos.z;

        return clip_pos * 0.5 + 0.5;
    }

    // Gets the view-space position from a UV coordinate.
    float3 GVPFUV(float2 uv)
    {
        float depth = GetDepth(uv);
        return UVToViewPos(uv, depth);
    }

    float3 Normal(float2 texcoord)
    {
        float3 offset_x = GVPFUV(texcoord + float2(ReShade::PixelSize.x, 0.0));
        float3 offset_y = GVPFUV(texcoord + float2(0.0, ReShade::PixelSize.y));
        float3 center = GVPFUV(texcoord);

        float3 dx = center - offset_x;
        float3 dy = center - offset_y;

        return normalize(cross(dx, dy));
    }
    
    float3 ApplyBumpMapping(float2 texcoord, float3 normal)
    {
        if (BumpIntensity == 0.0)
            return normal;

        float l00 = GetColor(texcoord + ReShade::PixelSize * float2(-1, -1)).g;
        float l10 = GetColor(texcoord + ReShade::PixelSize * float2( 0, -1)).g;
        float l20 = GetColor(texcoord + ReShade::PixelSize * float2( 1, -1)).g;
        float l01 = GetColor(texcoord + ReShade::PixelSize * float2(-1,  0)).g;
        float l21 = GetColor(texcoord + ReShade::PixelSize * float2( 1,  0)).g;
        float l02 = GetColor(texcoord + ReShade::PixelSize * float2(-1,  1)).g;
        float l12 = GetColor(texcoord + ReShade::PixelSize * float2( 0,  1)).g;
        float l22 = GetColor(texcoord + ReShade::PixelSize * float2( 1,  1)).g;

        float Gx = (l20 + 2.0 * l21 + l22) - (l00 + 2.0 * l01 + l02);
        float Gy = (l02 + 2.0 * l12 + l22) - (l00 + 2.0 * l10 + l20);

        if (length(float2(Gx, Gy)) < SobelEdgeThreshold)
            return normal;

        float2 slope = float2(Gx, Gy) * BumpIntensity;

        float3 up = abs(normal.y) < 0.99 ? float3(0, 1, 0) : float3(1, 0, 0);
        float3 T = normalize(cross(up, normal));
        float3 B = cross(normal, T);

        float3 bumpedNormal = normal + T * slope.x - B * slope.y;
        return normalize(bumpedNormal);
    }

    float3 GeometryCorrection(float2 texcoord, float3 normal)
    {
        if (GeoCorrectionIntensity == 0.0)
            return normal;

        float lumCenter = GetLuminance(GetColor(texcoord).rgb);
        float lumRight = GetLuminance(GetColor(texcoord + ReShade::PixelSize * int2(1, 0)).rgb);
        float lumDown = GetLuminance(GetColor(texcoord + ReShade::PixelSize * int2(0, 1)).rgb);

        float3 bumpNormal = normalize(float3(lumRight - lumCenter, lumDown - lumCenter, 1.0));

        return normalize(normal + bumpNormal * GeoCorrectionIntensity);
    }
//-------------------|
// :: Ray Tracing  ::|
//-------------------|

    // Traces a ray through the depth buffer to find an intersection.
    // Uses an adaptive step size that increases when far from surfaces and decreases when close.
    // A binary search (refinement steps) is used to find the precise intersection point.
    HitResult TraceRay(Ray r, int num_steps)
    {
        HitResult result;
        result.found = false;

        float step_scale, min_step_size, max_step_size;
        int refinement_steps;

        if (Quality == 1) // Performance
        {
            refinement_steps = 5;
            step_scale = 0.7;
            min_step_size = 0.001;
            max_step_size = 1.0;
        }
        else // Quality
        {
            refinement_steps = 5;
            step_scale = 0.1;
            min_step_size = 0.0001;
            max_step_size = 1.0;
        }

        float stepSize = min_step_size;
        float totalDist = 0.0;
        float3 prevPos = r.origin;

        for (int i = 0; i < num_steps; ++i)
        {
            float3 currPos = prevPos + r.direction * stepSize;
            totalDist += stepSize;

            float2 uvCurr = ViewPosToUV(currPos);
            if (any(uvCurr < 0.0) || any(uvCurr > 1.0) || totalDist > 10.0) // MaxTraceDistance
                break;

            float sceneDepth = GetDepth(uvCurr);
            float thickness = abs(currPos.z - sceneDepth);

            // If the ray is behind the scene geometry or hasn't hit anything substantial yet...
            if (currPos.z < sceneDepth || thickness > THICKNESS_THRESHOLD)
            {
                prevPos = currPos;
                // Adapt step size based on distance to the next surface.
                float distToDepth = abs(currPos.z - sceneDepth);
                stepSize = clamp(distToDepth * step_scale, min_step_size, max_step_size);
                continue;
            }

            // Intersection found, refine the hit position using binary search.
            float3 lo = prevPos, hi = currPos;
            for (int ref_step = 0; ref_step < refinement_steps; ++ref_step)
            {
                float3 mid = 0.5 * (lo + hi);
                float midDepth = GetDepth(ViewPosToUV(mid));
                if (mid.z >= midDepth)
                    hi = mid;
                else
                    lo = mid;
            }
            result.viewPos = hi;
            result.uv = ViewPosToUV(result.viewPos).xy;
            result.found = true;
            return result;
        }
        return result;
    }

//---------------|
// :: Glossy  :: |
//---------------|

    float specularPowerToConeAngle(float specularPower)
    {
        if (specularPower >= exp2(12.0))
            return 0.0f;
        const float xi = 0.244f;
        float exponent = 1.0f / (specularPower + 1.0f);
        return acos(pow(xi, exponent));
    }

    float isoscelesTriangleOpposite(float adjacentLength, float coneTheta)
    {
        return 2.0f * tan(coneTheta) * adjacentLength;
    }

    float isoscelesTriangleInRadius(float a, float h)
    {
        float a2 = a * a;
        float fh2 = 4.0f * h * h;
        return (a * (sqrt(a2 + fh2) - a)) / (4.0f * h);
    }
    
    float3 GetGlossySample(float2 sample_uv, float2 pixel_uv)
    {
        float3 reflectionColor;

        if (EnableGlossy && Roughness > 0.0)
        {
            float gloss = 1.0 - Roughness;
            float specularPower = pow(2.0, 10.0 * gloss + 1.0);
            float coneTheta = specularPowerToConeAngle(specularPower) * 0.5;

            if (coneTheta > 0.001)
            {
                float2 deltaP = (sample_uv - pixel_uv) * BUFFER_SCREEN_SIZE;
                float adjacentLength = length(deltaP);

                if (adjacentLength > 1.0)
                {
                    float oppositeLength = isoscelesTriangleOpposite(adjacentLength, coneTheta);
                    float incircleSize = isoscelesTriangleInRadius(oppositeLength, adjacentLength);
                    float blurRadiusUV = incircleSize * ReShade::PixelSize.x;

                    reflectionColor = 0.0.xxx;

                    for (int i = 0; i < GlossySamples; ++i)
                    {
                        float2 random_seed = pixel_uv * BUFFER_SCREEN_SIZE + float2(i, i * 2.0) + float2(float(FRAME_COUNT % 100), float(FRAME_COUNT % 50));
                        float2 u = frac(sin(float2(dot(random_seed, float2(12.9898, 78.233)), dot(random_seed, float2(39.345, 41.123)))) * 43758.5453);
                        float2 offset = ConcentricSquareMapping(u) * blurRadiusUV;
                        reflectionColor += GetColor(float4(sample_uv + offset, 0, 0)).rgb;
                    }
                    reflectionColor /= float(GlossySamples);
                }
                else
                {
                    reflectionColor = GetColor(float4(sample_uv, 0, 0)).rgb;
                }
            }
            else
            {
                reflectionColor = GetColor(float4(sample_uv, 0, 0)).rgb;
            }
        }
        else
        {
            reflectionColor = GetColor(float4(sample_uv, 0, 0)).rgb;
        }

        return reflectionColor;
    }
    
//------------|
// :: TAA  :: |
//------------|

    // Line-box clipping towards AABB center (PLAYDEAD's)
    float3 ClipToAABB(float3 aabb_min, float3 aabb_max, float3 history_sample, float3 current_sample)
    {
        float3 p_clip = 0.5 * (aabb_max + aabb_min);
        float3 e_clip = 0.5 * (aabb_max - aabb_min) + 1e-6; 
        float3 v_clip = history_sample - p_clip;
        float3 v_unit = v_clip / e_clip;
        float3 a_unit = abs(v_unit);
        float ma_unit = max(a_unit.x, max(a_unit.y, a_unit.z));
        
        if (ma_unit > 1.0)
            return p_clip + v_clip / ma_unit;
        else
            return history_sample; 
    }

    float2 GetVelocityFromClosestFragment(float2 texcoord)
    {
        float2 pixel_size = ReShade::PixelSize;
        float closest_depth = 1.0;
        float2 closest_velocity = float2(0, 0);
        
        // 3x3 neighborhood offsets
        const int2 offsets[9] =
        {
            int2(-1, -1), int2(0, -1), int2(1, -1),
            int2(-1, 0), int2(0, 0), int2(1, 0),
            int2(-1, 1), int2(0, 1), int2(1, 1)
        };
        
        [unroll]
        for (int i = 0; i < 9; i++)
        {
            float2 sample_coord = texcoord + offsets[i] * pixel_size;
            float sample_depth = GetDepth(sample_coord);
            
            if (sample_depth < closest_depth)
            {
                closest_depth = sample_depth;
                closest_velocity = SampleMotionVectors(sample_coord);
            }
        }
        
        return closest_velocity;
    }

    void ComputeNeighborhoodMinMax(sampler2D color_tex, float2 texcoord, out float3 color_min, out float3 color_max)
    {
        float2 pixel_size = ReShade::PixelSize / RenderScale;
        
        // Sample center
        float3 center_color = GetLod(color_tex, float4(texcoord, 0, 0)).rgb;
        color_min = center_color;
        color_max = center_color;
        
        // 3x3 neighborhood
        const int2 offsets_3x3[8] =
        {
            int2(-1, -1), int2(0, -1), int2(1, -1),
            int2(-1, 0), int2(1, 0),
            int2(-1, 1), int2(0, 1), int2(1, 1)
        };
        
        [unroll]
        for (int i = 0; i < 8; i++)
        {
            float2 sample_coord = texcoord + offsets_3x3[i] * pixel_size;
            float3 neighbor_color = GetLod(color_tex, float4(sample_coord, 0, 0)).rgb;
            color_min = min(color_min, neighbor_color);
            color_max = max(color_max, neighbor_color);
        }
        
        // Additional cross pattern (5-tap) 
        const int2 offsets_cross[4] =
        {
            int2(0, -2), int2(-2, 0), int2(2, 0), int2(0, 2)
        };
        
        float3 cross_min = center_color;
        float3 cross_max = center_color;
        
        [unroll]
        for (int j = 0; j < 4; j++)
        {
            float2 sample_coord = texcoord + offsets_cross[j] * pixel_size;
            float3 neighbor_color = GetLod(color_tex, float4(sample_coord, 0, 0)).rgb;
            cross_min = min(cross_min, neighbor_color);
            cross_max = max(cross_max, neighbor_color);
        }
        
        // Blend 3x3 with cross pattern
        color_min = lerp(cross_min, color_min, 0.5);
        color_max = lerp(cross_max, color_max, 0.5);
    }

    // Compute trust factor based on velocity magnitude
    float ComputeTrustFactor(float2 velocity_pixels, float low_threshold = 2.0, float high_threshold = 15.0)
    {
        float velocity_magnitude = length(velocity_pixels);
        return saturate((high_threshold - velocity_magnitude) / (high_threshold - low_threshold));
    }

//--------------------|
// :: Pixel Shaders ::|
//--------------------|

    void PS_TraceReflections(float4 pos : SV_Position, float2 uv : TEXCOORD, out float4 outReflection : SV_Target)
    {
        if (any(uv > RenderScale))
        {
            outReflection = 0;
            return;
        }

        float2 scaled_uv = uv / RenderScale;
        float depth = GetDepth(scaled_uv);
        if (depth >= 1.0)
        {
            outReflection = 0;
            return;
        }

        float3 viewPos = UVToViewPos(scaled_uv, depth);
        float3 viewDir = -normalize(viewPos);

        float3 normal = Normal(scaled_uv).xyz;
        normal = ApplyBumpMapping(scaled_uv, normal);
        normal = GeometryCorrection(scaled_uv, normal);
        normal = normalize(normal);

        bool isFloor = normal.y > OrientationThreshold;
        bool isCeiling = normal.y < -OrientationThreshold;
        bool isWall = abs(normal.y) <= OrientationThreshold;

        float orientationIntensity = (isFloor * fReflectFloorsIntensity) +
                                     (isWall * fReflectWallsIntensity) +
                                     (isCeiling * fReflectCeilingsIntensity);

        if (orientationIntensity <= 0.0)
        {
            outReflection = 0;
            return;
        }

        float3 eyeDir = -viewDir;
        Ray r;
        r.origin = viewPos;
        r.direction = normalize(reflect(eyeDir, normal));
        r.origin += r.direction * 0.0001; // Offset origin to avoid self-intersection

        HitResult hit;
        
#if __RENDERER__ == 0xd0900
        if (isWall)
            hit = TraceRay(r, STEPS_PER_RAY_WALLS_DX9);
        else
            hit = TraceRay(r, (Quality == 1) ? STEPS_PER_RAY_FLOOR_CEILING_PERF_DX9 : STEPS_PER_RAY_FLOOR_CEILING_QUALITY_DX9);
#else
        if (isWall)
            hit = TraceRay(r, STEPS_PER_RAY_WALLS);
        else
            hit = TraceRay(r, (Quality == 1) ? 128 : 256);
#endif

        float3 reflectionColor = 0;
        float reflectionAlpha = 0.0;

        if (hit.found)
        {
            reflectionColor = GetGlossySample(hit.uv, scaled_uv);

            float3 L = -r.direction;
            float3 V = eyeDir;
            float3 H = normalize(L + V);
            float VdotH = clamp(dot(V, H), 0.0, 1.0);

            float3 albedo = GetColor(float4(hit.uv, 0, 0)).rgb; // I'm testing simply using color for albedo
            float3 f0 = lerp(float3(DIELECTRIC_REFLECTANCE, DIELECTRIC_REFLECTANCE, DIELECTRIC_REFLECTANCE), albedo, Metallic);
            float3 F = F_Schlick(VdotH, f0);
            
            reflectionColor *= F;
            float distFactor = saturate(1.0 - length(hit.viewPos - viewPos) / 10.0); // MaxTraceDistance
            float fadeRange = max(FadeEnd, 0.001);
            float depthFade = saturate((FadeEnd - depth) / fadeRange);
            depthFade *= depthFade;

            reflectionAlpha = distFactor * depthFade;
        }
        else // Fallback for missed rays
        {
            float adaptiveDist = min(depth * 1.2 + 0.012, 10.0);
            float3 fbViewPos = viewPos + r.direction * adaptiveDist;
            float2 uvFb = saturate(ViewPosToUV(fbViewPos).xy);
            reflectionColor = GetGlossySample(uvFb, scaled_uv);
            float fresnel = pow(1.0 - saturate(dot(eyeDir, normal)), 3.0);
            float vertical_fade = pow(saturate(1.0 - scaled_uv.y), 3.0);
            reflectionAlpha = fresnel * vertical_fade;
        }

        float angleWeight = pow(saturate(dot(-viewDir, r.direction)), 2.0);
        reflectionAlpha *= angleWeight;
        reflectionAlpha *= orientationIntensity;

        outReflection = float4(reflectionColor, reflectionAlpha);
    }
    
    void PS_Accumulate(float4 pos : SV_Position, float2 uv : TEXCOORD, out float4 outBlended : SV_Target)
    {
        if (any(uv > RenderScale))
        {
            outBlended = 0;
            return;
        }

        float4 current_reflection = GetLod(sReflection, float4(uv, 0, 0));

        if (!EnableTAA)
        {
            outBlended = current_reflection;
            return;
        }
        
        float2 full_res_uv = uv / RenderScale;
        float2 velocity = GetVelocityFromClosestFragment(full_res_uv);
        float current_depth = GetDepth(full_res_uv);
        
        float2 reprojected_uv_full = full_res_uv + velocity;
        float history_depth = GetDepth(reprojected_uv_full);
        
        float2 reprojected_uv_low = reprojected_uv_full * RenderScale;

        bool valid_history = all(saturate(reprojected_uv_low) == reprojected_uv_low) &&
                             FRAME_COUNT > 1 &&
                             abs(history_depth - current_depth) < 0.01;

        if (!valid_history)
        {
            outBlended = current_reflection;
            return;
        }
        
        float4 history_reflection = GetLod(sHistory, float4(reprojected_uv_low, 0, 0));
        
        float3 color_min, color_max;
        ComputeNeighborhoodMinMax(sReflection, uv, color_min, color_max);
        
        float3 clipped_history_rgb = ClipToAABB(color_min, color_max, history_reflection.rgb, current_reflection.rgb);
        
        float clipping_distance = length(history_reflection.rgb - clipped_history_rgb);
        float rejection_factor = saturate(clipping_distance * 9000.0); // Sensitivity
        
        float final_feedback = lerp(FeedbackFactor, 0.1, rejection_factor);
        
        float3 temporal_rgb = lerp(current_reflection.rgb, clipped_history_rgb, final_feedback);
        float temporal_a = lerp(current_reflection.a, history_reflection.a, final_feedback);
        
        float trust_factor = ComputeTrustFactor(velocity * BUFFER_SCREEN_SIZE);
        if (trust_factor < 1.0)
        {
            float3 blurred_color = current_reflection.rgb;
            const int blur_samples = 5;
            [unroll]
            for (int i = 1; i < blur_samples; i++)
            {
                float t = (float) i / (float) (blur_samples - 1);
                float2 blur_coord_low = uv - (velocity * RenderScale) * 0.5 * t; // Blur in low-res space
                if (all(saturate(blur_coord_low) == blur_coord_low))
                {
                    blurred_color += GetLod(sReflection, float4(blur_coord_low, 0, 0)).rgb;
                }
            }
            blurred_color /= (float) blur_samples;
            temporal_rgb = lerp(blurred_color, temporal_rgb, trust_factor);
        }

        outBlended = float4(temporal_rgb, temporal_a);
    }

    void PS_UpdateHistory(float4 pos : SV_Position, float2 uv : TEXCOORD, out float4 outHistory : SV_Target)
    {
        outHistory = GetLod(sTemp, float4(uv, 0, 0));
    }

    void PS_Upscale(float4 pos : SV_Position, float2 uv : TEXCOORD, out float4 outUpscaled : SV_Target)
    {
        if (RenderScale >= 1.0)
        {
            outUpscaled = GetLod(sTemp, uv);
            return;
        }
        float2 scaled_uv = uv * RenderScale;
        outUpscaled = GetLod(sTemp, scaled_uv);
    }

    void PS_Output(float4 pos : SV_Position, float2 uv : TEXCOORD, out float4 outColor : SV_Target)
    {
        // Debug
        if (ViewMode != 0)
        {
            switch (ViewMode)
            {
                case 1: // Motion vectors
                {
                        float2 motion = GetMotionVectors(uv);
                        float velocity = length(motion) * 100.0;
                        float angle = atan2(motion.y, motion.x);
                        float3 hsv = float3((angle / (2.0 * PI)) + 0.5, 1.0, saturate(velocity));
                        outColor = float4(HSVToRGB(hsv), 1.0);
                        return;
                    }
                case 2: // Final Reflection (Upscaled)
                    outColor = float4(tex2Dlod(sUpscaled, float4(uv, 0, 0)).rgb, 1.0);
                    return;
                case 3: // Normals
                {
                        float3 normal = Normal(uv);
                        normal = ApplyBumpMapping(uv, normal);
                        normal = GeometryCorrection(uv, normal);
                        outColor = float4(normalize(normal) * 0.5 + 0.5, 1.0);
                        return;
                    }
                case 4: // Depth
                    outColor = GetDepth(uv).xxxx;
                    return;
                case 5: // Raw Low-Res Reflection
                    outColor = float4(GetLod(sReflection, float4(uv, 0, 0)).rgb, 1.0);
                    return;
                case 6: // Reflection Mask
                    outColor = GetLod(sUpscaled, float4(uv, 0, 0)).aaaa;
                    return;
            }
        }

        float3 originalColor = GetColor(uv).rgb;

        if (GetDepth(uv) >= 1.0)
        {
            outColor = float4(originalColor, 1.0);
            return;
        }

        float4 reflectionSample = GetLod(sUpscaled, float4(uv, 0, 0));
        float3 reflectionColor = reflectionSample.rgb;
        float reflectionMask = reflectionSample.a;

        reflectionColor *= SPIntensity;
        float reflectionLuminance = GetLuminance(reflectionColor);
        float conservation = saturate(reflectionLuminance * reflectionMask);

        float3 finalColor = originalColor * (1.0 - conservation) + (reflectionColor * reflectionMask);

        outColor = float4(finalColor, 1.0);
    }

//-----------------|
// :: Technique :: |
//-----------------|

    technique Barbatos_SSR_TEST
    {
        pass TraceReflections
        {
            VertexShader = PostProcessVS;
            PixelShader = PS_TraceReflections;
            RenderTarget = Reflection;
            ClearRenderTargets = true;
        }
        pass Accumulate
        {
            VertexShader = PostProcessVS;
            PixelShader = PS_Accumulate;
            RenderTarget = Temp;
        }
        pass UpdateHistory
        {
            VertexShader = PostProcessVS;
            PixelShader = PS_UpdateHistory;
            RenderTarget = History;
        }
        pass Upscale
        {
            VertexShader = PostProcessVS;
            PixelShader = PS_Upscale;
            RenderTarget = Upscaled;
        }
        pass Output
        {
            VertexShader = PostProcessVS;
            PixelShader = PS_Output;
        }
    }
}
