/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::zoneGenerators::Union

Description
    A zoneGenerator which selects all the elements from the zones generated by
    the given list of zoneGenerators

    A union of faceZones is oriented only if ALL the faceZones combined are
    oriented, otherwise the returned faceZone is unoriented.

Usage
    \table
        Property     | Description             | Required | Default value
        type         | Type: union             | yes      |
        name         | Name of the zone        | no       | zoneGenerator name
        zoneType     | Type of zone            | no       | all the zone types
        moveUpdate   | Switch to update after mesh motion | no | false
    \endtable
    These options are followed by a list of any number of zoneGenerators
    which can either generate the zones locally or return zones from the lists
    of zones held by the mesh using the zoneGenerators::lookup zoneGenerator.

    Each zoneGenerator returns a zoneSet containing a pointZone and/or a
    cellZone and/or a faceZone, the zoneGenerators::Union operation combines
    each of the zone types unless the \c zoneType is specified in which case
    only the zones of that type are combined.

    A cellZone named \c coneAndAnnulus containing the cells with centres
    inside the union of the predefined \c cone1 cellZone which is looked-up
    from the cellZoneList and the locally defined annulus is be generated by
    \verbatim
        coneAndAnnulus
        {
            type        union;

            cone1;

            annulus1
            {
                type        annulus;
                zoneType    cell;

                point1      (-0.0075 0 -1);
                point2      (-0.003 0.0025 1);
                outerRadius 0.002;
                innerRadius 0.001;
            }
        }
    \endverbatim

See also
    zoneGenerators::difference
    zoneGenerators::intersection
    zoneGeneratorList

SourceFiles
    Union.C

\*---------------------------------------------------------------------------*/

#ifndef union_zoneGenerator_H
#define union_zoneGenerator_H

#include "zoneGeneratorList.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{
namespace zoneGenerators
{

/*---------------------------------------------------------------------------*\
                             Class Union Declaration
\*---------------------------------------------------------------------------*/

class Union
:
    public zoneGenerator
{
    // Private Data

        //- The type of zone to lookup (optional)
        zoneTypesAll zoneType_;

        zoneGeneratorList zoneGenerators_;


    // Private Member Functions

        //- Add the given list of indices to the selected list
        void select
        (
            boolList& selected,
            const labelList& indices,
            const bool select = true
        ) const;

        //- Add the given list of face indices and corresponding flipMap
        //  to the selected list and flipMap
        void select
        (
            boolList& selected,
            boolList& flipMap,
            const labelList& indices,
            const boolList& zoneFlipMap,
            const bool select = true
        ) const;


protected:

    // Protected Member Functions

        //- Generate the zoneSet for union or difference
        //  optionally initialising all the points and/or cells and/or faces
        zoneSet generate(const bool diff, const bool all) const;


public:

    //- Runtime type information
    TypeName("union");


    // Constructors

        //- Construct from dictionary
        Union
        (
            const word& name,
            const polyMesh& mesh,
            const dictionary& dict
        );


    //- Destructor
    virtual ~Union();


    // Member Functions

        virtual zoneSet generate() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace zoneGenerators
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
