#!/bin/bash
######################################################################
#                                                                    #
#  Copyright (c) 2001, 2025 NoMachine, http://www.nomachine.com.     #
#                                                                    #
#  All rights reserved.                                              #
#                                                                    #
######################################################################

function getIp()
{
  ROUTE_NUMBER=$1

  #
  # Condition used when we want consider only default routes.
  #

  ROUTE_CONDITION="$2"

  if [ "$2" == "default" ] && [ "${MODE}" == "2" ]; then
    ROUTE_CONDITION="^0.0.0.0$"
  fi

  NETWORK_ADDRESS=""
  NETWORK_NETMASK=""
  DEFAULT_GATEWAY=""
  INTERFACE=""
  FRIENDLY_NAME=""

  #
  # Get row number in routing table ordered by metric.
  #
  # Find interface of that route.
  #

  case "${MODE}" in
    "1")
      ORDER_NUMBER="$( ${COMMAND_IP} route | \
                        ${COMMAND_AWK} -v var="${ROUTE_CONDITION}" -F"metric " '$0~var {print NR,$2}' | \
                            ${COMMAND_SORT} -n -k2 | \
                                ${COMMAND_AWK} -F" " -v var=${ROUTE_NUMBER} 'NR==var {print $1}' )"

      INTERFACE="$( ${COMMAND_IP} route | \
                     ${COMMAND_AWK} -v var=${ORDER_NUMBER} -F"dev " 'NR==var {print $2}' | \
                         ${COMMAND_AWK} -F" " '{print $1}' )"
      ;;
    "2")
      ORDER_NUMBER="$( ${COMMAND_ROUTE} -n | \
                        ${COMMAND_AWK} -v var1=${ROW_NUMBER} -v var2="${ROUTE_CONDITION}" '(NR>var1) && ($1~var2) {print NR,$5}' | \
                            ${COMMAND_SORT} -n -k2 | \
                                ${COMMAND_AWK} -F" " -v var=${ROUTE_NUMBER} 'NR==var {print $1}' )"

      INTERFACE="$( ${COMMAND_ROUTE} -n | ${COMMAND_AWK} '{print $8}' | \
                     ${COMMAND_AWK} -v var=${ORDER_NUMBER} 'NR==var {print $1}' )"
      ;;
  esac

  #
  # Friendly name of interface.
  #

  FRIENDLY_NAME="$INTERFACE"

  #
  # Read IP address and netmask of read interface.
  #

  case "${MODE}" in
    "1")
      FULL_ADDRESS="$( ${COMMAND_IP} address show dev ${INTERFACE} | \
                        ${COMMAND_AWK} -F"inet " '{print $2}' | \
                            ${COMMAND_AWK} -F" " '{print $1}' )"

      IP_ADDRESS="$( ${COMMAND_ECHO} ${FULL_ADDRESS} | \
                      ${COMMAND_AWK} -F"/" '{print $1}' )"

      NETWORK_NUMBER="$( ${COMMAND_ECHO} ${FULL_ADDRESS} | \
                           ${COMMAND_AWK} -F" " '{print $1}' | \
                               ${COMMAND_AWK} -F"/" '{print $2}' )"

      if test "$NETWORK_NUMBER" == ""; then
        NETWORK_NUMBER="32"
      fi

      for VAR in 1 2 3 4; do

        case "${NETWORK_NUMBER}" in
          "0")
            NETWORK_NETMASK="${NETWORK_NETMASK}0";;
          "1")
            NETWORK_NETMASK="${NETWORK_NETMASK}128";;
          "2")
            NETWORK_NETMASK="${NETWORK_NETMASK}192";;
          "3")
            NETWORK_NETMASK="${NETWORK_NETMASK}224";;
          "4")
            NETWORK_NETMASK="${NETWORK_NETMASK}240";;
          "5")
            NETWORK_NETMASK="${NETWORK_NETMASK}248";;
          "6")
            NETWORK_NETMASK="${NETWORK_NETMASK}252";;
          "7")
            NETWORK_NETMASK="${NETWORK_NETMASK}254";;
          *)
            NETWORK_NETMASK="${NETWORK_NETMASK}255";;
        esac

        if (( ${NETWORK_NUMBER} >= 8 )); then
          NETWORK_NUMBER=$((NETWORK_NUMBER - 8))
        else
          NETWORK_NUMBER="0"
        fi

        if test "$VAR" != "4"; then
          NETWORK_NETMASK=$NETWORK_NETMASK.
        fi

      done

      DEFAULT_GATEWAY="$( ${COMMAND_IP} route | \
                           ${COMMAND_GREP} "default.*${INTERFACE}" | \
                               ${COMMAND_AWK} -F" " '{print $3}' )"
      ;;
    "2")
      #
      # Read netmask from COMMAND_IFCONFIG output.
      #
      # Depending on Linux distribution netmask
      # is marked as "Mask:" or "netmask ".
      #

      case "$( ${COMMAND_IFCONFIG} )" in

        #
        # Some systems acts werid when trying
        # to use bash variables inside awk.
        #
        # Hardcoded command will do well.
        #

        *"Mask:"*)
          NETWORK_NETMASK="$( ${COMMAND_IFCONFIG} $INTERFACE | \
                               ${COMMAND_AWK} -F":" '/Mask:/{print $4}' | \
                                   ${COMMAND_AWK} 'NR==1 {print $1}' )"

          IP_ADDRESS="$( ${COMMAND_IFCONFIG} $INTERFACE | \
                          ${COMMAND_AWK} -F":" '/Mask:/{print $2}' | \
                              ${COMMAND_AWK} 'NR==1 {print $1}' )";;

        *"netmask "*)
          NETWORK_NETMASK="$( ${COMMAND_IFCONFIG} $INTERFACE | \
                               ${COMMAND_AWK} -F" " '/netmask /{print $4}' | \
                                   ${COMMAND_AWK} 'NR==1 {print $1}' )"

          IP_ADDRESS="$( ${COMMAND_IFCONFIG} $INTERFACE | \
                          ${COMMAND_AWK} -F" " '/netmask /{print $2}' | \
                              ${COMMAND_AWK} 'NR==1 {print $1}' )";;

        *)
          ${COMMAND_ECHO} "NX> 500 Error: Cannot find netmask keyword"

          exit 1;;
      esac

      DEFAULT_GATEWAY="$( ${COMMAND_ROUTE} -n | \
                           ${COMMAND_GREP} "0\.0\.0\.0.*0\.0\.0\.0.*${INTERFACE}" | \
                               ${COMMAND_AWK} 'NR==1 {print $2}' )"
      ;;
  esac

  if test "x$NETWORK_NETMASK" == "x"; then

    ${COMMAND_ECHO} "NX> 500 Error: Cannot find netmask"

    exit 1

  fi

  if test "x$IP_ADDRESS" == "x"; then

    ${COMMAND_ECHO} "NX> 500 Error: Cannot find ip address"

    exit 1

  fi

  #
  # Calculate network address using
  # interface IP and its subnet mask.
  #

  for VAR in 1 2 3 4; do

    TEMP="$(${COMMAND_ECHO} ${NETWORK_NETMASK} | \
              ${COMMAND_AWK} -F'.' '{print $'$VAR'}' | \
                  ${COMMAND_AWK} '{print $1}')"

    TEMP2="$(${COMMAND_ECHO} ${IP_ADDRESS} | \
               ${COMMAND_AWK} -F'.' '{print $'$VAR'}' | \
                   ${COMMAND_AWK} '{print $1}')"

    TEMP3=$((${TEMP2} & ${TEMP}))

    NETWORK_ADDRESS=${NETWORK_ADDRESS}${TEMP3}

    #
    # Add a dot after every octet.
    #

    if test "${VAR}" != "4"; then
      NETWORK_ADDRESS=${NETWORK_ADDRESS}.
    fi

  done
}

PARAMS_COUNT_MIN=0
PARAMS_COUNT_MAX=1

CONNECTION_TYPE=""

if test -x "/bin/echo"; then
  COMMAND_ECHO="/bin/echo"
else
  echo "NX> 500 Error: echo command not found."
  exit 1
fi

if test -x "/usr/bin/nawk"; then
  COMMAND_AWK="/usr/bin/nawk"
elif test -x "/usr/xpg4/bin/awk"; then
  COMMAND_AWK="/usr/xpg4/bin/awk"
elif test -x "/bin/awk"; then
  COMMAND_AWK="/bin/awk"
elif test -x "/usr/bin/awk"; then
  COMMAND_AWK="/usr/bin/awk"
else
  ${COMMAND_ECHO} "NX> 500 Error: awk command not found."
  exit 1
fi

if test -x "/bin/grep"; then
  COMMAND_GREP="/bin/grep"
elif test -x "/usr/bin/grep"; then
  COMMAND_GREP="/usr/bin/grep"
else
  ${COMMAND_ECHO} "NX> 500 Error: grep command not found."
  exit 1
fi

if test -x "/bin/sort"; then
  COMMAND_SORT="/bin/sort"
elif test -x "/usr/bin/sort"; then
  COMMAND_SORT="/usr/bin/sort"
else
  ${COMMAND_ECHO} "NX> 500 Error: sort command not found."
fi

#
# For correct executing this script you need either
# an 'ip' command or 'route' and 'ifconfig' commands.
#
# MODE defines which commands will be used.
#

if test -x "/sbin/ip"; then
  MODE=1

  COMMAND_IP="/sbin/ip"
else
  MODE=2

  if test -x "/bin/route"; then
    COMMAND_ROUTE="/bin/route"
  elif test -x "/sbin/route"; then
    COMMAND_ROUTE="/sbin/route"
  else
    ${COMMAND_ECHO} "NX> 500 Error: ip or route command not found."
    exit 1
  fi

  if test -x "/bin/ifconfig"; then
    COMMAND_IFCONFIG="/bin/ifconfig"
  elif test -x "/sbin/ifconfig"; then
    COMMAND_IFCONFIG="/sbin/ifconfig"
  else
    ${COMMAND_ECHO} "NX> 500 Error: ip or ifconfig command not found"
    exit 1
  fi
fi

if test "x$1" != "x"; then
  CONNECTION_TYPE=$1
fi

#
# Count all rows in routing table.
#

case "${MODE}" in
  "1")
    if [ "${CONNECTION_TYPE}" == "default" ]; then

      TOTAL_ROWS="$( ${COMMAND_IP} route | ${COMMAND_GREP} -c -E \
                      "^(default)[[:space:]]" )"

    else

      TOTAL_ROWS="$( ${COMMAND_IP} route | ${COMMAND_GREP} "" -c )"

    fi
    ;;

  "2")
    if [ "${CONNECTION_TYPE}" == "default" ]; then

      TOTAL_ROWS="$( ${COMMAND_ROUTE} -n | ${COMMAND_GREP} -c -E \
                      "^(\0\.\0\.\0\.\0)[[:space:]]" )"

      ROW_NUMBER="0"

    else

      TOTAL_ROWS="$( ${COMMAND_ROUTE} -n | ${COMMAND_GREP} "" -c )"

      ROW_NUMBER="$( ${COMMAND_ROUTE} -n | ${COMMAND_AWK} '(/Destination/ || \
                      /Gateway/ || /Genmask/ || /Flags/ || /Metric/ || \
                          /Ref/ || /Use/ || /Iface/) {print NR}' )"

      TOTAL_ROWS=$((TOTAL_ROWS - ROW_NUMBER))

    fi
    ;;
esac

#
# Exit if there are no rows in the table.
#

if (( ${TOTAL_ROWS} == 0 )); then
  ${COMMAND_ECHO} "||||"

  exit 1
fi

#
# Get entries row by row ascending by metric.
#
# Save all uniqe networks.
#

for i in $(seq 1 ${TOTAL_ROWS}); do
  getIp ${i} ${CONNECTION_TYPE}

  ENTRY="$NETWORK_ADDRESS|$NETWORK_NETMASK|$DEFAULT_GATEWAY|$INTERFACE|$FRIENDLY_NAME"

  ADD_ROUTE=1

  for j in ${array[*]}; do
    if [ "$j" == "${ENTRY}" ]; then
      ADD_ROUTE=0
      break;
    fi
  done

  if [ "${ADD_ROUTE}" == "1" ]; then
    #
    # Add new uniqe network.
    #

    array[${#array[*]}]=${ENTRY}
  fi
done

#
# Get all routes in one variable.
#

for i in ${!array[*]}; do
  ROUTES="${ROUTES}${array[$i]}"

  COUNT=$(( ${#array[*]} - 1 ))

  if [ "${i}" != "${COUNT}" ]; then
    ROUTES="${ROUTES}|"
  fi
done

${COMMAND_ECHO} "${ROUTES}"
