"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerDiagnosticsPushSupport = registerDiagnosticsPushSupport;
exports.registerDiagnosticsPullSupport = registerDiagnosticsPullSupport;
const vscode_languageserver_1 = require("vscode-languageserver");
const runner_1 = require("./runner");
function registerDiagnosticsPushSupport(documents, connection, runtime, validate) {
    const pendingValidationRequests = {};
    const validationDelayMs = 500;
    const disposables = [];
    documents.onDidChangeContent(change => {
        triggerValidation(change.document);
    }, undefined, disposables);
    documents.onDidClose(event => {
        cleanPendingValidation(event.document);
        connection.sendDiagnostics({ uri: event.document.uri, diagnostics: [] });
    }, undefined, disposables);
    function cleanPendingValidation(textDocument) {
        const request = pendingValidationRequests[textDocument.uri];
        if (request) {
            request.dispose();
            delete pendingValidationRequests[textDocument.uri];
        }
    }
    function triggerValidation(textDocument) {
        cleanPendingValidation(textDocument);
        const request = pendingValidationRequests[textDocument.uri] = runtime.timer.setTimeout(() => __awaiter(this, void 0, void 0, function* () {
            if (request === pendingValidationRequests[textDocument.uri]) {
                try {
                    const diagnostics = yield validate(textDocument);
                    if (request === pendingValidationRequests[textDocument.uri]) {
                        connection.sendDiagnostics({ uri: textDocument.uri, diagnostics });
                    }
                    delete pendingValidationRequests[textDocument.uri];
                }
                catch (e) {
                    connection.console.error((0, runner_1.formatError)(`Error while validating ${textDocument.uri}`, e));
                }
            }
        }), validationDelayMs);
    }
    return {
        requestRefresh: () => {
            documents.all().forEach(triggerValidation);
        },
        dispose: () => {
            disposables.forEach(d => d.dispose());
            disposables.length = 0;
            const keys = Object.keys(pendingValidationRequests);
            for (const key of keys) {
                pendingValidationRequests[key].dispose();
                delete pendingValidationRequests[key];
            }
        }
    };
}
function registerDiagnosticsPullSupport(documents, connection, runtime, validate) {
    function newDocumentDiagnosticReport(diagnostics) {
        return {
            kind: vscode_languageserver_1.DocumentDiagnosticReportKind.Full,
            items: diagnostics
        };
    }
    const registration = connection.languages.diagnostics.on((params, token) => __awaiter(this, void 0, void 0, function* () {
        return (0, runner_1.runSafe)(runtime, () => __awaiter(this, void 0, void 0, function* () {
            const document = documents.get(params.textDocument.uri);
            if (document) {
                return newDocumentDiagnosticReport(yield validate(document));
            }
            return newDocumentDiagnosticReport([]);
        }), newDocumentDiagnosticReport([]), `Error while computing diagnostics for ${params.textDocument.uri}`, token);
    }));
    function requestRefresh() {
        connection.languages.diagnostics.refresh();
    }
    return {
        requestRefresh,
        dispose: () => {
            registration.dispose();
        }
    };
}
