<?php

namespace Relay;

/**
 * Relay client.
 */
class Relay
{
    /**
     * Relay's version.
     *
     * @var string
     */
    public const VERSION = "0.12.0";

    /**
     * Relay's version.
     *
     * @var string
     */
    public const Version = "0.12.0";

    /**
     * Integer representing no compression algorithm.
     *
     * @var int
     */
    public const COMPRESSION_NONE = 0;

    /**
     * Integer representing the LZF compression algorithm.
     *
     * @var int
     */
    public const COMPRESSION_LZF = 1;

    /**
     * Integer representing the Zstandard compression algorithm.
     *
     * @var int
     */
    public const COMPRESSION_ZSTD = 2;

    /**
     * Integer representing the LZ4 compression algorithm.
     *
     * @var int
     */
    public const COMPRESSION_LZ4 = 3;

    /**
     * Integer representing no serializer.
     *
     * @var int
     */
    public const SERIALIZER_NONE = 0;

    /**
     * Integer representing the PHP serializer.
     *
     * @var int
     */
    public const SERIALIZER_PHP = 1;

    /**
     * Integer representing the igbinary serializer.
     *
     * @var int
     */
    public const SERIALIZER_IGBINARY = 2;

    /**
     * Integer representing the MessagePack serializer.
     *
     * @var int
     */
    public const SERIALIZER_MSGPACK = 3;

    /**
     * Integer representing the JSON serializer.
     *
     * @var int
     */
    public const SERIALIZER_JSON = 4;

    /**
     * Integer representing the atomic mode.
     *
     * @see Relay::getMode()
     * @var int
     */
    public const ATOMIC = 0x00;

    /**
     * Integer representing the pipeline mode.
     *
     * @see Relay::getMode()
     * @var int
     */
    public const PIPELINE = 0x02;

    /**
     * Integer representing the `MULTI` mode.
     *
     * @see Relay::getMode()
     * @var int
     */
    public const MULTI = 0x01;

    /**
     * Integer representing we're SUBSCRIBED. Note that this constant can
     * only really be accessed when `true` is passed to `getMask()` telling
     * relay to return the complete bitmasked mode.
     *
     * @see Relay::getMode()
     * @var int
     **/
    public const SUBSCRIBED = 0x04;

    /**
     * Integer representing the prefix option.
     *
     * @var int
     */
    public const OPT_PREFIX = 2;

    /**
     * Integer representing the read timeout option.
     *
     * @var int
     */
    public const OPT_READ_TIMEOUT = 3;

    /**
     * Integer representing the maximum retries option.
     *
     * @var int
     */
    public const OPT_MAX_RETRIES = 11;

    /**
     * Integer representing the backoff algorithm.
     *
     * @var int
     */
    public const OPT_BACKOFF_ALGORITHM = 12;

    /**
     * Toggle TCP_KEEPALIVE on a connection.
     *
     * @var int
     */
    public const OPT_TCP_KEEPALIVE = 6;

    /**
     * Integer representing the default backoff algorithm.
     *
     * @var int
     */
    public const BACKOFF_ALGORITHM_DEFAULT = 0;

    /**
     * Integer representing the decorrelated jitter backoff algorithm.
     *
     * @var int
     */
    public const BACKOFF_ALGORITHM_DECORRELATED_JITTER = 1;

    /**
     * Integer representing the full jitter backoff algorithm.
     *
     * @var int
     */
    public const BACKOFF_ALGORITHM_FULL_JITTER = 2;

    /**
     * Integer representing the base for backoff computation.
     *
     * @var int
     */
    public const OPT_BACKOFF_BASE = 13;

    /**
     * Integer representing the backoff time cap.
     *
     * @var int
     */
    public const OPT_BACKOFF_CAP = 14;

    /**
     * Integer representing the PhpRedis compatibility mode option.
     *
     * Enabled by default. Disabling will cause Relay to:
     * 1. Return `null` when a key doesn't exist, instead of `false`
     * 2. Throw exceptions when an error occurs, instead of returning `false`
     *
     * @var int
     */
    public const OPT_PHPREDIS_COMPATIBILITY = 100;

    /**
     * Integer representing the serializer option.
     *
     * @var int
     */
    public const OPT_SERIALIZER = 1;

    /**
     * Integer representing the compression option.
     *
     * @var int
     */
    public const OPT_COMPRESSION = 7;

    /**
     * Integer representing the compression level option.
     *
     * @var int
     */
    public const OPT_COMPRESSION_LEVEL = 9;

    /**
     * Integer representing the reply literal option.
     *
     * @var int
     */
    public const OPT_REPLY_LITERAL = 8;

    /**
     * Integer representing the null-multi-bulk-as-null option.
     *
     * @var int
     */
    public const OPT_NULL_MULTIBULK_AS_NULL = 10;

    /**
     * @var int
     *
     * When enabled, this option tells Relay to ignore purely numeric values
     * when packing and unpacking data. This does not include numeric strings.
     * If you want numeric strings to be ignored, typecast them to an int or float.
     *
     * The primary purpose of this option is to make it more ergonomic when
     * setting keys that will later be incremented or decremented.
     *
     * Note: This option incurs a small performance penalty when reading data
     * because we have to see if the data is a string representation of an int
     * or float.
     *
     * @example
     * <pre>
     * <code>
     * $redis->setOption(Relay::OPT_SERIALIZER, Relay::SERIALIZER_IGBINARY);
     * $redis->setOption(Relay::OPT_PACK_IGNORE_NUMBERS, true);
     *
     * $redis->set('answer', 32);
     *
     * var_dump($redis->incrBy('answer', 10));  // int(42)
     * var_dump($redis->get('answer'));         // int(42)
     * </code>
     * </pre>
     */
    public const OPT_PACK_IGNORE_NUMBERS = 15;

    /**
     * Integer representing the throw-on-error option.
     *
     * Disabled by default. When enabled, Relay will throw exceptions when errors occur.
     *
     * @var int
     */
    public const OPT_THROW_ON_ERROR = 106;

    /**
     * Integer representing Relay’s invalidation option.
     *
     * Enabled by default. When disabled will prevent Relay from
     * performing instantaneous client-side invalidation when a key
     * is changed without waiting for Redis to send an `INVALIDATE`
     * message. The invalidation occurs only in the same FPM pool.
     *
     * @var int
     */
    public const OPT_CLIENT_INVALIDATIONS = 101;

    /**
     * Integer representing Relay’s allow patterns option.
     *
     * When set only keys matching these patterns will be cached,
     * unless they also match an `OPT_IGNORE_PATTERNS`.
     *
     * @var int
     */
    public const OPT_ALLOW_PATTERNS = 102;

    /**
     * Integer representing Relay’s ignore patterns option.
     *
     * When set keys matching these patterns will not be cached.
     *
     * @var int
     */
    public const OPT_IGNORE_PATTERNS = 103;

    /**
     * Whether use in-memory caching. Enabled by default.
     *
     * @var int
     */
    public const OPT_USE_CACHE = 104;

    /**
     * Whether to enable client tracking for the connection.
     *
     * @var int
     */
    public const OPT_CLIENT_TRACKING = 105;

    /**
     * Integer representing the scan option.
     *
     * @var int
     */
    public const OPT_SCAN = 4;

    /**
     * Whether client capable of handling redirect messages.
     *
     * @var int
     */
    public const OPT_CAPA_REDIRECT = 107;

    /**
     * Should we restore subscriptions after reconnecting.
     *
     * @var int
     */
    public const OPT_RESTORE_PUBSUB = 108;

    /**
     * Adaptive caching configuration.
     *
     * @var int
     */
    public const OPT_ADAPTIVE_CACHE = 109;

    /**
     * Issue one `SCAN` command at a time, sometimes returning an empty array of results.
     *
     * @var int
     */
    public const SCAN_NORETRY = 0;

    /**
     * Retry the `SCAN` command until keys come back, or iterator of zero is returned.
     *
     * @var int
     */
    public const SCAN_RETRY = 1;

    /**
     * Prepend the set prefix to any `MATCH` pattern.
     *
     * @var int
     */
    public const SCAN_PREFIX = 2;

    /**
     * Do not prepend the set prefix to any `MATCH` pattern.
     *
     * @var int
     */
    public const SCAN_NOPREFIX = 3;

    /**
     * Redis command argument.
     *
     * @internal
     * @var string
     */
    public const BEFORE = 'BEFORE';

    /**
     * Redis command argument.
     *
     * @internal
     * @var string
     */
    public const AFTER = 'AFTER';

    /**
     * Redis command argument.
     *
     * @internal
     * @var string
     */
    public const LEFT = 'LEFT';

    /**
     * Redis command argument.
     *
     * @internal
     * @var string
     */
    public const RIGHT = 'RIGHT';

    /**
     * Integer representing "key not found".
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_NOT_FOUND = 0;

    /**
     * Integer representing Redis `string` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_STRING = 1;

    /**
     * Integer representing Redis `set` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_SET = 2;

    /**
     * Integer representing Redis `list` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_LIST = 3;

    /**
     * Integer representing Redis `zset` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_ZSET = 4;

    /**
     * Integer representing Redis `hash` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_HASH = 5;

    /**
     * Integer representing Redis `stream` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_STREAM = 6;

    /**
     * Integer representing Redis `vectorset` type.
     *
     * @see Relay::type()
     * @var int
     */
    public const REDIS_VECTORSET = 7;

    /**
     * The adaptive cache object.
     *
     * @readonly
     * @var AdaptiveCache
     */
    public AdaptiveCache $adaptiveCache;

    /**
     * Establishes a new connection to Redis, or reuses already opened connection.
     *
     * @example $context array{
     *   use-cache: bool,                       // Whether to use in-memory caching
     *   adaptive-cache: array{
     *     enabled: bool                        // Whether to disable adaptive caching
     *     width: int                           // Number of horizontal cells in the adaptive cache (Supported values: 512 - (2^31)
     *     depth: int                           // Number of vertical cells (Supported values: 1 - 8)
     *     min_ratio: float                     // Minimum number of reads + writes before a key should be cached.
     *     min_events: int                      // Minimum read-write ratio before a key is cached
     *     formula: string                      // The formula used to calculate the read/write ratio for each key
     *   },
     *   client-tracking: bool,                 // Whether to disable Redis' client tracking (write to in-memory cache only)
     *   client-invalidations: bool,            // Whether to invalidate cached keys without waiting for client tracking
     *   throw-on-error: bool,                  // Whether to throw exceptions when read errors occur
     *   phpredis-compatibility: bool,          // Whether to use PhpRedis compatibility mode (https://relay.so/docs/compatibility)
     *   persistent: bool,                      // Whether to use a persistent connection
     *   prefix: string,                        // Prefix used for all keys
     *   database: int,                         // Database index to switch to
     *   auth: string|array<string>,            // Password or ACL credentials
     *   max-retries: int,                      // Number of reconnection attempts when a command or connection fails
     *   serializer: int,                       // The serializer to use (see `OPT_SERIALIZER_*` constants)
     *   compression: int,                      // The compression algorithm to use (see `OPT_COMPRESSION_*` constants)
     *   compression-level: int                 // The compression level to use
     *   stream: array,                         // TLS options (see https://www.php.net/manual/en/context.ssl.php)
     *   reply-literal: bool,                   // Whether to return reply literals like `OK` instead of `true`
     *   null-mbulk-as-null: bool,              // Whether to return multibulk as empty array or `null`
     *   capa-redirect: bool,                   // Whether to use CAPA redirects (https://valkey.io/commands/client-capa/)
     * } $context
     *
     * @param  string|array|null  $host
     * @param  int  $port
     * @param  float  $connect_timeout
     * @param  float  $command_timeout
     * @param  array  $context
     * @param  int  $database
     */
    #[Attributes\Server]
    public function __construct(
        string|array|null $host = null,
        int $port = 6379,
        float $connect_timeout = 0.0,
        float $command_timeout = 0.0,
        #[\SensitiveParameter] array $context = [],
        int $database = 0,
    ) {}

    /**
     * Establishes a new connection to Redis, or reuses already opened connection.
     *
     * @see self::__construct() for context options.
     *
     * @param  string  $host
     * @param  int  $port
     * @param  float  $timeout
     * @param  string|null  $persistent_id
     * @param  int  $retry_interval
     * @param  float  $read_timeout
     * @param  array  $context
     * @param  int  $database
     * @return bool
     */
    #[Attributes\Server]
    public function connect(
        string $host,
        int $port = 6379,
        float $timeout = 0.0,
        ?string $persistent_id = null,
        int $retry_interval = 0,
        float $read_timeout = 0.0,
        #[\SensitiveParameter] array $context = [],
        int $database = 0
    ): bool {}

    /**
     * Establishes a persistent connection to Redis.
     *
     * @see self::__construct() for context options.
     *
     * @param  string  $host
     * @param  int  $port
     * @param  float  $timeout
     * @param  string|null  $persistent_id
     * @param  int  $retry_interval
     * @param  float  $read_timeout
     * @param  array  $context
     * @param  int  $database
     * @return bool
     */
    #[Attributes\Server]
    public function pconnect(
        string $host,
        int $port = 6379,
        float $timeout = 0.0,
        ?string $persistent_id = null,
        int $retry_interval = 0,
        float $read_timeout = 0.0,
        #[\SensitiveParameter] array $context = [],
        int $database = 0
    ): bool {}

    /**
     * Closes the current connection, unless it's persistent.
     *
     * @return bool
     */
    #[Attributes\Local]
    public function close(): bool {}

    /**
     * Closes the current connection, if it's persistent.
     *
     * @return bool
     */
    #[Attributes\Local]
    public function pclose(): bool {}

    /**
     * Registers a new event listener.
     *
     * @param  callable  $callback
     * @return bool
     */
    #[Attributes\Local]
    public function listen(?callable $callback): bool {}

    /**
     * Registers a new `flushed` event listener.
     *
     * @param  callable  $callback
     * @return bool
     */
    #[Attributes\Local]
    public function onFlushed(?callable $callback): bool {}

    /**
     * Registers a new `invalidated` event listener.
     *
     * @param  callable  $callback
     * @param  string|null  $pattern
     * @return bool
     */
    #[Attributes\Local]
    public function onInvalidated(?callable $callback, ?string $pattern = null): bool {}

    /**
     * Dispatches all pending events.
     *
     * @return int|false
     */
    #[Attributes\Local]
    public function dispatchEvents(): int|false {}

    /**
     * Returns a client option.
     *
     * @param  int  $option
     * @return mixed
     */
    #[Attributes\Local]
    public function getOption(int $option): mixed {}

    /**
     * Returns or sets a client option.
     *
     * @param  int  $option
     * @param  mixed  $value
     * @return mixed
     */
    #[Attributes\Local]
    public function option(int $option, mixed $value = null): mixed {}

    /**
     * Sets a client option.
     *
     * Relay specific options:
     *
     * - `OPT_ALLOW_PATTERNS`
     * - `OPT_IGNORE_PATTERNS`
     * - `OPT_THROW_ON_ERROR`
     * - `OPT_CLIENT_INVALIDATIONS`
     * - `OPT_PHPREDIS_COMPATIBILITY`
     *
     * Supported PhpRedis options:
     *
     * - `OPT_PREFIX`
     * - `OPT_READ_TIMEOUT`
     * - `OPT_COMPRESSION`
     * - `OPT_COMPRESSION_LEVEL`
     * - `OPT_MAX_RETRIES`
     * - `OPT_BACKOFF_ALGORITHM`
     * - `OPT_BACKOFF_BASE`
     * - `OPT_BACKOFF_CAP`
     * - `OPT_SCAN`
     * - `OPT_REPLY_LITERAL`
     * - `OPT_NULL_MULTIBULK_AS_NULL`
     *
     * @param  int  $option
     * @param  mixed  $value
     * @return bool
     */
    #[Attributes\Local]
    public function setOption(int $option, mixed $value): bool {}

    /**
     * Adds ignore pattern(s). Matching keys will not be cached in memory.
     *
     * @param  string  $pattern,...
     * @return int
     */
    #[Attributes\Local]
    public function addIgnorePatterns(string ...$pattern): int {}

    /**
     * Adds allow pattern(s). Only matching keys will be cached in memory.
     *
     * @param  string  $pattern,...
     * @return int
     */
    #[Attributes\Local]
    public function addAllowPatterns(string ...$pattern): int {}

    /**
     * Returns the connection timeout.
     *
     * @return float|false
     */
    #[Attributes\Local]
    public function getTimeout(): float|false {}

    /**
     * @see Relay\Relay::getTimeout()
     *
     * @return float|false
     */
    #[Attributes\Local]
    public function timeout(): float|false {}

    /**
     * Returns the read timeout.
     *
     * @return float|false
     */
    #[Attributes\Local]
    public function getReadTimeout(): float|false {}

    /**
     * @see Relay\Relay::getReadTimeout()
     *
     * @return float|false
     */
    #[Attributes\Local]
    public function readTimeout(): float|false {}

    /**
     * Returns the number of bytes sent and received over the network during the Relay object's
     * lifetime, or since the last time {@link Relay::clearBytes()} was called.
     *
     * @return array
     */
    #[Attributes\Local]
    public function getBytes(): array {}

    /**
     * @see Relay\Relay::getBytes()
     *
     * @return array
     */
    #[Attributes\Local]
    public function bytes(): array {}

    /**
     * Returns the host or unix socket.
     *
     * @return string|false
     */
    #[Attributes\Local]
    public function getHost(): string|false {}

    /**
     * Whether Relay is connected to Redis.
     *
     * @return bool
     */
    #[Attributes\Local]
    public function isConnected(): bool {}

    /**
     * Returns the port.
     *
     * @return int|false
     */
    #[Attributes\Local]
    public function getPort(): int|false {}

    /**
     * Returns the authentication information.
     * In PhpRedis compatibility mode this method returns any configured password in plain-text.
     *
     * @return mixed
     */
    #[Attributes\Local]
    public function getAuth(): mixed {}

    /**
     * Returns the currently selected database.
     *
     * @return int|false
     */
    #[Attributes\Local]
    public function getDbNum(): mixed {}

    /**
     * Returns the serialized value.
     *
     * @param  mixed  $value
     * @return mixed
     */
    #[Attributes\Local]
    public function _serialize(mixed $value): mixed {}

    /**
     * Returns the unserialized value.
     *
     * @param  mixed  $value
     * @return mixed
     */
    #[Attributes\Local]
    public function _unserialize(mixed $value): mixed {}

    /**
     * Compress data with Relay's currently configured compression algorithm.
     *
     * @param  string  $value
     * @return string
     */
    #[Attributes\Local]
    public function _compress(string $value): string {}

    /**
     * Uncompress data with Relay's currently configured compression algorithm.
     *
     * @param  string  $value
     * @return string
     */
    #[Attributes\Local]
    public function _uncompress(string $value): string {}

    /**
     * Returns the serialized and compressed value.
     *
     * @param  mixed  $value
     * @return string
     */
    #[Attributes\Local]
    public function _pack(mixed $value): string {}

    /**
     * Returns the unserialized and decompressed value.
     *
     * @param  mixed  $value
     * @return mixed
     */
    #[Attributes\Local]
    public function _unpack(mixed $value): mixed {}

    /**
     * Returns the value with the prefix.
     *
     * @param  mixed  $value
     * @return string
     */
    #[Attributes\Local]
    public function _prefix(mixed $value): string {}

    /**
     * Returns the last error message, if any.
     *
     * @return string|null
     */
    #[Attributes\Local]
    public function getLastError(): string|null {}

    /**
     * Clears the last error that is set, if any.
     *
     * @return bool
     */
    #[Attributes\Local]
    public function clearLastError(): bool {}

    /**
     * Returns the connection's endpoint identifier.
     *
     * @return string|false
     */
    #[Attributes\Local]
    public function endpointId(): string|false {}

    /**
     * @see Relay\Relay::endpointId()
     *
     * @return string|false
     */
    public function getPersistentID(): string|false {}

    /**
     * Returns a unique representation of the underlying socket connection identifier.
     *
     * @return string|false
     */
    #[Attributes\Local]
    public function socketId(): string|false {}

    /**
     * Returns statistics about Relay.
     *
     * - `usage.total_requests`: The total number of requests we've seen
     * - `usage.active_requests`: The number of requests currently in-flight
     * - `usage.max_active_requests`: The most concurrent in-flight requests we've seen
     * - `usage.free_epoch_records`: The estimated number of free epoch reclamation records
     *
     * - `stats.requests`: The total number of requests the cache has received
     * - `stats.misses`: Requests where we had to ask Redis for a value
     * - `stats.hits`: Requests where we did not have to ask redis for the value
     * - `stats.dirty_skips`: The number of times Relay has skipped an entire database because it was dirty.
     * - `stats.errors`: How many times a 'severe' error occurs (presently this is only incremented if we get a `null` response from hiredis)
     * - `stats.empty`: How many times we've run out of free requests (indicating the size of the ring buffers should be increased)
     * - `stats.oom`: The number of times we've run out of memory
     * - `stats.ops_per_sec`: The number of commands processed per second
     * - `stats.bytes_sent`: The number of bytes Relay has written to the network
     * - `stats.bytes_received`: The number of bytes Relay has read from the network
     * - `stats.command_usec`: Deprecated
     * - `stats.rinit_usec`: The number of microseconds Relay has spent in request initialization
     * - `stats.rshutdown_usec`: The number of microseconds Relay has spent in request shutdown
     * - `stats.sigio_usec`: The number of microseconds Relay has spent in its SIGIO handler
     *
     * - `memory.total`: The total bytes of allocated memory
     * - `memory.limit`: The capped number of bytes Relay has available to use
     * - `memory.active`: The total amount of memory mapped into the allocator
     * - `memory.used`: The amount of memory pointing to live objects including metadata
     *
     * - `endpoints.*.redis`: Information about the connected Redis server of that endpoint
     * - `endpoints.*.connections`: Information about the connection of each worker
     * - `endpoints.*.adaptive_cache`: Information about the adaptive cache for each endpoint
     *
     * @return array
     */
    #[Attributes\Local]
    public static function stats(): array {}

    /**
     * Returns the number of bytes allocated, or `0` in client-only mode.
     *
     * @return int
     */
    #[Attributes\Local]
    public static function maxMemory(): int {}

    /**
     * Returns the number of bytes allocated, or `0` in client-only mode.
     *
     * @deprecated 0.5.0 Use `Relay:maxMemory()`
     *
     * @return int
     */
    #[Attributes\Local]
    public static function memory(): int {}

    /**
     * Execute any command against Redis, without applying
     * the prefix, compression and serialization.
     *
     * @param  string  $cmd
     * @param  mixed  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function rawCommand(string $cmd, mixed ...$args): mixed {}

    /**
     * Select the Redis logical database having the specified zero-based numeric index.
     *
     * @param  int  $db
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function select(int $db): Relay|bool {}

    /**
     * Authenticate the connection using a password or an ACL username and password.
     *
     * @param  mixed  $auth
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function auth(#[\SensitiveParameter] mixed $auth): bool {}

    /**
     * The INFO command returns information and statistics about Redis in a format
     * that is simple to parse by computers and easy to read by humans.
     *
     * @see https://redis.io/commands/info
     *
     * @param  string  $sections,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function info(string ...$sections): Relay|array|false {}

    /**
     * Deletes all the keys of the currently selected database.
     *
     * @param  bool|null  $sync
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function flushdb(?bool $sync = null): Relay|bool {}

    /**
     * Deletes all the keys of all the existing databases, not just the currently selected one.
     *
     * @param  bool|null  $sync
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function flushall(?bool $sync = null): Relay|bool {}

    /**
     * Invokes a Redis function.
     *
     * @param  string  $name
     * @param  array  $keys
     * @param  array  $argv
     * @param  callable|null  $handler
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function fcall(string $name, array $keys = [], array $argv = [], ?callable $handler = null): mixed {}

    /**
     * Invokes a read-only Redis function.
     *
     * @param  string  $name
     * @param  array  $keys
     * @param  array  $argv
     * @param  callable|null  $handler
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function fcall_ro(string $name, array $keys = [], array $argv = [], callable $handler = null): mixed {}

    /**
     * Calls `FUNCTION` sub-command.
     *
     * @param  string  $op
     * @param  string  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function function(string $op, string ...$args): mixed {}

    /**
     * Flushes Relay's in-memory cache of all databases.
     * When given an endpoint, only that connection will be flushed.
     * When given an endpoint and database index, only that database
     * for that connection will be flushed.
     *
     * @param  string|null  $endpointId
     * @param  int|null  $db
     * @return bool
     */
    #[Attributes\Local]
    public static function flushMemory(?string $endpointId = null, int $db = null): bool {}

    /**
     * Retrieve the timestamp of the last *user initiated* flush of the in-memory cache.
     * User initiated flushes can be done globally, specific to a single endpoint, or
     * specific to a single endpoint and database.
     *
     * Since flushes at higher levels imply flushes at lower levels, Relay will return
     * the highest level relevant flush given which level was requested.
     *
     * @param  string|null  $endpointId
     * @param  int|null  $db
     * @return float|false
     */
    #[Attributes\Local]
    public static function lastMemoryFlush(?string $endpointId = null, int $db = null): float|false {}

    /**
     * Run a search query on an index, and perform aggregate
     * transformations on the results, extracting statistics etc from them.
     *
     * @param  mixed  $index
     * @param  string  $query
     * @param  array|null  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftAggregate(mixed $index, string $query, ?array $options = null): Relay|array|false {}

    /**
     * Add an alias to an index.
     *
     * @param  mixed  $index
     * @param  string  $alias
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftAliasAdd(mixed $index, string $alias): Relay|bool {}

    /**
     * Remove an alias from an index.
     *
     * @param  string  $alias
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftAliasDel(string $alias): Relay|bool {}

    /**
     * Add an alias to an index.
     * If the alias is already associated with another index,
     * removes the alias association with the previous index.
     *
     * @param  mixed  $index
     * @param  string  $alias
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftAliasUpdate(mixed $index, string $alias): Relay|bool {}

    /**
     * Add a new attribute to the index.
     * Adding an attribute to the index causes any future doacument updates
     * to use the new attribute when indexing and reindexing existing documents.
     *
     * @param  mixed  $index
     * @param  array  $schema
     * @param  bool  $skipinitialscan
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftAlter(mixed $index, array $schema, bool $skipinitialscan = false): Relay|bool {}

    /**
     * Container command for get/set RediSearch configuration parameter.
     *
     * @param  string  $operation
     * @param  string  $option
     * @param  mixed  $value
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftConfig(string $operation, string $option, mixed $value = null): Relay|array|bool {}

    /**
     * Create an index with the given specification.
     *
     * @param  mixed  $index
     * @param  array  $schema
     * @param  array|null  $options
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftCreate(mixed $index, array $schema, ?array $options = null): Relay|bool {}

    /**
     * Container command for del/read existing cursor.
     *
     * @param  string  $operation
     * @param  mixed  $index
     * @param  mixed  $cursor
     * @param  array|null  $options
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftCursor(string $operation, mixed $index, mixed $cursor, ?array $options = null): Relay|array|bool {}

    /**
     * Add terms to a dictionary.
     *
     * @param  mixed  $dict
     * @param  mixed  $term
     * @param  mixed  $other_terms,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftDictAdd(mixed $dict, mixed $term, mixed ...$other_terms): Relay|int|false {}

    /**
     * Delete terms from a dictionary.
     *
     * @param  mixed  $dict
     * @param  mixed  $term
     * @param  mixed  $other_terms,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftDictDel(mixed $dict, mixed $term, mixed ...$other_terms): Relay|int|false {}

    /**
     * Dump all terms in the given dictionary.
     *
     * @param  mixed  $dict
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftDictDump(mixed $dict): Relay|array|false {}

    /**
     * Delete an index.
     *
     * @param  mixed  $index
     * @param  bool  $dd
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftDropIndex(mixed $index, bool $dd = false): Relay|bool {}

    /**
     * Return the execution plan for a complex query.
     *
     * @param  mixed  $index
     * @param  string  $query
     * @param  int  $dialect
     * @return Relay|string|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftExplain(mixed $index, string $query, int $dialect = 0): Relay|string|false {}

    /**
     * Return the execution plan for a complex query but formatted for easier reading from CLI.
     *
     * @param  mixed  $index
     * @param  string  $query
     * @param  int  $dialect
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftExplainCli(mixed $index, string $query, int $dialect = 0): Relay|array|false {}

    /**
     * Returns information and statistics about a given index.
     *
     * @param  mixed  $index
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftInfo(mixed $index): Relay|array|false {}

    /**
     * Apply FT.SEARCH or FT.AGGREGATE command to collect performance details.
     *
     * @param  mixed  $index
     * @param  string  $command
     * @param  string  $query
     * @param  bool  $limited
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftProfile(mixed $index, string $command, string $query, bool $limited = false): Relay|array|false {}

    /**
     * Search the index with a textual query, returning either documents or just ids.
     *
     * @param  mixed  $index
     * @param  string  $query
     * @param  array|null  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftSearch(mixed $index, string $query, ?array $options = null): Relay|array|false {}

    /**
     * Perform spelling correction on a query, returning suggestions for misspelled terms.
     *
     * @param  mixed  $index
     * @param  string  $query
     * @param  array|null  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftSpellCheck(mixed $index, string $query, ?array $options = null): Relay|array|false {}

    /**
     * Dump the contents of a synonym group.
     *
     * @param  mixed  $index
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftSynDump(mixed $index): Relay|array|false {}

    /**
     * Update a synonym group.
     *
     * @param  mixed  $index
     * @param  string  $synonym
     * @param  mixed  $term_or_terms
     * @param  bool  $skipinitialscan
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftSynUpdate(mixed $index, string $synonym, mixed $term_or_terms, bool $skipinitialscan = false): Relay|bool {}

    /**
     * Return a distinct set of values indexed in a Tag field.
     *
     * @param  mixed  $index
     * @param  string  $tag
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ftTagVals(mixed $index, string $tag): Relay|array|false {}

    /**
     * Returns the number of keys in the currently-selected database.
     *
     * @return Relay|int
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function dbsize(): Relay|int|false {}

    /**
     * Serialize and return the value stored at key in a Redis-specific format.
     *
     * @param  mixed  $key
     * @return Relay|string|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function dump(mixed $key): Relay|string|null|false {}

    /**
     * Attach or detach the instance as a replica of another instance.
     *
     * @param  string|null  $host
     * @param  int  $port
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function replicaof(?string $host = null, $port = 0): Relay|bool {}

    /**
     * Pause the client until sufficient local and/or remote AOF data has been flushed to disk.
     *
     * @param  int  $numlocal
     * @param  int  $numremote
     * @return Relay|array
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function waitaof(int $numlocal, int $numremote, int $timeout): Relay|array|false {}

    /**
     * Create a key associated with a value that is obtained by deserializing the provided serialized value.
     *
     * @param  mixed  $key
     * @param  int  $ttl
     * @param  string  $value
     * @param  array  $options
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function restore(mixed $key, int $ttl, string $value, ?array $options = null): Relay|bool {}

    /**
     * Atomically transfer a key from a Redis instance to another one.
     *
     * @param  string  $host
     * @param  int  $port
     * @param  string|array  $key
     * @param  int  $dstdb
     * @param  int  $timeout
     * @param  bool  $copy
     * @param  bool  $replace
     * @param  mixed  $credentials
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function migrate(
        string $host,
        int $port,
        string|array $key,
        int $dstdb,
        int $timeout,
        bool $copy = false,
        bool $replace = false,
        #[\SensitiveParameter] mixed $credentials = null
    ): Relay|bool {}

    /**
     * This command copies the value stored at the source key to the destination key.
     *
     * @param  mixed  $src
     * @param  mixed  $dst
     * @param  array  $options
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function copy(mixed $src, mixed $dst, ?array $options = null): Relay|bool {}

    /**
     * Asks Redis to echo back the provided string.
     *
     * @param  string  $arg
     * @return Relay|bool|string
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function echo(string $arg): Relay|bool|string {}

    /**
     * Returns PONG if no argument is provided, otherwise return a copy of the argument as a bulk.
     *
     * @param  string  $arg
     * @return Relay|bool|string
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ping(string $arg = null): Relay|bool|string {}

    /**
     * Returns the number of milliseoconds since Relay has seen activity from the server.
     *
     * @return Relay|int|false
     */
    #[Attributes\Local]
    public function idleTime(): Relay|int|false {}

    /**
     * Returns a random key from Redis.
     *
     * @return Relay|string|null|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function randomkey(): Relay|string|null|bool {}

    /**
     * Returns the current time from Redis.
     *
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function time(): Relay|array|false {}

    /**
     * Asynchronously rewrite the append-only file.
     *
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bgrewriteaof(): Relay|bool {}

    /**
     * Returns the UNIX time stamp of the last successful save to disk.
     *
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lastsave(): Relay|int|false {}

    /**
     * Get the longest common subsequence between two string keys.
     *
     * @param  mixed  $key1
     * @param  mixed  $key2
     * @param  array|null  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lcs(mixed $key1, mixed $key2, ?array $options = null): mixed {}

    /**
     * Asynchronously save the dataset to disk.
     *
     * @param  null|string  $arg
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bgsave(null|string $arg = null): Relay|bool {}

    /**
     * Synchronously save the dataset to disk.
     *
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function save(): Relay|bool {}

    /**
     * Returns the role of the instance in the context of replication.
     *
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function role(): Relay|array|false {}

    /**
     * Returns the remaining time to live of a key that has a timeout in seconds.
     *
     * @param  mixed  $key
     * @return Relay|int
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ttl(mixed $key): Relay|int|false {}

    /**
     * Returns the remaining time to live of a key that has a timeout in milliseconds.
     *
     * @param  mixed  $key
     * @return Relay|int
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pttl(mixed $key): Relay|int|false {}

    /**
     * Returns if key(s) exists.
     *
     * @param  mixed  $keys,...
     * @return Relay|bool|int
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function exists(mixed ...$keys): Relay|bool|int {}

    /**
     * Evaluate script using the Lua interpreter.
     *
     * @see https://redis.io/commands/eval
     *
     * @param  mixed  $script
     * @param  array  $args
     * @param  int  $num_keys
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function eval(mixed $script, array $args = [], int $num_keys = 0): mixed {}

    /**
     * Evaluate script using the Lua interpreter. This is just the "read-only" variant of EVAL
     * meaning it can be run on read-only replicas.
     *
     * @see https://redis.io/commands/eval_ro
     *
     * @param  mixed  $script
     * @param  array  $args
     * @param  int  $num_keys
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function eval_ro(mixed $script, array $args = [], int $num_keys = 0): mixed {}

    /**
     * Evaluates a script cached on the server-side by its SHA1 digest.
     *
     *
     * @param  string  $sha
     * @param  array  $args
     * @param  int  $num_keys
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function evalsha(string $sha, array $args = [], int $num_keys = 0): mixed {}

    /**
     * Evaluates a script cached on the server-side by its SHA1 digest. This is just the "read-only" variant
     * of `EVALSHA` meaning it can be run on read-only replicas.
     *
     * @param  string  $sha
     * @param  array  $args
     * @param  int  $num_keys
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function evalsha_ro(string $sha, array $args = [], int $num_keys = 0): mixed {}

    /**
     * Executes `CLIENT` command operations.
     *
     * @param  string  $operation
     * @param  mixed  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function client(string $operation, mixed ...$args): mixed {}

    /**
     * Add one or more members to a geospacial sorted set.
     *
     * @param  string  $key
     * @param  float  $lng
     * @param  float  $lat
     * @param  string  $member
     * @param  mixed  $other_triples_and_options,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geoadd(
        string $key,
        float $lng,
        float $lat,
        string $member,
        mixed ...$other_triples_and_options
    ): Relay|int|false {}

    /**
     * Get the distance between two members of a geospacially encoded sorted set.
     *
     * @param  string  $key
     * @param  string  $src
     * @param  string  $dst
     * @param  string|null  $unit
     * @return Relay|float|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geodist(string $key, string $src, string $dst, ?string $unit = null): Relay|float|null|false {}

    /**
     * Retrieve one or more GeoHash encoded strings for members of the set.
     *
     * @param  string  $key
     * @param  string  $member
     * @param  string  $other_members,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geohash(string $key, string $member, string ...$other_members): Relay|array|false {}

    /**
     * Retrieve members of a geospacially sorted set that are within a certain radius of a location.
     *
     * @param  string  $key
     * @param  float  $lng
     * @param  float  $lat
     * @param  float  $radius
     * @param  string  $unit
     * @param  array  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function georadius(string $key, float $lng, float $lat, float $radius, string $unit, array $options = []): mixed {}

    /**
     * Similar to `GEORADIUS` except it uses a member as the center of the query.
     *
     * @param  string  $key
     * @param  string  $member
     * @param  float  $radius
     * @param  string  $unit
     * @param  array  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function georadiusbymember(string $key, string $member, float $radius, string $unit, array $options = []): mixed {}

    /**
     * Similar to `GEORADIUS` except it uses a member as the center of the query.
     *
     * @param  string  $key
     * @param  string  $member
     * @param  float  $radius
     * @param  string  $unit
     * @param  array  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function georadiusbymember_ro(string $key, string $member, float $radius, string $unit, array $options = []): mixed {}

    /**
     * Retrieve members of a geospacially sorted set that are within a certain radius of a location.
     *
     * @param  string  $key
     * @param  float  $lng
     * @param  float  $lat
     * @param  float  $radius
     * @param  string  $unit
     * @param  array  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function georadius_ro(string $key, float $lng, float $lat, float $radius, string $unit, array $options = []): mixed {}

    /**
     * Search a geospacial sorted set for members in various ways.
     *
     * @param  string  $key
     * @param  array|string  $position
     * @param  array|int|float  $shape
     * @param  string  $unit
     * @param  array  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geosearch(
        string $key,
        array|string $position,
        array|int|float $shape,
        string $unit,
        array $options = []
    ): Relay|array|false {}

    /**
     * Search a geospacial sorted set for members within a given area or range,
     * storing the results into a new set.
     *
     * @param  string  $dst
     * @param  string  $src
     * @param  array|string  $position
     * @param  array|int|float  $shape
     * @param  string  $unit
     * @param  array  $options
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geosearchstore(
        string $dst,
        string $src,
        array|string $position,
        array|int|float $shape,
        string $unit,
        array $options = []
    ): Relay|int|false {}

    /**
     * Get the value of key.
     *
     * @param  mixed  $key
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function get(mixed $key): mixed {}

    /**
     * Get the value and metadata of key.
     *
     * Result is an array with value and metadata or `false` in case of error.
     * Currently metadata contains following elements:
     *  - cached  whether value comes from in-memory cache or from server
     *  - length  number of bytes used to store value
     *
     * @param  mixed  $key
     * @return Relay|array{0: mixed, 1: array{cached: bool, length: int}}|false
     */
    #[Attributes\Server, Attributes\Cached]
    public function getWithMeta(mixed $key): Relay|array|false {}

    /**
     * Atomically sets key to value and returns the old value stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function getset(mixed $key, mixed $value): mixed {}

    /**
     * Returns the substring of the string value stored at key,
     * determined by the offsets start and end (both are inclusive).
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $end
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function getrange(mixed $key, int $start, int $end): mixed {}

    /**
     * Overwrites part of the string stored at key, starting at
     * the specified offset, for the entire length of value.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  mixed  $value
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function setrange(mixed $key, int $start, mixed $value): Relay|int|false {}

    /**
     * Returns the bit value at offset in the string value stored at key.
     *
     * @param  mixed  $key
     * @param  int  $pos
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function getbit(mixed $key, int $pos): Relay|int|false {}

    /**
     * Count the number of set bits (population counting) in a string.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $end
     * @param  bool  $by_bit
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function bitcount(mixed $key, int $start = 0, int $end = -1, bool $by_bit = false): Relay|int|false {}

    /**
     * Perform various bitfield operations on a string key,
     * such as getting/setting bit ranges, incrementing, etc.
     *
     * @param  mixed  $key
     * @param  mixed  $args,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bitfield(mixed $key, mixed ...$args): Relay|array|false {}

    /**
     * This is a container command for runtime configuration commands.
     *
     * @param  string  $operation
     * @param  mixed  $key
     * @param  string|null  $value
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function config(string $operation, mixed $key = null, ?string $value = null): Relay|array|bool {}

    /**
     * Return an array with details about every Redis command.
     *
     * @param  array  $args,...
     * @return Relay|array|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function command(mixed ...$args): Relay|array|int|false {}

    /**
     * Perform a bitwise operation on one or more keys, storing the result in a new key.
     *
     * @param  string  $operation
     * @param  string  $dstkey
     * @param  string  $srckey
     * @param  string  $other_keys,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bitop(string $operation, string $dstkey, string $srckey, string ...$other_keys): Relay|int|false {}

    /**
     * Return the position of the first bit set to 1 or 0 in a string.
     *
     * @param  mixed  $key
     * @param  int  $bit
     * @param  int  $start
     * @param  int  $end
     * @param  bool  $bybit
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bitpos(mixed $key, int $bit, int $start = null, int $end = null, bool $bybit = false): Relay|int|false {}

    /**
     * Sets or clears the bit at offset in the string value stored at key.
     *
     * @param  mixed  $key
     * @param  int  $pos
     * @param  int  $val
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function setbit(mixed $key, int $pos, int $val): Relay|int|false {}

    /**
     * Interact with ACLs.
     *
     * @param  string  $cmd
     * @param  string  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function acl(string $cmd, string ...$args): mixed {}

    /**
     * If key already exists and is a string, this command appends
     * the value at the end of the string. If key does not exist
     * it is created and set as an empty string, so APPEND will
     * be similar to SET in this special case.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function append(mixed $key, mixed $value): Relay|int|false {}

    /**
     * Set key to hold the string value. If key already holds
     * a value, it is overwritten, regardless of its type.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @param  mixed  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function set(mixed $key, mixed $value, mixed $options = null): mixed {}

    /**
     * Get the value of key and optionally set its expiration.
     * GETEX is similar to GET, but is a write command with additional options.
     *
     * @param  mixed  $key
     * @param  array  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function getex(mixed $key, ?array $options = null): mixed {}

    /**
     * Get the value of key and delete the key. This command is similar to GET,
     * except for the fact that it also deletes the key on success
     * (if and only if the key's value type is a string).
     *
     * @param  mixed  $key
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function getdel(mixed $key): mixed {}

    /**
     * Set key to hold the string value and set key to timeout after a given number of seconds.
     *
     * @param  mixed  $key
     * @param  int  $seconds
     * @param  mixed  $value
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function setex(mixed $key, int $seconds, mixed $value): Relay|bool {}

    /**
     * Adds the specified elements to the specified HyperLogLog.
     *
     * @param  string  $key
     * @param  array  $elements
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pfadd(string $key, array $elements): Relay|int|false {}

    /**
     * Return the approximated cardinality of the set(s) observed by the HyperLogLog at key(s).
     *
     * @param  string|array  $key_or_keys
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pfcount(string|array $key_or_keys): Relay|int|false {}

    /**
     * Merge given HyperLogLogs into a single one.
     *
     * @param  string  $dst
     * @param  array  $srckeys
     * @return Relay|bool
     */
    public function pfmerge(string $dst, array $srckeys): Relay|bool {}

    /**
     * Set key to hold the string value and set key to timeout after a given number of milliseconds.
     *
     *
     * @param  mixed  $key
     * @param  int  $milliseconds
     * @param  mixed  $value
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function psetex(mixed $key, int $milliseconds, mixed $value): Relay|bool {}

    /**
     * Posts a message to the given channel.
     *
     * @param  string  $channel
     * @param  string  $message
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function publish(string $channel, string $message): Relay|int|false {}

    /**
     * A container command for Pub/Sub introspection commands.
     *
     * @param  string  $operation
     * @param  mixed  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pubsub(string $operation, mixed ...$args): mixed {}

    /**
     * Posts a message to the given shard channel.
     *
     * @param  string  $channel
     * @param  string  $message
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function spublish(string $channel, string $message): Relay|int|false {}

    /**
     * Set key to hold string value if key does not exist. In that case, it is equal to SET.
     * When key already holds a value, no operation is performed.
     * SETNX is short for "SET if Not eXists".
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function setnx(mixed $key, mixed $value): Relay|bool {}

    /**
     * Returns the values of all specified keys.
     *
     * @param  array  $keys
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function mget(array $keys): Relay|array|false {}

    /**
     * Move key from the currently selected database to the specified destination database.
     *
     * @param  mixed  $key
     * @param  int  $db
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function move(mixed $key, int $db): Relay|int|false {}

    /**
     * Sets the given keys to their respective values.
     * MSET replaces existing values with new values, just as regular SET.
     *
     * @param  array  $kvals
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function mset(array $kvals): Relay|bool {}

    /**
     * Sets the given keys to their respective values.
     * MSETNX will not perform any operation at all even if just a single key already exists.
     *
     * @param  array  $kvals
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function msetnx(array $kvals): Relay|bool {}

    /**
     * Renames key.
     *
     * @param  mixed  $key
     * @param  mixed  $newkey
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function rename(mixed $key, mixed $newkey): Relay|bool {}

    /**
     * Renames key if the new key does not yet exist.
     *
     * @param  mixed  $key
     * @param  mixed  $newkey
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function renamenx(mixed $key, mixed $newkey): Relay|bool {}

    /**
     * Removes the specified keys.
     *
     * @param  mixed  $keys,...
     * @return Relay|int|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function del(mixed ...$keys): Relay|int|bool {}

    /**
     * Remove a key if it equals the provided value.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @return Relay|int|false
     */
    #[Attributes\ValkeyCommand]
    public function delifeq(mixed $key, mixed $value): Relay|int|false {}

    /**
     * Removes the specified keys without blocking Redis.
     *
     * @param  mixed  $keys,...
     * @return Relay|int
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function unlink(mixed ...$keys): Relay|int|false {}

    /**
     * Set a timeout on key.
     *
     * @param  mixed  $key
     * @param  int  $seconds
     * @param  string|null  $mode
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function expire(mixed $key, int $seconds, ?string $mode = null): Relay|bool {}

    /**
     * Set a key's time to live in milliseconds.
     *
     * @param  mixed  $key
     * @param  int  $milliseconds
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pexpire(mixed $key, int $milliseconds): Relay|bool {}

    /**
     * Set a timeout on key using a unix timestamp.
     *
     * @param  mixed  $key
     * @param  int  $timestamp
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function expireat(mixed $key, int $timestamp): Relay|bool {}

    /**
     * Returns the absolute Unix timestamp in seconds at which the given key will expire.
     * If the key exists but doesn't have a TTL this function return -1.
     * If the key does not exist -2.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function expiretime(mixed $key): Relay|int|false {}

    /**
     * Set the expiration for a key as a UNIX timestamp specified in milliseconds.
     *
     * @param  mixed  $key
     * @param  int  $timestamp_ms
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pexpireat(mixed $key, int $timestamp_ms): Relay|bool {}

    /**
     * Semantic the same as EXPIRETIME, but returns the absolute Unix expiration
     * timestamp in milliseconds instead of seconds.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function pexpiretime(mixed $key): Relay|int|false {}

    /**
     * Remove the existing timeout on key, turning the key from volatile to persistent.
     *
     * @param  mixed  $key
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function persist(mixed $key): Relay|bool {}

    /**
     * Returns the type of a given key.
     *
     * In PhpRedis compatibility mode this will return an integer
     * (one of the REDIS_<type>) constants. Otherwise it will
     * return the string that Redis returns.
     *
     * @param  mixed  $key
     * @return Relay|int|string|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function type(mixed $key): Relay|int|string|bool {}

    /**
     * Atomically returns and removes the first/last element of the list
     * stored at source, and pushes the element at the first/last
     * element of the list stored at destination.
     *
     * @param  mixed  $srckey
     * @param  mixed  $dstkey
     * @param  string  $srcpos
     * @param  string  $dstpos
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lmove(mixed $srckey, mixed $dstkey, string $srcpos, string $dstpos): mixed {}

    /**
     * BLMOVE is the blocking variant of LMOVE. When source contains elements,
     * this command behaves exactly like LMOVE. When used inside a
     * MULTI/EXEC block, this command behaves exactly like LMOVE.
     *
     * @param  mixed  $srckey
     * @param  mixed  $dstkey
     * @param  string  $srcpos
     * @param  string  $dstpos
     * @param  float  $timeout
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function blmove(mixed $srckey, mixed $dstkey, string $srcpos, string $dstpos, float $timeout): mixed {}

    /**
     * Returns the specified elements of the list stored at key.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $stop
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function lrange(mixed $key, int $start, int $stop): Relay|array|false {}

    /**
     * Insert all the specified values at the head of the list stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  mixed  $mems,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lpush(mixed $key, mixed $mem, mixed ...$mems): Relay|int|false {}

    /**
     * Insert all the specified values at the tail of the list stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  mixed  $mems,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function rpush(mixed $key, mixed $mem, mixed ...$mems): Relay|int|false {}

    /**
     * Inserts specified values at the head of the list stored at key,
     * only if key already exists and holds a list.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  mixed  $mems,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lpushx(mixed $key, mixed $mem, mixed ...$mems): Relay|int|false {}

    /**
     * Inserts specified values at the tail of the list stored at key,
     * only if key already exists and holds a list.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  mixed  $mems,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function rpushx(mixed $key, mixed $mem, mixed ...$mems): Relay|int|false {}

    /**
     * Sets the list element at index to element.
     *
     * @param  mixed  $key
     * @param  int  $index
     * @param  mixed  $mem
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lset(mixed $key, int $index, mixed $mem): Relay|bool {}

    /**
     * Removes and returns the first elements of the list stored at key.
     *
     * @param  mixed  $key
     * @param  int  $count
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lpop(mixed $key, int $count = 1): mixed {}

    /**
     * The command returns the index of matching elements inside a Redis list.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @param  array  $options
     * @return Relay|int|array|false|null
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function lpos(mixed $key, mixed $value, ?array $options = null): Relay|int|array|false|null {}

    /**
     * Removes and returns the last elements of the list stored at key.
     *
     * @param  mixed  $key
     * @param  int  $count
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function rpop(mixed $key, int $count = 1): mixed {}

    /**
     * Atomically returns and removes the last element (tail) of the list stored at source,
     * and pushes the element at the first element (head) of the list stored at destination.
     *
     * @param  mixed  $source
     * @param  mixed  $dest
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function rpoplpush(mixed $source, mixed $dest): mixed {}

    /**
     * Atomically returns and removes the last element (tail) of the list stored at source,
     * and pushes the element at the first element (head) of the list stored at destination.
     * This command will block for an element up to the provided timeout.
     *
     * @param  mixed  $source
     * @param  mixed  $dest
     * @param  float  $timeout
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function brpoplpush(mixed $source, mixed $dest, float $timeout): mixed {}

    /**
     * BLPOP is a blocking list pop primitive. It is the blocking version of LPOP because
     * it blocks the connection when there are no elements to pop from any of the given lists.
     *
     * @param  string|array  $key
     * @param  string|float  $timeout_or_key
     * @param  array  $extra_args,...
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function blpop(string|array $key, string|float $timeout_or_key, mixed ...$extra_args): Relay|array|null|false {}

    /**
     * Pop elements from a list, or block until one is available.
     *
     * @param  float  $timeout
     * @param  array  $keys
     * @param  string  $from
     * @param  int  $count
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function blmpop(float $timeout, array $keys, string $from, int $count = 1): Relay|array|null|false {}

    /**
     * Remove and return members with scores in a sorted set or block until one is available.
     *
     * @param  float  $timeout
     * @param  array  $keys
     * @param  string  $from
     * @param  int  $count
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bzmpop(float $timeout, array $keys, string $from, int $count = 1): Relay|array|null|false {}

    /**
     * Pops one or more elements from the first non-empty list key from the list of provided key names.
     *
     * @param  array  $keys
     * @param  string  $from
     * @param  int  $count
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lmpop(array $keys, string $from, int $count = 1): Relay|array|null|false {}

    /**
     * Pops one or more elements, that are member-score pairs, from the
     * first non-empty sorted set in the provided list of key names.
     *
     * @param  array  $keys
     * @param  string  $from
     * @param  int  $count
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zmpop(array $keys, string $from, int $count = 1): Relay|array|null|false {}

    /**
     * BRPOP is a blocking list pop primitive. It is the blocking version of RPOP because
     * it blocks the connection when there are no elements to pop from any of the given lists.
     *
     * @param  string|array  $key
     * @param  string|float  $timeout_or_key
     * @param  array  $extra_args,...
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function brpop(string|array $key, string|float $timeout_or_key, mixed ...$extra_args): Relay|array|null|false {}

    /**
     * BZPOPMAX is the blocking variant of the sorted set ZPOPMAX primitive.
     *
     * @param  string|array  $key
     * @param  string|float  $timeout_or_key
     * @param  array  $extra_args,...
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bzpopmax(string|array $key, string|float $timeout_or_key, mixed ...$extra_args): Relay|array|null|false {}

    /**
     * BZPOPMIN is the blocking variant of the sorted set ZPOPMIN primitive.
     *
     * @param  string|array  $key
     * @param  string|float  $timeout_or_key
     * @param  array  $extra_args,...
     * @return Relay|array|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function bzpopmin(string|array $key, string|float $timeout_or_key, mixed ...$extra_args): Relay|array|null|false {}

    /**
     * This is a container command for object introspection commands.
     *
     * @param  string  $op
     * @param  mixed  $key
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function object(string $op, mixed $key): mixed {}

    /**
     * Return the positions (longitude,latitude) of all the specified members
     * of the geospatial index represented by the sorted set at key.
     *
     * @param  mixed  $key
     * @param  mixed  $members,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function geopos(mixed $key, mixed ...$members): Relay|array|false {}

    /**
     * Removes the first count occurrences of elements equal to element from the list stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  int  $count
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function lrem(mixed $key, mixed $mem, int $count = 0): Relay|int|false {}

    /**
     * Returns the element at index index in the list stored at key.
     *
     * @param  mixed  $key
     * @param  int  $index
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function lindex(mixed $key, int $index): mixed {}

    /**
     * Inserts element in the list stored at key either before or after the reference value pivot.
     *
     * @param  mixed  $key
     * @param  string  $op
     * @param  mixed  $pivot
     * @param  mixed  $element
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function linsert(mixed $key, string $op, mixed $pivot, mixed $element): Relay|int|false {}

    /**
     * Trim an existing list so that it will contain only the specified range of elements specified.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $end
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ltrim(mixed $key, int $start, int $end): Relay|bool {}

    /**
     * Returns the value associated with field in the hash stored at key.
     *
     * @param  mixed  $hash
     * @param  mixed  $member
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hget(mixed $hash, mixed $member): mixed {}

    /**
     * Returns one or more fields while also setting an expiration on them.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @param  mixed  $expiry = null
     * @return Relay|array|false
     */
    public function hgetex(mixed $hash, array $fields, mixed $expiry = null): Relay|array|false {}

    /**
     * Returns the string length of the value associated with field in the hash stored at key.
     *
     * @param  mixed  $hash
     * @param  mixed  $member
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hstrlen(mixed $hash, mixed $member): Relay|int|false {}

    /**
     * Returns all fields and values of the hash stored at key.
     *
     * @param  mixed  $hash
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hgetall(mixed $hash): Relay|array|false {}

    /**
     * Returns all field names in the hash stored at key.
     *
     * @param  mixed  $hash
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hkeys(mixed $hash): Relay|array|false {}

    /**
     * Returns all values in the hash stored at key.
     *
     * @param  mixed  $hash
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hvals(mixed $hash): Relay|array|false {}

    /**
     * Returns the values associated with the specified fields in the hash stored at key.
     *
     * @param  mixed  $hash
     * @param  array  $members
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hmget(mixed $hash, array $members): Relay|array|false {}

    /**
     * Gets and deletes one or more hash fields.
     *
     * @param  mixed  $key
     * @param  array  $fields
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hgetdel(mixed $key, array $fields): Relay|array|false {}

    /**
     * When called with just the key argument, return a random field from the hash value stored at key.
     *
     * @param  mixed  $hash
     * @param  array  $options
     * @return Relay|array|string|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hrandfield(mixed $hash, ?array $options = null): Relay|array|string|null|false {}

    /**
     * Sets the specified fields to their respective values in the hash stored at key.
     *
     * @param  mixed  $hash
     * @param  array  $members
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function hmset(mixed $hash, array $members): Relay|bool {}

    /**
     * Returns if field is an existing field in the hash stored at key.
     *
     * @param  mixed  $hash
     * @param  mixed  $member
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hexists(mixed $hash, mixed $member): Relay|bool {}

    /**
     * Se an expiration for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  int  $ttl
     * @param  array  $fields
     * @param  string  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hexpire(mixed $hash, int $ttl, array $fields, ?string $mode = null): Relay|array|false {}

    /**
     * Set a millisecond resolution expiry on one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  int  $ttl
     * @param  array  $fields
     * @param  string  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hpexpire(mixed $hash, int $ttl, array $fields, ?string $mode = null): Relay|array|false {}

    /**
     * Set a unix timestamp expiration for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  int  $ttl
     * @param  array  $fields
     * @param  string  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hexpireat(mixed $hash, int $ttl, array $fields, ?string $mode = null): Relay|array|false {}

    /**
     * Set a millisecond resolution unix timestamp expiration for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  int  $ttl
     * @param  array  $fields
     * @param  string  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hpexpireat(mixed $hash, int $ttl, array $fields, ?string $mode = null): Relay|array|false {}

    /**
     * Get the expire time in seconds for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function httl(mixed $hash, array $fields): Relay|array|false {}

    /**
     * Get the expire time in milliseconds for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hpttl(mixed $hash, array $fields): Relay|array|false {}

    /**
     * Get the unix timestamp expiration time for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hexpiretime(mixed $hash, array $fields): Relay|array|false {}

    /**
     * Get the millisecond precision unix timestamp
     * expiration time for one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function hpexpiretime(mixed $hash, array $fields): Relay|array|false {}

    /**
     * Persist one or more hash fields.
     *
     * @param  mixed  $hash
     * @param  array  $fields
     * @return Relay|array|false
     */
    public function hpersist(mixed $hash, array $fields): Relay|array|false {}

    /**
     * Sets field in the hash stored at key to value, only if field does not yet exist.
     *
     * @param  mixed  $hash
     * @param  mixed  $member
     * @param  mixed  $value
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hsetnx(mixed $hash, mixed $member, mixed $value): Relay|bool {}

    /**
     * Sets field in the hash stored at key to value.
     *
     * @param  mixed  $key
     * @param  mixed  $keys_and_vals...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hset(mixed $key, mixed ...$keys_and_vals): Relay|int|false {}

    /**
     * Set one or more hash fields and values with expiration options.
     *
     * @param  mixed  $key
     * @param  array  $fields
     * @param  null|int|float|array  $expiry = null
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function hsetex(mixed $key, array $fields, null|int|float|array $expiry = null): Relay|int|false {}

    /**
     * Removes the specified fields from the hash stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  string  $mems,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hdel(mixed $key, mixed $mem, string ...$mems): Relay|int|false {}

    /**
     * Increments the number stored at field in the hash stored at key by increment.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  int  $value
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hincrby(mixed $key, mixed $mem, int $value): Relay|int|false {}

    /**
     * Increment the specified field of a hash stored at key, and representing
     * a floating point number, by the specified increment.
     *
     * @param  mixed  $key
     * @param  mixed  $mem
     * @param  float  $value
     * @return Relay|float|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hincrbyfloat(mixed $key, mixed $mem, float $value): Relay|float|bool {}

    /**
     * Increments the number stored at key by one.
     *
     * @param  mixed  $key
     * @param  int  $by
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function incr(mixed $key, int $by = 1): Relay|int|false {}

    /**
     * Decrements the number stored at key by one.
     *
     * @param  mixed  $key
     * @param  int  $by
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function decr(mixed $key, int $by = 1): Relay|int|false {}

    /**
     * Increments the number stored at key by increment.
     *
     * @param  mixed  $key
     * @param  int  $value
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function incrby(mixed $key, int $value): Relay|int|false {}

    /**
     * Decrements the number stored at key by decrement.
     *
     * @param  mixed  $key
     * @param  int  $value
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function decrby(mixed $key, int $value): Relay|int|false {}

    /**
     * Increment the string representing a floating point number stored at key by the specified increment.
     *
     * @param  mixed  $key
     * @param  float  $value
     * @return Relay|float|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function incrbyfloat(mixed $key, float $value): Relay|float|false {}

    /**
     * Append the json values into the array at path after the last element in it.
     *
     * @param  mixed  $key
     * @param  mixed  $value_or_array
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrAppend(mixed $key, mixed $value_or_array, ?string $path = null): Relay|array|false {}

    /**
     * Search for the first occurrence of a JSON value in an array.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  mixed  $value
     * @param  int|null  $start
     * @param  int|null  $stop
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrIndex(mixed $key, string $path, mixed $value, ?int $start = 0, ?int $stop = -1): Relay|array|false {}

    /**
     * Insert the json values into the array at path before the index (shifts to the right).
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  int  $index
     * @param  mixed  $value
     * @param  mixed  $other_values,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrInsert(mixed $key, string $path, int $index, mixed $value, mixed ...$other_values): Relay|array|false {}

    /**
     * Report the length of the JSON array at path in key.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrLen(mixed $key, ?string $path = null): Relay|array|false {}

    /**
     * Remove and return an element from the index in the array.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @param  int  $index
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrPop(mixed $key, ?string $path = null, int $index = -1): Relay|array|false {}

    /**
     * Trim an array so that it contains only the specified inclusive range of elements.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  int  $start
     * @param  int  $stop
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonArrTrim(mixed $key, string $path, int $start, int $stop): Relay|array|false {}

    /**
     * Clear container values (arrays/objects) and set numeric values to 0.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonClear(mixed $key, ?string $path = null): Relay|int|false {}

    /**
     * Container command for JSON debugging related tasks.
     *
     * @param  string  $command
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonDebug(string $command, mixed $key, ?string $path = null): Relay|int|false {}

    /**
     * Delete a value.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonDel(mixed $key, ?string $path = null): Relay|int|false {}

    /**
     * @see Relay::jsonDel
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonForget(mixed $key, ?string $path = null): Relay|int|false {}

    /**
     * Return the value at path in JSON serialized form.
     *
     * @param  mixed  $key
     * @param  array  $options
     * @param  string  $paths,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonGet(mixed $key, array $options = [], string ...$paths): mixed {}

    /**
     * Merge a given JSON value into matching paths. Consequently, JSON values
     * at matching paths are updated, deleted, or expanded with new children.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  mixed  $value
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonMerge(mixed $key, string $path, mixed $value): Relay|bool {}

    /**
     * Return the values at path from multiple key arguments.
     *
     * @param  mixed  $key_or_array
     * @param  string  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonMget(mixed $key_or_array, string $path): Relay|array|false {}

    /**
     * Set or update one or more JSON values according to the specified key-path-value triplets.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  mixed  $value
     * @param  mixed  $other_triples
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonMset(mixed $key, string $path, mixed $value, mixed ...$other_triples): Relay|bool {}

    /**
     * Increment the number value stored at path by number.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  int  $value
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonNumIncrBy(mixed $key, string $path, int $value): Relay|array|false {}

    /**
     * Multiply the number value stored at path by number.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  int  $value
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonNumMultBy(mixed $key, string $path, int $value): Relay|array|false {}

    /**
     * Return the keys in the object that's referenced by path.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonObjKeys(mixed $key, ?string $path = null): Relay|array|false {}

    /**
     * Report the number of keys in the JSON object at path in key.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonObjLen(mixed $key, ?string $path = null): Relay|array|false {}

    /**
     * Return the JSON in key in RESP specification form.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|string|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonResp(mixed $key, ?string $path = null): Relay|array|string|int|false {}

    /**
     * Set the JSON value at path in key.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @param  mixed  $value
     * @param  string|null  $condition
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonSet(mixed $key, string $path, mixed $value, ?string $condition = null): Relay|bool {}

    /**
     * Append the json-string values to the string at path.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonStrAppend(mixed $key, mixed $value, ?string $path = null): Relay|array|false {}

    /**
     * Report the length of the JSON String at path in key.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonStrLen(mixed $key, ?string $path = null): Relay|array|false {}

    /**
     * Toggle a Boolean value stored at path.
     *
     * @param  mixed  $key
     * @param  string  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonToggle(mixed $key, string $path): Relay|array|false {}

    /**
     * Report the type of JSON value at path.
     *
     * @param  mixed  $key
     * @param  string|null  $path
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function jsonType(mixed $key, ?string $path = null): Relay|array|false {}

    /**
     * Returns the members of the set resulting from the difference
     * between the first set and all the successive sets.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function sdiff(mixed $key, mixed ...$other_keys): Relay|array|false {}

    /**
     * This command is equal to SDIFF, but instead of returning the resulting set,
     * it is stored in destination. If destination already exists, it is overwritten.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sdiffstore(mixed $key, mixed ...$other_keys): Relay|int|false {}

    /**
     * Returns the members of the set resulting from the intersection of all the given sets.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function sinter(mixed $key, mixed ...$other_keys): Relay|array|false {}

    /**
     * Intersect multiple sets and return the cardinality of the result.
     *
     * @param  array  $keys
     * @param  int  $limit
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function sintercard(array $keys, int $limit = -1): Relay|int|false {}

    /**
     * This command is equal to SINTER, but instead of returning the resulting set,
     * it is stored in destination. If destination already exists, it is overwritten.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sinterstore(mixed $key, mixed ...$other_keys): Relay|int|false {}

    /**
     * Returns the members of the set resulting from the union of all the given sets.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function sunion(mixed $key, mixed ...$other_keys): Relay|array|false {}

    /**
     * This command is equal to SUNION, but instead of returning the resulting set,
     * it is stored in destination. If destination already exists, it is overwritten.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sunionstore(mixed $key, mixed ...$other_keys): Relay|int|false {}

    /**
     * Subscribes to the specified channels.
     *
     * @param  array  $channels
     * @param  callable  $callback
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function subscribe(array $channels, callable $callback): bool {}

    /**
     * Unsubscribes from the given channels, or from all of them if none is given.
     *
     * @param  array  $channels
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function unsubscribe(array $channels = []): bool {}

    /**
     * Subscribes to the given patterns.
     *
     * @param  array  $patterns
     * @param  callable  $callback
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function psubscribe(array $patterns, callable $callback): bool {}

    /**
     * Unsubscribes from the given patterns, or from all of them if none is given.
     *
     * @param  array  $patterns
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function punsubscribe(array $patterns = []): bool {}

    /**
     * Subscribes to the specified shard channels.
     *
     * @param  array  $channels
     * @param  callable  $callback
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function ssubscribe(array $channels, callable $callback): bool {}

    /**
     * Unsubscribes from the given shard channels, or from all of them if none is given.
     *
     * @param  array  $channels
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sunsubscribe(array $channels = []): bool {}

    /**
     * Alters the last access time of a key(s).
     *
     * @param  array|string  $key_or_array
     * @param  mixed  $more_keys,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function touch(array|string $key_or_array, mixed ...$more_keys): Relay|int|false {}

    /**
     * A pipeline block is simply transmitted faster to the server (like `MULTI`), but without any guarantee of atomicity.
     *
     * @return Relay|bool
     */
    #[Attributes\Local]
    public function pipeline(): Relay|bool {}

    /**
     * Marks the start of a transaction block. Subsequent commands will be queued for atomic execution using EXEC.
     *
     * Accepts `Relay::MULTI` and `Relay::PIPELINE` modes.
     *
     * @param  int  $mode
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function multi(int $mode = 0): Relay|bool {}

    /**
     * Executes all previously queued commands in a transaction and restores the connection state to normal.
     *
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function exec(): Relay|array|bool {}

    /**
     * Wait for the synchronous replication of all the write
     * commands sent in the context of the current connection.
     *
     * @param  int  $replicas
     * @param  int  $timeout
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function wait(int $replicas, $timeout): Relay|int|false {}

    /**
     * Marks the given keys to be watched for conditional execution of a transaction.
     *
     * @param  mixed  $key
     * @param  mixed  $other_keys,...
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function watch(mixed $key, mixed ...$other_keys): Relay|bool {}

    /**
     * Flushes all the previously watched keys for a transaction.
     * If you call EXEC or DISCARD, there's no need to manually call UNWATCH.
     *
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function unwatch(): Relay|bool {}

    /**
     * Flushes all previously queued commands in a transaction and restores the connection state to normal.
     * If WATCH was used, DISCARD unwatches all keys watched by the connection.
     *
     * @return bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function discard(): bool {}

    /**
     * Get the server name as reported by the `HELLO` response.
     *
     * @return string|false
     */
    #[Attributes\Local]
    public function serverName(): string|false {}

    /**
     * Get the server version as reported by the `HELLO` response.
     *
     * @return string|false
     */
    #[Attributes\Local]
    public function serverVersion(): string|false {}

    /**
     * Get the mode Relay is currently in.
     * `Relay::ATOMIC`, `Relay::PIPELINE` or `Relay::MULTI`.
     *
     * @param  bool  $masked
     * @return int
     */
    #[Attributes\Local]
    public function getMode(bool $masked = false): int {}

    /**
     * Clear the accumulated sent and received bytes.
     *
     * @return void
     */
    #[Attributes\Local]
    public function clearBytes(): void {}

    /**
     * Scan the keyspace for matching keys.
     *
     * @param  mixed  &$iterator
     * @param-out  mixed  $iterator
     * @param  mixed  $match
     * @param  int  $count
     * @param  string|null  $type
     * @return array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function scan(mixed &$iterator, mixed $match = null, int $count = 0, ?string $type = null): array|false {}

    /**
     * Iterates fields of Hash types and their associated values.
     *
     * @param  mixed  $key
     * @param  mixed  &$iterator
     * @param-out mixed $iterator
     * @param  mixed  $match
     * @param  int  $count
     * @return array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function hscan(mixed $key, mixed &$iterator, mixed $match = null, int $count = 0): array|false {}

    /**
     * Iterates elements of Sets types.
     *
     * @param  mixed  $key
     * @param  mixed  &$iterator
     * @param-out mixed $iterator
     * @param  mixed  $match
     * @param  int  $count
     * @return array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sscan(mixed $key, mixed &$iterator, mixed $match = null, int $count = 0): array|false {}

    /**
     * Iterates elements of Sorted Set types and their associated scores.
     *
     * @param  mixed  $key
     * @param  mixed  &$iterator
     * @param-out mixed $iterator
     * @param  mixed  $match
     * @param  int  $count
     * @return array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zscan(mixed $key, mixed &$iterator, mixed $match = null, int $count = 0): array|false {}

    /**
     * Returns all keys matching pattern.
     *
     * @param  mixed  $pattern
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function keys(mixed $pattern): Relay|array|false {}

    /**
     * Interact with Valkey's COMMANDLOG command.
     *
     * @param  string  $subcmd
     * @param  mixed  $args...
     * @return Relay|array|int|bool
     */
    #[Attributes\ValkeyCommand]
    public function commandlog(string $subcmd, mixed ...$args): Relay|array|int|bool {}

    /**
     * Interact with the Redis slowlog.
     *
     * @param  string  $operation
     * @param  string  $extra_args,...
     * @return Relay|array|int|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function slowlog(string $operation, string ...$extra_args): Relay|array|int|bool {}

    /**
     * Returns all the members of the set value stored at `$key`.
     *
     * @param  mixed  $set
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function smembers(mixed $set): Relay|array|false {}

    /**
     * Returns if `$member` is a member of the set stored at `$key`.
     *
     * @param  mixed  $set
     * @param  mixed  $member
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function sismember(mixed $set, mixed $member): Relay|bool {}

    /**
     * Returns whether each member is a member of the set stored at `$key`.
     *
     * @param  mixed  $set
     * @param  mixed  $members,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function smismember(mixed $set, mixed ...$members): Relay|array|false {}

    /**
     * Remove the specified members from the set stored at `$key`.
     *
     * @param  mixed  $set
     * @param  mixed  $member
     * @param  mixed  $members,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function srem(mixed $set, mixed $member, mixed ...$members): Relay|int|false {}

    /**
     * Add the specified members to the set stored at `$key`.
     *
     * @param  mixed  $set
     * @param  mixed  $member
     * @param  mixed  $members,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sadd(mixed $set, mixed $member, mixed ...$members): Relay|int|false {}

    /**
     * Sort the elements in a list, set or sorted set.
     *
     * @param  mixed  $key
     * @param  array  $options
     * @return Relay|array|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sort(mixed $key, array $options = []): Relay|array|int|false {}

    /**
     * Sort the elements in a list, set or sorted set. Read-only variant of SORT.
     *
     * @param  mixed  $key
     * @param  array  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function sort_ro(mixed $key, array $options = []): Relay|array|false {}

    /**
     * Move member from the set at source to the set at destination.
     *
     * @param  mixed  $srcset
     * @param  mixed  $dstset
     * @param  mixed  $member
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function smove(mixed $srcset, mixed $dstset, mixed $member): Relay|bool {}

    /**
     * Removes and returns one or more random members from the set value store at `$key`.
     *
     * @param  mixed  $set
     * @param  int  $count
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function spop(mixed $set, int $count = 1): mixed {}

    /**
     * Returns one or multiple random members from a set.
     *
     * @param  mixed  $set
     * @param  int  $count
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function srandmember(mixed $set, int $count = 1): mixed {}

    /**
     * Returns the set cardinality (number of elements) of the set stored at `$key`.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function scard(mixed $key): Relay|int|false {}

    /**
     * Execute a script management command.
     *
     * @param  string  $command
     * @param  string  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function script(string $command, string ...$args): mixed {}

    /**
     * Returns the length of the string value stored at `$key`.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function strlen(mixed $key): Relay|int|false {}

    /**
     * This command swaps two Redis databases,
     * so that immediately all the clients connected to a given database
     * will see the data of the other database, and the other way around.
     *
     * @param  int  $index1
     * @param  int  $index2
     * @return Relay|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function swapdb(int $index1, int $index2): Relay|bool {}

    /**
     * Returns the number of fields contained in the hash stored at `$key`.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function hlen(mixed $key): Relay|int|false {}

    /**
     * Returns the length of the list stored at `$key`.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function llen(mixed $key): Relay|int|false {}

    /**
     * Add an element to a vector set.
     *
     * @param  mixed  $key
     * @param  array  $values
     * @param  mixed  $element
     * @param  array|null  $options
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function vadd(mixed $key, array $values, mixed $element, ?array $options = null): Relay|int|false {}

    /**
     * Return the cardinality (number of elements) in a vector set.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function vcard(mixed $key): Relay|int|false {}

    /**
     * Return the dimensionality of vectors in a vector set.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function vdim(mixed $key): Relay|int|false {}

    /**
     * Get the embedding for a given vector set member.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @param  bool  $raw
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function vemb(mixed $key, mixed $element, bool $raw = false): Relay|array|false {}

    /**
     * Get any attributes for a given vector set member.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @param  bool  $raw
     * @return Relay|array|string|false
     */
    #[Attributes\RedisCommand]
    public function vgetattr(mixed $key, mixed $element, bool $raw = false): Relay|array|string|false {}

    /**
     * Return metadata about a vector set.
     *
     * @param  mixed  $key
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function vinfo(mixed $key): Relay|array|false {}

    /**
     * Returns whether or not the element is a member of a vectorset.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @return Relay|bool
     */
    #[Attributes\RedisCommand]
    public function vismember(mixed $key, mixed $element): Relay|bool {}

    /**
     * Get neighbors for a given vector element optionally withscores.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @param  bool  $withscores
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function vlinks(mixed $key, mixed $element, bool $withscores): Relay|array|false {}

    /**
     * Get one or more random members from a vector set.
     *
     * @param  mixed  $key
     * @param  int  $count
     * @return Relay|array|string|false
     */
    #[Attributes\RedisCommand]
    public function vrandmember(mixed $key, int $count = 0): Relay|array|string|false {}

    /**
     * Get a lexicographical range of elements from a vector set.
     *
     * @param  mixed  $key
     * @param  string  $min
     * @param  string  $max
     * @param  int  $count = 0
     */
    #[Attributes\RedisCommand]
    public function vrange(mixed $key, string $min, string $max, int $count = -1): Relay|array|false {}

    /**
     * Remove an element from a vector set.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function vrem(mixed $key, mixed $element): Relay|int|false {}

    /**
     * Set attributes for a given vector set member.
     *
     * @param  mixed  $key
     * @param  mixed  $element
     * @param  array|string  $attributes
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand]
    public function vsetattr(mixed $key, mixed $element, array|string $attributes): Relay|int|false {}

    /**
     * Do a similarity search on encodings or an element of a vector set.
     *
     * @param  mixed  $key
     * @param  mixed  $member
     * @param  array|null  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function vsim(mixed $key, mixed $member, array|null $options = null): Relay|array|false {}

    /**
     * Acknowledge one or more IDs as having been processed by the consumer group.
     *
     * @param  mixed  $key
     * @param  string  $group
     * @param  array  $ids
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xack(mixed $key, string $group, array $ids): Relay|int|false {}

    /**
     * Awknowledge and delete one or more IDs in a stream.
     *
     * @param  string  $key
     * @param  string  $group
     * @param  array  $ids
     * @param  string|null  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function xackdel(string $key, string $group, array $ids, ?string $mode = null): Relay|array|false {}

    /**
     * Append a message to a stream.
     *
     * @param  string  $key
     * @param  string  $id
     * @param  int  $maxlen
     * @param  bool  $approx
     * @param  bool  $nomkstream
     * @return Relay|string|null|false
     */
    public function xadd(
        string $key,
        string $id,
        array $values,
        int $maxlen = 0,
        bool $approx = false,
        bool $nomkstream = false
    ): Relay|string|null|false {}

    /**
     * Claim ownership of stream message(s).
     *
     * @param  string  $key
     * @param  string  $group
     * @param  string  $consumer
     * @param  int  $min_idle
     * @param  array  $ids
     * @param  array  $options
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xclaim(
        string $key,
        string $group,
        string $consumer,
        int $min_idle,
        array $ids,
        array $options
    ): Relay|array|bool {}

    /**
     * Automatically take ownership of stream message(s) by metrics.
     *
     * @param  string  $key
     * @param  string  $group
     * @param  string  $consumer
     * @param  int  $min_idle
     * @param  string  $start
     * @param  int  $count
     * @param  bool  $justid
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xautoclaim(
        string $key,
        string $group,
        string $consumer,
        int $min_idle,
        string $start,
        int $count = -1,
        bool $justid = false
    ): Relay|bool|array {}

    /**
     * Get the length of a stream.
     *
     * @param  string  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xlen(string $key): Relay|int|false {}

    /**
     * Perform utility operations having to do with consumer groups.
     *
     * @param  string  $operation
     * @param  mixed  $key
     * @param  string  $group
     * @param  string  $id_or_consumer
     * @param  bool  $mkstream
     * @param  int  $entries_read
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xgroup(
        string $operation,
        mixed $key = null,
        string $group = null,
        string $id_or_consumer = null,
        bool $mkstream = false,
        int $entries_read = -2
    ): mixed {}

    /**
     * Remove one or more specific IDs from a stream.
     *
     * @param  string  $key
     * @param  array  $ids
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xdel(string $key, array $ids): Relay|int|false {}

    /**
     * Remove one or more IDs from a stream with optional mode argument.
     *
     * @param  string  $key
     * @param  array  $ids
     * @param  string|null  $mode
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand]
    public function xdelex(string $key, array $ids, ?string $mode = null): Relay|array|false {}

    /**
     * Retrieve information about a stream key.
     *
     * @param  string  $operation
     * @param  string|null  $arg1
     * @param  string|null  $arg2
     * @param  int  $count
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xinfo(string $operation, ?string $arg1 = null, ?string $arg2 = null, int $count = -1): mixed {}

    /**
     * Query pending entries in a stream.
     *
     * @param  string  $key
     * @param  string  $group
     * @param  string|null  $start
     * @param  string|null  $end
     * @param  int  $count
     * @param  string|null  $consumer
     * @param  int  $idle
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xpending(
        string $key,
        string $group,
        ?string $start = null,
        ?string $end = null,
        int $count = -1,
        ?string $consumer = null,
        int $idle = 0
    ): Relay|array|false {}

    /**
     * Lists elements in a stream.
     *
     * @param  mixed  $key
     * @param  string  $start
     * @param  string  $end
     * @param  int  $count = -1
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xrange(mixed $key, string $start, string $end, int $count = -1): Relay|array|false {}

    /**
     * Get a range of entries from a STREAM ke in reverse chronological order.
     *
     * @param  string  $key
     * @param  string  $end
     * @param  string  $start
     * @param  int  $count
     * @return Relay|array|bool
     */
    #[Attributes\RedisCommand]
    public function xrevrange(string $key, string $end, string $start, int $count = -1): Relay|array|bool {}

    /**
     * Read messages from a stream.
     *
     * @param  array  $streams
     * @param  int  $count
     * @param  int  $block
     * @return Relay|array|bool|null
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xread(array $streams, int $count = -1, int $block = -1): Relay|array|bool|null {}

    /**
     * Read messages from a stream using a consumer group.
     *
     * @param  string  $group
     * @param  string  $consumer
     * @param  array  $streams
     * @param  int  $count
     * @param  int  $block
     * @return Relay|array|bool|null
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xreadgroup(
        string $group,
        string $consumer,
        array $streams,
        int $count = 1,
        int $block = 1
    ): Relay|array|bool|null {}

    /**
     * Truncate a STREAM key in various ways.
     *
     * @param  string  $key
     * @param  string  $threshold
     * @param  bool  $approx
     * @param  bool  $minid
     * @param  int  $limit
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function xtrim(
        string $key,
        string $threshold,
        bool $approx = false,
        bool $minid = false,
        int $limit = -1
    ): Relay|int|false {}

    /**
     * Adds all the specified members with the specified scores to the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $args,...
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zadd(mixed $key, mixed ...$args): mixed {}

    /**
     * When called with just the key argument, return a random element from the sorted set value stored at key.
     * If the provided count argument is positive, return an array of distinct elements.
     *
     * @param  mixed  $key
     * @param  array|null  $options
     * @return mixed
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zrandmember(mixed $key, ?array $options = null): mixed {}

    /**
     * Returns the specified range of elements in the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  string|int  $start
     * @param  string|int  $end
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function zrange(mixed $key, string|int $start, string|int $end, mixed $options = null): Relay|array|false {}

    /**
     * Returns the specified range of elements in the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $end
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached, Attributes\Deprecated]
    public function zrevrange(mixed $key, int $start, int $end, mixed $options = null): Relay|array|false {}

    /**
     * Returns all the elements in the sorted set at key with a score between
     * min and max (including elements with score equal to min or max).
     *
     * @param  mixed  $key
     * @param  mixed  $start
     * @param  mixed  $end
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached, Attributes\Deprecated]
    public function zrangebyscore(mixed $key, mixed $start, mixed $end, mixed $options = null): Relay|array|false {}

    /**
     * Returns all the elements in the sorted set at key with a score between
     * max and min (including elements with score equal to max or min).
     *
     * @param  mixed  $key
     * @param  mixed  $start
     * @param  mixed  $end
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached, Attributes\Deprecated]
    public function zrevrangebyscore(mixed $key, mixed $start, mixed $end, mixed $options = null): Relay|array|false {}

    /**
     * Returns all the elements in the sorted set at key with a score between
     * max and min (including elements with score equal to max or min).
     *
     * @param  mixed  $dst
     * @param  mixed  $src
     * @param  mixed  $start
     * @param  mixed  $end
     * @param  mixed  $options
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zrangestore(mixed $dst, mixed $src, mixed $start, mixed $end, mixed $options = null): Relay|int|false {}

    /**
     * When all the elements in a sorted set are inserted with the same score,
     * in order to force lexicographical ordering, this command returns all
     * the elements in the sorted set at key with a value between min and max.
     *
     * @param  mixed  $key
     * @param  mixed  $min
     * @param  mixed  $max
     * @param  int  $offset
     * @param  int  $count
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function zrangebylex(mixed $key, mixed $min, mixed $max, int $offset = -1, int $count = -1): Relay|array|false {}

    /**
     * When all the elements in a sorted set are inserted with the same score,
     * in order to force lexicographical ordering, this command returns all
     * the elements in the sorted set at key with a value between max and min.
     *
     * @param  mixed  $key
     * @param  mixed  $max
     * @param  mixed  $min
     * @param  int  $offset
     * @param  int  $count
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Deprecated]
    public function zrevrangebylex(mixed $key, mixed $max, mixed $min, int $offset = -1, int $count = -1): Relay|array|false {}

    /**
     * Returns the rank of member in the sorted set stored at key, with the scores
     * ordered from low to high. The rank (or index) is 0-based, which means
     * that the member with the lowest score has rank 0.
     *
     * @param  mixed  $key
     * @param  mixed  $rank
     * @param  bool  $withscore
     * @return Relay|array|int|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zrank(mixed $key, mixed $rank, bool $withscore = false): Relay|array|int|null|false {}

    /**
     * Returns the rank of member in the sorted set stored at key, with the scores
     * ordered from high to low. The rank (or index) is 0-based, which means
     * that the member with the highest score has rank 0.
     *
     * @param  mixed  $key
     * @param  mixed  $rank
     * @param  bool  $withscore
     * @return Relay|array|int|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zrevrank(mixed $key, mixed $rank, bool $withscore = false): Relay|array|int|null|false {}

    /**
     * Removes the specified members from the sorted set stored at key.
     * Non existing members are ignored.
     *
     * @param  mixed  $key
     * @param  mixed  $args,...
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zrem(mixed $key, mixed ...$args): Relay|int|false {}

    /**
     * When all the elements in a sorted set are inserted with the same score,
     * in order to force lexicographical ordering, this command removes all
     * elements in the sorted set stored at key between the
     * lexicographical range specified by min and max.
     *
     * @param  mixed  $key
     * @param  mixed  $min
     * @param  mixed  $max
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zremrangebylex(mixed $key, mixed $min, mixed $max): Relay|int|false {}

    /**
     * Removes all elements in the sorted set stored at key with rank between
     * start and stop. Both start and stop are 0 -based indexes with 0 being
     * the element with the lowest score.
     *
     * @param  mixed  $key
     * @param  int  $start
     * @param  int  $end
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zremrangebyrank(mixed $key, int $start, int $end): Relay|int|false {}

    /**
     * Removes all elements in the sorted set stored at key with
     * a score between min and max (inclusive).
     *
     * @param  mixed  $key
     * @param  mixed  $min
     * @param  mixed  $max
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zremrangebyscore(mixed $key, mixed $min, mixed $max): Relay|int|false {}

    /**
     * Returns the sorted set cardinality (number of elements) of the sorted set stored at key.
     *
     * @param  mixed  $key
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand, Attributes\Cached]
    public function zcard(mixed $key): Relay|int|false {}

    /**
     * Returns the number of elements in the sorted set at key with a score between min and max.
     *
     * @param  mixed  $key
     * @param  mixed  $min
     * @param  mixed  $max
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zcount(mixed $key, mixed $min, mixed $max): Relay|int|false {}

    /**
     * This command is similar to ZDIFFSTORE, but instead of storing the
     * resulting sorted set, it is returned to the client.
     *
     * @param  array  $keys
     * @param  array  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zdiff(array $keys, ?array $options = null): Relay|array|false {}

    /**
     * Computes the difference between the first and all successive
     * input sorted sets and stores the result in destination.
     *
     * @param  mixed  $dst
     * @param  array  $keys
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zdiffstore(mixed $dst, array $keys): Relay|int|false {}

    /**
     * Increments the score of member in the sorted set stored at key by increment.
     *
     * @param  mixed  $key
     * @param  float  $score
     * @param  mixed  $mem
     * @return Relay|float|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zincrby(mixed $key, float $score, mixed $mem): Relay|float|false {}

    /**
     * When all the elements in a sorted set are inserted with the same score,
     * in order to force lexicographical ordering, this command returns the
     * number of elements in the sorted set at key with a value between min and max.
     *
     * @param  mixed  $key
     * @param  mixed  $min
     * @param  mixed  $max
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zlexcount(mixed $key, mixed $min, mixed $max): Relay|int|false {}

    /**
     * Returns the scores associated with the specified members in the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  mixed  $mems,...
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zmscore(mixed $key, mixed ...$mems): Relay|array|false {}

    /**
     * Returns the score of member in the sorted set at key.
     *
     * @param  mixed  $key
     * @param  mixed  $member
     * @return Relay|float|null|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zscore(mixed $key, mixed $member): Relay|float|null|false {}

    /**
     * This command is similar to ZINTERSTORE, but instead of storing
     * the resulting sorted set, it is returned to the client.
     *
     * @param  array  $keys
     * @param  array  $weights
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zinter(array $keys, ?array $weights = null, mixed $options = null): Relay|array|false {}

    /**
     * Intersect multiple sorted sets and return the cardinality of the result.
     *
     * @param  array  $keys
     * @param  int  $limit
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zintercard(array $keys, int $limit = -1): Relay|int|false {}

    /**
     * Computes the intersection of numkeys sorted sets given by the
     * specified keys, and stores the result in destination.
     *
     * @param  mixed  $dst
     * @param  array  $keys
     * @param  array  $weights
     * @param  mixed  $options
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zinterstore(mixed $dst, array $keys, ?array $weights = null, mixed $options = null): Relay|int|false {}

    /**
     * This command is similar to ZUNIONSTORE, but instead of storing
     * the resulting sorted set, it is returned to the client.
     *
     * @param  array  $keys
     * @param  array  $weights
     * @param  mixed  $options
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zunion(array $keys, ?array $weights = null, mixed $options = null): Relay|array|false {}

    /**
     * Computes the union of numkeys sorted sets given by the
     * specified keys, and stores the result in destination.
     *
     * @param  mixed  $dst
     * @param  array  $keys
     * @param  array  $weights
     * @param  mixed  $options
     * @return Relay|int|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zunionstore(mixed $dst, array $keys, ?array $weights = null, mixed $options = null): Relay|int|false {}

    /**
     * Removes and returns up to count members with the lowest
     * scores in the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  int  $count
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zpopmin(mixed $key, int $count = 1): Relay|array|false {}

    /**
     * Removes and returns up to count members with the highest
     * scores in the sorted set stored at key.
     *
     * @param  mixed  $key
     * @param  int  $count
     * @return Relay|array|false
     */
    #[Attributes\RedisCommand, Attributes\ValkeyCommand]
    public function zpopmax(mixed $key, int $count = 1): Relay|array|false {}

    /**
     * Initialize a Redis CMS (Count-Min Sketch) by dimensions.
     *
     * @param  mixed  $key
     * @param  int  $width
     * @param  int  $depth
     * @return Relay|bool
     */
    #[Attributes\RedisCommand]
    public function cmsInitByDim(mixed $key, int $width, int $depth): Relay|bool {}

    /**
     * Initialize a Redis CMS (Count-Min Sketch) by desired probabilities.
     *
     * @param  mixed  $key
     * @param  float  $error
     * @param  float  $probability
     * @return Relay|bool
     */
    #[Attributes\RedisCommand]
    public function cmsInitByProb(mixed $key, float $error, float $probability): Relay|bool {}

    /**
     * Get information about a Count-Min Sketch key.
     *
     * @param  mixed  $key
     * @return Relay|array
     */
    #[Attributes\RedisCommand]
    public function cmsInfo(mixed $key): Relay|array|false {}

    /**
     * Increment one or more fields in a Count-Min Sketch key.
     *
     * @param  mixed  $key
     * @param  mixed  $field
     * @param  int  $value
     * @param  mixed  $fields_and_falues,...
     * @return Relay|array
     */
    #[Attributes\RedisCommand]
    public function cmsIncrBy(mixed $key, mixed $field, int $value, ...$fields_and_falues): Relay|array|false {}

    /**
     * Merge one or more Count-Min Sketch keys with optional weights.
     *
     * @param  mixed  $dstkey
     * @param  array  $keys
     * @param  array  $weights = []
     */
    public function cmsMerge(mixed $dstkey, array $keys, array $weights = []): Relay|bool {}

    /**
     * Query a Count-Min Sketch key.
     *
     * @param  mixed  $key
     * @param  mixed  $fields,...
     * @return Relay|array
     */
    #[Attributes\RedisCommand]
    public function cmsQuery(mixed $key, ...$fields): Relay|array|false {}

    /**
     * Returns keys cached in runtime memory.
     *
     * @internal Temporary debug helper. Do not use.
     * @return mixed
     */
    #[Attributes\Local]
    public function _getKeys() {}

    /**
     * Returns whether a key is tracked in memory.
     *
     * This can mean the entire key is cached in-memory or that we are tracking
     * the existence or length of the key.
     *
     * @param  string  $key
     * @return bool
     */
    #[Attributes\Local]
    public function isTracked(string $key): bool {}

    /**
     * Returns information about the license.
     *
     * @return array
     */
    #[Attributes\Local]
    public static function license(): array {}
}
