#!/bin/bash
# MotiveWave Java Launcher for Linux.
# This script combines the robust argument handling of the macOS script
# with Linux-specific path conventions and existing scaling detection.

# Exit immediately if a command exits with a non-zero status.
set -e

# --- 1. Resolve Base Directories ---
# This approach makes the script more portable if you move it within the app structure.
# SCRIPT_PATH: Resolves symlinks and gets the absolute path of this script.
# SCRIPT_DIR: The directory containing this script.
SCRIPT_PATH="$(readlink -f "$0")"
SCRIPT_DIR="$(dirname "$SCRIPT_PATH")"

# SCRIPTDIR: This is the assumed root installation directory of MotiveWave.
# Based on your previous script, it's explicitly set.
# If your script is *inside* a 'bin' folder (e.g., /usr/share/motivewave/bin/motivewave.sh),
# you might adjust SCRIPTDIR like this:
# SCRIPTDIR="$(dirname "$SCRIPT_DIR")"
# Otherwise, if your script is directly in the root (e.g., /usr/share/motivewave/motivewave.sh),
# then SCRIPTDIR is simply SCRIPT_DIR.
# For consistency with your original script, we'll keep it hardcoded for now,
# but the dynamic approach is below if you want to switch.
SCRIPTDIR="/usr/share/motivewave" # Your original hardcoded path

# If you prefer a dynamic SCRIPTDIR based on script location:
# if [[ "$SCRIPT_DIR" == */bin ]]; then
#    SCRIPTDIR="$(dirname "$SCRIPT_DIR")" # If script is in 'bin', base is its parent
# else
#    SCRIPTDIR="$SCRIPT_DIR"             # Otherwise, script dir is the base
# fi


# FX_DIR: Directory containing JavaFX modules.
# Ensure this path is correct for your MotiveWave installation.
# It's commonly "$SCRIPTDIR/javafx" or "$SCRIPTDIR/lib/javafx".
FX_DIR="$SCRIPTDIR/javafx"

# JAVA_JARS_DIR: Directory containing the main application JARs.
# This could be "$SCRIPTDIR/jar", "$SCRIPTDIR/Java", or "$SCRIPTDIR/lib".
JAVA_JARS_DIR="$SCRIPTDIR/jar"

# --- 2. Build Classpath from .jar files ---
CP=""
# Find all .jar files in the specified directory (non-recursive)
# Using read while loop with process substitution for safety with filenames containing spaces.
# This is more robust than the simple 'for entry' loop.
while IFS= read -r jar_file; do
    if [ -n "$CP" ]; then
        CP="$CP:$jar_file"
    else
        CP="$jar_file"
    fi
done < <(find "$JAVA_JARS_DIR" -maxdepth 1 -name "*.jar")


# --- 3. Initialize Variables from startup.ini (Improved Parsing) ---

MAX_HEAP=""
MAX_VRAM="2048M" # Default value
VM_ARGS=""       # Raw value string for splitting later
SCALE_FROM_INI="" # To store scale if read from startup.ini

# Path to the startup.ini file. This is a common Linux location for user config.
SETTINGS_FILE="$HOME/.motivewave/startup.ini"

if [ -f "$SETTINGS_FILE" ]; then
    echo "Reading Startup Options from: $SETTINGS_FILE"
    # Read the file line by line, ignoring comments and empty lines
    while IFS= read -r line; do
        # Trim leading/trailing whitespace and remove comments starting with '#'
        line=$(echo "$line" | sed 's/#.*//' | xargs)

        # Skip empty lines after trimming
        [[ -z "$line" ]] && continue

        # Skip lines that do not contain '='
        [[ ! "$line" =~ = ]] && continue

        # Split line into key and value at the first '='
        key="${line%%=*}"
        value="${line#*=}"

        # Trim key and value again
        key=$(echo "$key" | xargs)
        value=$(echo "$value" | xargs)

        case "$key" in
            MAX_HEAP)
                if [ -n "$value" ]; then
                    MAX_HEAP="-Xmx$value"
                fi
                ;;
            MAX_VRAM)
                if [ -n "$value" ]; then
                    MAX_VRAM="$value"
                fi
                ;;
            SCALE)
                if [ -n "$value" ]; then
                    SCALE_FROM_INI="$value" # Store it, will override auto-detected
                fi
                ;;
            VM_ARGS)
                # Store the raw value. We will split this into an array later for safety.
                VM_ARGS="$value"
                ;;
        esac
    done < "$SETTINGS_FILE"
fi

# --- 4. Auto-Detect Scaling (Retained from your original script) ---
# If SCALE was not set in startup.ini, proceed with auto-detection
if [ -z "$SCALE_FROM_INI" ]; then
    echo "Attempting to auto-detect display scaling..."
    SCALE=""
    # Try to get the scale setting for the gnome desktop
    # gsettings can return 0 if no explicit scaling is set or if not on GNOME.
    GNOME_SCALE_RAW=$(gsettings get org.gnome.desktop.interface scaling-factor 2>/dev/null | awk '{print $2}')

    # Validate if it's a number and not 0 or empty
    if [[ "$GNOME_SCALE_RAW" =~ ^[0-9]*\.?[0-9]*$ ]] && (( $(echo "$GNOME_SCALE_RAW > 0" | bc -l) )); then
        SCALE="$GNOME_SCALE_RAW"
        echo "Detected GNOME scaling factor: $SCALE"
    fi

    # If scaling is still not detected, try guessing from screen resolution
    if [[ -z "$SCALE" || "$SCALE" == "0" ]]; then
        echo "GNOME scaling not detected or is 0, attempting resolution-based guess..."
        H=$(xrandr --query | grep -A 1 "connected primary" | grep -v connected | awk '{print $1}' | awk -Fx '{print $2}')
        echo "Primary screen height (pixels): $H"
        SCALE=1
        if [[ -n "$H" && "$H" -gt 1800 ]]; then SCALE=2; fi
        if [[ -n "$H" && "$H" -gt 4000 ]]; then SCALE=3; fi
        echo "Guessed scaling factor based on resolution: $SCALE"
    fi

    # Convert this to a percentage for VM argument
    SCALE=$(echo "$SCALE*100" | bc)
    SCALE="${SCALE%.*}%"
else
    # Use scale from startup.ini if it was present
    SCALE="$SCALE_FROM_INI"
    echo "Using scaling from startup.ini: $SCALE"
fi

# Final checks for values
if [[ -z "$MAX_HEAP" ]]; then MAX_HEAP="-Xmx1G"; fi # Set a default if not found
if [[ -z "$MAX_VRAM" ]]; then MAX_VRAM="2048M"; fi

echo "Effective Scaling: $SCALE"
echo "Effective Max Heap: ${MAX_HEAP:-Not set}"
echo "Effective Max VRAM: ${MAX_VRAM:-Not set}"
echo "Effective VM Args: ${VM_ARGS:-Not set}"


# --- 5. Determine Java Binary (More Robust) ---
# Check for bundled JRE first, then system Java.
BUNDLED_JRE_PATH="$SCRIPTDIR/jre"
BUNDLED_JAVA_BIN="$BUNDLED_JRE_PATH/bin/motivewave"

if [ -x "$BUNDLED_JAVA_BIN" ]; then
    JAVA_BINARY="$BUNDLED_JAVA_BIN"
    echo "Using bundled Java: $JAVA_BINARY"
else
    echo "Bundled JRE not found or executable. Attempting to use system 'java'."
    JAVA_BINARY="$(which java)"
    if [ -z "$JAVA_BINARY" ]; then
        echo "Error: 'java' executable not found in system PATH. Please ensure Java (JDK 17+) is installed." >&2
        exit 1
    fi
    echo "Using system Java: $JAVA_BINARY"
fi

# Verify the Java binary exists and is executable
if [ ! -x "$JAVA_BINARY" ]; then
    echo "Error: Java binary not found or not executable at $JAVA_BINARY" >&2
    exit 1
fi

# --- 6. Build the Java Command Array (Safe and Robust) ---
# Using an array ensures that arguments with spaces are handled correctly.
COMMAND=()

# Add the Java binary
COMMAND+=("$JAVA_BINARY")

# Add MAX_HEAP if set
if [ -n "$MAX_HEAP" ]; then
    COMMAND+=("$MAX_HEAP")
fi

# Add VM_ARGS, splitting the string into separate array elements.
# This handles multiple custom VM args defined in startup.ini gracefully.
if [ -n "$VM_ARGS" ]; then
    read -ra vm_args_array <<< "$VM_ARGS"
    COMMAND+=("${vm_args_array[@]}")
fi

# Add standard JVM arguments
# Removed -Djdk.gtk.version=2 as it's often not needed with modern JavaFX/GTK3
# and can cause issues. If you have problems, add it back.
COMMAND+=("-Dprism.forceGPU=true")
COMMAND+=("-Dsun.java2d.opengl=true")
COMMAND+=("-Dprism.order=es2,es1,sw,j2d")
COMMAND+=("-Dsun.java2d.pmoffscreen=false")
COMMAND+=("-Dprism.vsync=false")
COMMAND+=("-Dprism.lcdtext=false")
COMMAND+=("-Dawt.useSystemAAFontSettings=false")

# MotiveWave specific arguments
COMMAND+=("-javaagent:$SCRIPTDIR/jar/MotiveWave.jar")
COMMAND+=("-Dname=MotiveWave")
COMMAND+=("-Djava.library.path=$SCRIPTDIR/lib") # Ensure this path is correct for native libs if any

# Directory properties
# Only include the essential ones. macOS specific ones are removed.
COMMAND+=("-DUserHome=$HOME")
COMMAND+=("-Duser.dir=$HOME") # Sets current working directory for the JVM process
COMMAND+=("-DappDir=$SCRIPTDIR")
COMMAND+=("-DDocumentsDirectory=$HOME/Documents")
COMMAND+=("-DDesktopDirectory=$HOME/Desktop")
COMMAND+=("-DDownloadsDirectory=$HOME/Downloads")
COMMAND+=("-DSharedPublicDirectory=$HOME/Public") # Public folder

# JavaFX related properties
COMMAND+=("-Dprism.verbose=true")
COMMAND+=("-Dprism.maxvram=$MAX_VRAM")
COMMAND+=("-Dglass.gtk.uiScale=$SCALE") # For GTK scaling
COMMAND+=("-Dsun.java2d.uiScale=$SCALE") # For AWT/Swing scaling

# JavaFX module path
COMMAND+=("-p")
COMMAND+=("$FX_DIR")

# JavaFX Modules and Exports/Opens arguments - These are cross-platform.
COMMAND+=("--add-modules=javafx.controls,javafx.base,javafx.graphics,javafx.media,javafx.web,javafx.swing")
COMMAND+=("--add-exports=javafx.web/com.sun.webkit.network=ALL-UNNAMED")
COMMAND+=("--add-opens=javafx.graphics/javafx.stage=ALL-UNNAMED")
COMMAND+=("--add-opens=javafx.graphics/javafx.scene=ALL-UNNAMED")
COMMAND+=("--add-exports=javafx.graphics/com.sun.javafx.tk.quantum=ALL-UNNAMED")
COMMAND+=("--add-exports=javafx.base/com.sun.javafx.event=ALL-UNNAMED")

# Classpath
COMMAND+=("-classpath")
COMMAND+=("$CP")

# The main class/module to launch
COMMAND+=("MotiveWave")

echo "Launching MotiveWave with command:"
# Print the command array, properly quoted for display using printf %q
printf "%q " "${COMMAND[@]}"; echo
echo # Add a newline for readability

# --- 7. Launch MotiveWave ---
echo "--- MotiveWave Process Output ---"
# Execute the command array. 'exec' replaces the current bash process.
exec "${COMMAND[@]}"
