"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rule = exports.name = void 0;
const icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
const context_compat_1 = require("../context-compat");
const util_1 = require("../util");
exports.name = 'no-camel-case';
const CAMEL_CASE_REGEX = /[A-Z]/;
function verifyAst(ast) {
    const errors = [];
    for (const el of ast) {
        if ((0, icu_messageformat_parser_1.isArgumentElement)(el)) {
            if (CAMEL_CASE_REGEX.test(el.value)) {
                errors.push({ messageId: 'camelcase', data: {} });
            }
            continue;
        }
        if ((0, icu_messageformat_parser_1.isPluralElement)(el)) {
            if (CAMEL_CASE_REGEX.test(el.value)) {
                errors.push({ messageId: 'camelcase', data: {} });
            }
            const { options } = el;
            for (const selector of Object.keys(options)) {
                errors.push(...verifyAst(options[selector].value));
            }
        }
    }
    return errors;
}
function checkNode(context, node) {
    const settings = (0, util_1.getSettings)(context);
    const msgs = (0, util_1.extractMessages)(node, settings);
    for (const [{ message: { defaultMessage }, messageNode, },] of msgs) {
        if (!defaultMessage || !messageNode) {
            continue;
        }
        const errors = verifyAst((0, icu_messageformat_parser_1.parse)(defaultMessage, {
            ignoreTag: settings.ignoreTag,
        }));
        for (const error of errors) {
            context.report({
                node: messageNode,
                ...error,
            });
        }
    }
}
exports.rule = {
    meta: {
        type: 'problem',
        docs: {
            description: 'Disallow camel case placeholders in message',
            url: 'https://formatjs.github.io/docs/tooling/linter#no-camel-case',
        },
        fixable: 'code',
        schema: [],
        messages: {
            camelcase: 'Camel case arguments are not allowed',
        },
    },
    defaultOptions: [],
    create(context) {
        const callExpressionVisitor = (node) => checkNode(context, node);
        const parserServices = (0, context_compat_1.getParserServices)(context);
        //@ts-expect-error defineTemplateBodyVisitor exists in Vue parser
        if (parserServices?.defineTemplateBodyVisitor) {
            //@ts-expect-error
            return parserServices.defineTemplateBodyVisitor({
                CallExpression: callExpressionVisitor,
            }, {
                CallExpression: callExpressionVisitor,
            });
        }
        return {
            JSXOpeningElement: (node) => checkNode(context, node),
            CallExpression: callExpressionVisitor,
        };
    },
};
