"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rule = exports.name = void 0;
const unicode_emoji_utils_1 = require("unicode-emoji-utils");
const context_compat_1 = require("../context-compat");
const util_1 = require("../util");
exports.name = 'no-emoji';
function checkNode(context, node) {
    const msgs = (0, util_1.extractMessages)(node, (0, util_1.getSettings)(context));
    let allowedEmojis = [];
    let versionAbove;
    const [emojiConfig] = context.options;
    if (emojiConfig?.versionAbove &&
        (0, unicode_emoji_utils_1.isValidEmojiVersion)(emojiConfig.versionAbove) &&
        !versionAbove &&
        allowedEmojis.length === 0) {
        versionAbove = emojiConfig.versionAbove;
        allowedEmojis = (0, unicode_emoji_utils_1.getAllEmojis)((0, unicode_emoji_utils_1.filterEmojis)(versionAbove));
    }
    for (const [{ message: { defaultMessage }, messageNode, },] of msgs) {
        if (!defaultMessage || !messageNode) {
            continue;
        }
        if ((0, unicode_emoji_utils_1.hasEmoji)(defaultMessage)) {
            if (versionAbove) {
                for (const emoji of (0, unicode_emoji_utils_1.extractEmojis)(defaultMessage)) {
                    if (!allowedEmojis.includes(emoji)) {
                        context.report({
                            node: messageNode,
                            messageId: 'notAllowedAboveVersion',
                            data: {
                                versionAbove,
                                emoji,
                            },
                        });
                    }
                }
            }
            else {
                context.report({
                    node: messageNode,
                    messageId: 'notAllowed',
                });
            }
        }
    }
}
const versionAboveEnums = [
    '0.6',
    '0.7',
    '1.0',
    '2.0',
    '3.0',
    '4.0',
    '5.0',
    '11.0',
    '12.0',
    '12.1',
    '13.0',
    '13.1',
    '14.0',
    '15.0',
];
exports.rule = {
    meta: {
        type: 'problem',
        docs: {
            description: 'Disallow emojis in message',
            url: 'https://formatjs.github.io/docs/tooling/linter#no-emoji',
        },
        fixable: 'code',
        schema: [
            {
                type: 'object',
                properties: { versionAbove: { type: 'string', enum: versionAboveEnums } },
                additionalProperties: false,
            },
        ],
        messages: {
            notAllowed: 'Emojis are not allowed',
            notAllowedAboveVersion: 'Emojis above version {{versionAbove}} are not allowed - Emoji: {{emoji}}',
        },
    },
    defaultOptions: [],
    create(context) {
        const callExpressionVisitor = (node) => checkNode(context, node);
        const parserServices = (0, context_compat_1.getParserServices)(context);
        //@ts-expect-error defineTemplateBodyVisitor exists in Vue parser
        if (parserServices?.defineTemplateBodyVisitor) {
            //@ts-expect-error
            return parserServices.defineTemplateBodyVisitor({
                CallExpression: callExpressionVisitor,
            }, {
                CallExpression: callExpressionVisitor,
            });
        }
        return {
            JSXOpeningElement: (node) => checkNode(context, node),
            CallExpression: callExpressionVisitor,
        };
    },
};
