"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.visitor = void 0;
const tslib_1 = require("tslib");
const t = tslib_1.__importStar(require("@babel/types"));
const utils_1 = require("../utils");
const icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
function assertObjectExpression(path, callee) {
    if (!path || !path.isObjectExpression()) {
        throw path.buildCodeFrameError(`[React Intl] \`${callee.get('property').node.name}()\` must be called with an object expression with values that are React Intl Message Descriptors, also defined as object expressions.`);
    }
}
function isFormatMessageCall(callee, functionNames) {
    if (functionNames.find(name => callee.isIdentifier({ name }))) {
        return true;
    }
    if (callee.isMemberExpression()) {
        const property = callee.get('property');
        return !!functionNames.find(name => property.isIdentifier({ name }));
    }
    return false;
}
function getMessagesObjectFromExpression(nodePath) {
    let currentPath = nodePath;
    while (t.isTSAsExpression(currentPath.node) ||
        t.isTSTypeAssertion(currentPath.node) ||
        t.isTypeCastExpression(currentPath.node)) {
        currentPath = currentPath.get('expression');
    }
    return currentPath;
}
const visitor = function (path, { opts, file: { opts: { filename }, }, }) {
    const { overrideIdFn, idInterpolationPattern, removeDefaultMessage, ast, preserveWhitespace, } = opts;
    if ((0, utils_1.wasExtracted)(path)) {
        return;
    }
    const { messages, functionNames } = this;
    const callee = path.get('callee');
    const args = path.get('arguments');
    /**
     * Process MessageDescriptor
     * @param messageDescriptor Message Descriptor
     */
    function processMessageObject(messageDescriptor) {
        assertObjectExpression(messageDescriptor, callee);
        const properties = messageDescriptor.get('properties');
        const descriptorPath = (0, utils_1.createMessageDescriptor)(properties.map(prop => [prop.get('key'), prop.get('value')]));
        // If the message is already compiled, don't re-compile it
        if (descriptorPath.defaultMessage?.isArrayExpression()) {
            return;
        }
        // Evaluate the Message Descriptor values, then store it.
        const descriptor = (0, utils_1.evaluateMessageDescriptor)(descriptorPath, false, filename || undefined, idInterpolationPattern, overrideIdFn, preserveWhitespace);
        (0, utils_1.storeMessage)(descriptor, messageDescriptor, opts, filename || undefined, messages);
        const firstProp = properties[0];
        const defaultMessageProp = properties.find(prop => {
            const keyProp = prop.get('key');
            return (keyProp.isIdentifier({ name: 'defaultMessage' }) ||
                keyProp.isStringLiteral({ value: 'defaultMessage' }));
        });
        const idProp = properties.find(prop => {
            const keyProp = prop.get('key');
            return (keyProp.isIdentifier({ name: 'id' }) ||
                keyProp.isStringLiteral({ value: 'id' }));
        });
        // Insert ID potentially 1st before removing nodes
        if (idProp) {
            idProp.get('value').replaceWith(t.stringLiteral(descriptor.id));
        }
        else {
            firstProp.insertBefore(t.objectProperty(t.identifier('id'), t.stringLiteral(descriptor.id)));
        }
        // Remove description
        properties
            .find(prop => {
            const keyProp = prop.get('key');
            return (keyProp.isIdentifier({ name: 'description' }) ||
                keyProp.isStringLiteral({ value: 'description' }));
        })
            ?.remove();
        // Pre-parse or remove defaultMessage
        if (defaultMessageProp) {
            if (removeDefaultMessage) {
                defaultMessageProp?.remove();
            }
            else if (descriptor.defaultMessage) {
                const valueProp = defaultMessageProp.get('value');
                if (ast) {
                    valueProp.replaceWithSourceString(JSON.stringify((0, icu_messageformat_parser_1.parse)(descriptor.defaultMessage)));
                }
                else {
                    valueProp.replaceWith(t.stringLiteral(descriptor.defaultMessage));
                }
            }
        }
        (0, utils_1.tagAsExtracted)(path);
    }
    // Check that this is `defineMessages` call
    if (callee.isIdentifier({ name: 'defineMessages' }) ||
        callee.isIdentifier({ name: 'defineMessage' })) {
        const firstArgument = args[0];
        const messagesObj = getMessagesObjectFromExpression(firstArgument);
        assertObjectExpression(messagesObj, callee);
        if (callee.isIdentifier({ name: 'defineMessage' })) {
            processMessageObject(messagesObj);
        }
        else {
            const properties = messagesObj.get('properties');
            if (Array.isArray(properties)) {
                properties
                    .map(prop => prop.get('value'))
                    .forEach(processMessageObject);
            }
        }
    }
    // Check that this is `intl.formatMessage` call
    if (isFormatMessageCall(callee, functionNames)) {
        const messageDescriptor = args[0];
        if (messageDescriptor && messageDescriptor.isObjectExpression()) {
            processMessageObject(messageDescriptor);
        }
    }
};
exports.visitor = visitor;
