// src/index.ts
import { brotliCompressSync } from "zlib";
import fullReloadPlugin from "vite-plugin-full-reload";
import manifestSRI from "vite-plugin-manifest-sri";
import ruby from "vite-plugin-ruby";
import environment from "vite-plugin-environment";
import stimulusHMRPlugin from "vite-plugin-stimulus-hmr";
import gzipPlugin from "rollup-plugin-gzip";
var defaultReloadPaths = [
  "config/routes.rb",
  "config/routes/*.rb",
  "app/views/**/*"
];
function ViteRubyPlugin(options = {}) {
  let { compress = true } = options;
  if (compress !== false)
    compress = { gzip: true, brotli: true, ...compress === true ? {} : compress };
  const envOptions = { defineOn: "import.meta.env", ...options.envOptions };
  const { overridePaths, additionalPaths = [], ...reloadOptions } = options.fullReload || {};
  const reloadPaths = overridePaths || [...defaultReloadPaths, ...wrapArray(additionalPaths)];
  const pluginOptions = [
    // Convention over configuration for Ruby projects using Vite.
    ruby(),
    // Expose environment variables to your client code.
    options.envVars && environment(options.envVars, envOptions),
    // Subresource Integrity for JavaScript and CSS assets.
    options.sri !== false && manifestSRI(options.sri),
    // See changes to your controllers instantly without refreshing the page.
    options.stimulus !== false && stimulusHMRPlugin(options.stimulus),
    // Automatically reload the page when making changes to server files.
    options.fullReload !== false && fullReloadPlugin(reloadPaths, reloadOptions),
    // Create gzip copies of relevant assets.
    compress && compress.gzip && gzipPlugin(options.compression),
    // Create brotli copies of relevant assets.
    compress && compress.brotli && gzipPlugin({
      ...options.compression,
      customCompression: (content) => brotliCompressSync(Buffer.from(content)),
      fileName: ".br"
    })
  ];
  const plugins = pluginOptions.flat(Infinity).filter((plugin) => plugin);
  return [...plugins, dupDetector(plugins.map((plugin) => plugin.name))];
}
function wrapArray(array) {
  return Array.isArray(array) ? array : [array];
}
function dupDetector(originalPluginNames) {
  return {
    name: "vite-plugin-rails:dup-detector",
    configResolved(config) {
      if (config.plugins) {
        const pluginNames = new Set(originalPluginNames);
        pluginNames.delete("gzip");
        const pluginCounts = /* @__PURE__ */ Object.create(null);
        config.plugins.filter((plugin) => pluginNames.has(plugin.name)).forEach((plugin) => {
          pluginCounts[plugin.name] = (pluginCounts[plugin.name] || 0) + 1;
        });
        const duplicates = Object.entries(pluginCounts).filter(([name, count]) => count > 1).map(([name, count]) => name);
        if (duplicates.length) {
          throw new Error(`
            [vite-plugin-rails] Duplicate plugins detected: ${duplicates.join(", ")}.

            If migrating from vite-plugin-ruby, make sure to remove these manually added
            plugins, and instead pass the options to vite-plugin-rails.
          `);
        }
      }
    }
  };
}
export {
  ViteRubyPlugin as default
};
