"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.rule = exports.name = void 0;
const icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
const context_compat_1 = require("../context-compat");
const util_1 = require("../util");
function verifyAst(ast) {
    if (ast.length !== 1) {
        return;
    }
    switch (ast[0].type) {
        case icu_messageformat_parser_1.TYPE.argument:
            return 'unnecessaryFormat';
        case icu_messageformat_parser_1.TYPE.number:
            return 'unnecessaryFormatNumber';
        case icu_messageformat_parser_1.TYPE.date:
            return 'unnecessaryFormatDate';
        case icu_messageformat_parser_1.TYPE.time:
            return 'unnecessaryFormatTime';
    }
}
function checkNode(context, node) {
    const settings = (0, util_1.getSettings)(context);
    const msgs = (0, util_1.extractMessages)(node, settings);
    for (const [{ message: { defaultMessage }, messageNode, },] of msgs) {
        if (!defaultMessage || !messageNode) {
            continue;
        }
        const messageId = verifyAst((0, icu_messageformat_parser_1.parse)(defaultMessage, {
            ignoreTag: settings.ignoreTag,
        }));
        if (messageId)
            context.report({
                node: messageNode,
                messageId,
            });
    }
}
exports.name = 'no-useless-message';
exports.rule = {
    meta: {
        type: 'problem',
        docs: {
            description: 'Disallow unnecessary formatted message',
            url: 'https://formatjs.github.io/docs/tooling/linter#no-useless-message',
        },
        fixable: 'code',
        schema: [],
        messages: {
            unnecessaryFormat: 'Unnecessary formatted message.',
            unnecessaryFormatNumber: 'Unnecessary formatted message: just use FormattedNumber or intl.formatNumber.',
            unnecessaryFormatDate: 'Unnecessary formatted message: just use FormattedDate or intl.formatDate.',
            unnecessaryFormatTime: 'Unnecessary formatted message: just use FormattedTime or intl.formatTime.',
        },
    },
    defaultOptions: [],
    create(context) {
        const callExpressionVisitor = (node) => checkNode(context, node);
        const parserServices = (0, context_compat_1.getParserServices)(context);
        //@ts-expect-error defineTemplateBodyVisitor exists in Vue parser
        if (parserServices?.defineTemplateBodyVisitor) {
            //@ts-expect-error
            return parserServices.defineTemplateBodyVisitor({
                CallExpression: callExpressionVisitor,
            }, {
                CallExpression: callExpressionVisitor,
            });
        }
        return {
            JSXOpeningElement: (node) => checkNode(context, node),
            CallExpression: callExpressionVisitor,
        };
    },
};
