# frozen_string_literal: true

class StatusCacheHydrator
  def initialize(status)
    @status = status
  end

  def hydrate(account_id, nested: false)
    # The cache of the serialized hash is generated by the fan-out-on-write service
    payload = Rails.cache.fetch("fan-out/#{@status.id}") { InlineRenderer.render(@status, nil, :status) }

    # If we're delivering to the author who disabled the display of the application used to create the
    # status, we need to hydrate the application, since it was not rendered for the basic payload
    payload[:application] = payload_application if payload[:application].nil? && @status.account_id == account_id

    # We take advantage of the fact that some relationships can only occur with an original status, not
    # the reblog that wraps it, so we can assume that some values are always false
    if payload[:reblog]
      hydrate_reblog_payload(payload, account_id, nested:)
    else
      hydrate_non_reblog_payload(payload, account_id, nested:)
    end
  end

  private

  def hydrate_non_reblog_payload(empty_payload, account_id, nested: false)
    empty_payload.tap do |payload|
      fill_status_payload(payload, @status, account_id, nested:)

      if payload[:poll]
        payload[:poll][:voted] = @status.account_id == account_id
        payload[:poll][:own_votes] = []
      end
    end
  end

  def hydrate_reblog_payload(empty_payload, account_id, nested: false)
    empty_payload.tap do |payload|
      payload[:muted]      = false
      payload[:bookmarked] = false
      payload[:pinned]     = false if @status.account_id == account_id

      # If the reblogged status is being delivered to the author who disabled the display of the application
      # used to create the status, we need to hydrate it here too
      payload[:reblog][:application] = payload_reblog_application if payload[:reblog][:application].nil? && @status.reblog.account_id == account_id

      fill_status_payload(payload[:reblog], @status.reblog, account_id, nested:)

      if payload[:reblog][:poll]
        if @status.reblog.account_id == account_id
          payload[:reblog][:poll][:voted] = true
          payload[:reblog][:poll][:own_votes] = []
        else
          own_votes = PollVote.where(poll_id: @status.reblog.poll_id, account_id: account_id).pluck(:choice)
          payload[:reblog][:poll][:voted] = !own_votes.empty?
          payload[:reblog][:poll][:own_votes] = own_votes
        end
      end

      payload[:filtered]   = payload[:reblog][:filtered]
      payload[:favourited] = payload[:reblog][:favourited]
      payload[:reblogged]  = payload[:reblog][:reblogged]
    end
  end

  def fill_status_payload(payload, status, account_id, nested: false)
    payload[:favourited] = Favourite.exists?(account_id: account_id, status_id: status.id)
    payload[:reblogged]  = Status.exists?(account_id: account_id, reblog_of_id: status.id)
    payload[:muted]      = ConversationMute.exists?(account_id: account_id, conversation_id: status.conversation_id)
    payload[:bookmarked] = Bookmark.exists?(account_id: account_id, status_id: status.id)
    payload[:pinned]     = StatusPin.exists?(account_id: account_id, status_id: status.id) if status.account_id == account_id
    payload[:filtered]   = mapped_applied_custom_filter(account_id, status)
    payload[:quote] = hydrate_quote_payload(payload[:quote], status.quote, account_id, nested:) if payload[:quote]
  end

  def hydrate_quote_payload(empty_payload, quote, account_id, nested: false)
    return unless quote&.acceptable?

    empty_payload.tap do |payload|
      payload.delete(:quoted_status) if nested

      # TODO: performance improvements
      if quote.accepted?
        if quote.quoted_status.nil?
          payload[nested ? :quoted_status_id : :quoted_status] = nil
          payload[:state] = 'deleted'
        elsif StatusFilter.new(quote.quoted_status, Account.find_by(id: account_id)).filtered_for_quote?
          payload[nested ? :quoted_status_id : :quoted_status] = nil
          payload[:state] = 'unauthorized'
        else
          payload[:state] = 'accepted'
          if nested
            payload[:quoted_status_id] = quote.quoted_status_id&.to_s
          else
            payload[:quoted_status] = StatusCacheHydrator.new(quote.quoted_status).hydrate(account_id, nested: true)
          end
        end
      else
        payload[nested ? :quoted_status_id : :quoted_status] = nil
      end
    end
  end

  def mapped_applied_custom_filter(account_id, status)
    CustomFilter
      .apply_cached_filters(CustomFilter.cached_filters_for(account_id), status)
      .map { |filter| serialized_filter(filter) }
  end

  def serialized_filter(filter)
    ActiveModelSerializers::SerializableResource.new(
      filter,
      serializer: REST::FilterResultSerializer
    ).as_json
  end

  def payload_application
    @status.application.present? ? serialized_status_application_json : nil
  end

  def serialized_status_application_json
    ActiveModelSerializers::SerializableResource.new(
      @status.application,
      serializer: REST::StatusSerializer::ApplicationSerializer
    ).as_json
  end

  def payload_reblog_application
    @status.reblog.application.present? ? serialized_status_reblog_application_json : nil
  end

  def serialized_status_reblog_application_json
    ActiveModelSerializers::SerializableResource.new(
      @status.reblog.application,
      serializer: REST::StatusSerializer::ApplicationSerializer
    ).as_json
  end
end
