/*
 * @(#)ImageProducer.java	1.20 06/10/10
 *
 * Copyright  1990-2008 Sun Microsystems, Inc. All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 only, as published by the Free Software Foundation. 
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License version 2 for more details (a copy is
 * included at /legal/license.txt). 
 * 
 * You should have received a copy of the GNU General Public License
 * version 2 along with this work; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA 
 * 
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 or visit www.sun.com if you need additional
 * information or have any questions. 
 *
 */

package jail.java.awt.image;

/**
 * The interface for objects which can produce the image data for Images. Each
 * image contains an ImageProducer which is used to reconstruct the image
 * whenever it is needed, for example, when a new size of the Image is scaled,
 * or when the width or height of the Image is being requested.
 *
 * @see ImageConsumer
 *
 * @version 1.16 08/19/02
 * @author Jim Graham
 */
public interface ImageProducer {
    /**
     * This method is used to register an ImageConsumer with the ImageProducer
     * for access to the image data during a later reconstruction of the Image.
     * The ImageProducer may, at its discretion, start delivering the image data
     * to the consumer using the ImageConsumer interface immediately, or when
     * the next available image reconstruction is triggered by a call to the
     * startProduction method.
     * 
     * @see #startProduction
     */
    public void addConsumer(ImageConsumer ic);

    /**
     * This method determines if a given ImageConsumer object is currently
     * registered with this ImageProducer as one of its consumers.
     */
    public boolean isConsumer(ImageConsumer ic);

    /**
     * This method removes the given ImageConsumer object from the list of
     * consumers currently registered to receive image data. It is not
     * considered an error to remove a consumer that is not currently
     * registered. The ImageProducer should stop sending data to this consumer
     * as soon as is feasible.
     */
    public void removeConsumer(ImageConsumer ic);

    /**
     * This method both registers the given ImageConsumer object as a consumer
     * and starts an immediate reconstruction of the image data which will then
     * be delivered to this consumer and any other consumer which may have
     * already been registered with the producer. This method differs from the
     * addConsumer method in that a reproduction of the image data should be
     * triggered as soon as possible.
     * 
     * @see #addConsumer
     */
    public void startProduction(ImageConsumer ic);

    /**
     * This method is used by an ImageConsumer to request that the ImageProducer
     * attempt to resend the image data one more time in TOPDOWNLEFTRIGHT order
     * so that higher quality conversion algorithms which depend on receiving
     * pixels in order can be used to produce a better output version of the
     * image. The ImageProducer is free to ignore this call if it cannot resend
     * the data in that order. If the data can be resent, then the ImageProducer
     * should respond by executing the following minimum set of ImageConsumer
     * method calls:
     * 
     * <pre>
     *	ic.setHints(TOPDOWNLEFTRIGHT | < otherhints >);
     *	ic.setPixels(...);	// As many times as needed
     *	ic.imageComplete();
     * </pre>
     * 
     * @see ImageConsumer#setHints
     */
    public void requestTopDownLeftRightResend(ImageConsumer ic);
}
