/*
 * Decompiled with CFR 0.152.
 */
package org.limine.snapper.processes;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.SecureRandom;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Scanner;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.limine.snapper.formats.limine8.LimineKey;
import org.limine.snapper.objects.Config;
import org.limine.snapper.objects.ConsoleColor;
import org.limine.snapper.objects.Hash;
import org.limine.snapper.objects.LogLevel;
import org.limine.snapper.objects.Output;

public class Utility {
    private static final Scanner INPUT_SCANNER = new Scanner(System.in);
    private static final int MACHINE_ID_LENGTH = 32;
    private static final String MACHINE_ID_FILE_PATH = "/etc/machine-id";
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter ISO_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss");
    private static final Pattern FORBIDDEN_PATTERN = Pattern.compile("\\x1B\\[[A-Za-z]|\n");

    public static boolean areYouInBtrfs() {
        Output output = Utility.getTextFromCommand("findmnt --mountpoint / -no FSTYPE", false, "");
        if (output.isSuccess()) {
            return output.text().get(0).toLowerCase().contains("btrfs");
        }
        String errMessage = "Command line fails: 'findmnt --mountpoint / -no FSTYPE'";
        Utility.errorMessage(errMessage);
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean areYouInSnapshot() {
        String snapshotPattern = "subvol=.*?/([0-9]+)/snapshot";
        try (BufferedReader reader = new BufferedReader(new FileReader("/proc/cmdline"));){
            String line = reader.readLine();
            if (line == null) return false;
            Pattern pattern = Pattern.compile(snapshotPattern);
            Matcher matcher = pattern.matcher(line);
            boolean bl = matcher.find();
            return bl;
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return false;
    }

    public static boolean askUser(String ask) {
        boolean isYes = false;
        while (true) {
            System.out.print(ask + String.valueOf((Object)ConsoleColor.YELLOW) + "\nEnter your choice: " + String.valueOf((Object)ConsoleColor.RESET));
            String userInput = INPUT_SCANNER.nextLine();
            if (userInput.trim().isEmpty()) {
                System.err.println(String.valueOf((Object)ConsoleColor.RED) + "The input should not be empty." + String.valueOf((Object)ConsoleColor.RESET) + userInput);
                continue;
            }
            if ("yes".startsWith(userInput.toLowerCase())) {
                isYes = true;
                break;
            }
            if ("no".startsWith(userInput.toLowerCase())) break;
            if ("cancel".startsWith(userInput.toLowerCase())) {
                System.out.println("The process is aborted.");
                System.exit(1);
                isYes = false;
                continue;
            }
            System.err.println(String.valueOf((Object)ConsoleColor.RED) + "The unknown input: " + String.valueOf((Object)ConsoleColor.RESET) + userInput);
        }
        return isYes;
    }

    public static String buildPath(String ... parts) {
        String path = Paths.get("", parts).toString();
        if (path.startsWith(File.separator)) {
            return path;
        }
        return File.separator + path;
    }

    public static String calculateHash(String absoluteFilePath) {
        return Utility.fileHashIntegrityCheck(absoluteFilePath, Config.HASH_FUNCTION, 2);
    }

    private static String calculateHash(String absoluteFilePath, Hash hashFunction) {
        Output output = Utility.getTextFromCommand(hashFunction.command + " " + absoluteFilePath, false, "Hash calculation failed, because the file: " + absoluteFilePath + " does not exist.");
        if (output.isSuccess()) {
            return Utility.extractHash(output.text().get(0));
        }
        return "is_not_available";
    }

    public static String fileHashIntegrityCheck(String absoluteFilePath, Hash hashFunction, int repeatIfCorrupted) {
        for (int attempts = 0; attempts <= repeatIfCorrupted; ++attempts) {
            try {
                CompletableFuture<String> hashFuture1 = CompletableFuture.supplyAsync(() -> Utility.calculateHash(absoluteFilePath, hashFunction));
                CompletableFuture<String> hashFuture2 = CompletableFuture.supplyAsync(() -> Utility.calculateHash(absoluteFilePath, hashFunction));
                String hashResult1 = hashFuture1.get();
                String hashResult2 = hashFuture2.get();
                if (hashResult1.equals(hashResult2)) {
                    return hashResult1;
                }
                String errMessage = String.format("Hash mismatch detected! This is a hardware issue (RAM or CPU is faulty). File: %s (Attempt %d/%d).", absoluteFilePath, attempts + 1, repeatIfCorrupted + 1);
                Utility.errorMessage(errMessage);
                if (attempts != repeatIfCorrupted) continue;
                errMessage = "Max retry attempts reached. Returning a corrupted hash. Do NOT trust this system and hardware!";
                Utility.errorMessage(errMessage);
                return hashResult1;
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException("Error occurred while calculating the hash.", e);
            }
        }
        throw new IllegalStateException("Unexpected state in hash calculation.");
    }

    private static String extractHash(String outputLine) {
        char c;
        StringBuilder hash = new StringBuilder();
        outputLine = outputLine.trim();
        for (int i = 0; i < outputLine.length() && (c = outputLine.charAt(i)) != ' '; ++i) {
            hash.append(c);
        }
        return hash.toString();
    }

    public static void changeFilePermissions(String filePath, String permissions) {
        Path path = Paths.get(filePath, new String[0]);
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(permissions);
        try {
            Files.setPosixFilePermissions(path, perms);
            Utility.infoMessage("Permissions " + permissions + " changed for file:", filePath);
        }
        catch (IOException e) {
            String errMessage = "Failed to change permissions " + permissions + " for file:" + filePath;
            Utility.errorMessage(errMessage);
        }
    }

    public static String cleanName(String name) {
        int i;
        if (name == null || name.isEmpty()) {
            return name;
        }
        name = name.trim();
        boolean isColonFound = false;
        for (i = 0; i < name.length() && name.startsWith(LimineKey.ENTRY_KEY.toString(), i); ++i) {
            isColonFound = true;
        }
        if (isColonFound && name.startsWith(LimineKey.EXPAND.toString(), i)) {
            ++i;
        }
        return name.substring(i);
    }

    public static List<String> collectFilesFromDirectory(String directoryPath) {
        ArrayList<String> list = new ArrayList<String>();
        Path path = Paths.get(directoryPath, new String[0]);
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path);){
            for (Path entry : stream) {
                String fileName = entry.getFileName().toString();
                list.add(fileName);
            }
        }
        catch (IOException ignored) {
            Utility.errorMessage("The directory cannot be listed:", directoryPath);
        }
        return list;
    }

    public static int compareTimestamps(String timestampA, String timestampB) {
        return timestampA.compareTo(timestampB);
    }

    public static String convertLocalToUTC(String localTime) {
        LocalDateTime localLdt = LocalDateTime.parse(localTime, TIME_FORMATTER);
        ZonedDateTime utcZdt = localLdt.atZone(ZoneId.systemDefault()).withZoneSameInstant(ZoneOffset.UTC);
        return TIME_FORMATTER.format(utcZdt);
    }

    public static String convertUTCtoLocal(String utcTime) {
        LocalDateTime utcLdt = LocalDateTime.parse(utcTime, TIME_FORMATTER);
        ZonedDateTime localZdt = utcLdt.atZone(ZoneOffset.UTC).withZoneSameInstant(ZoneId.systemDefault());
        return TIME_FORMATTER.format(localZdt);
    }

    public static void copyFile(String sourceFilePath, String targetFilePath, boolean force) {
        Utility.copyFile(sourceFilePath, targetFilePath, force, true, true);
    }

    public static void copyFile(String sourceFilePath, String targetFilePath, boolean force, boolean showOutput, boolean showCommand) {
        if (force) {
            Utility.runCommand("cp -f '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
        } else {
            Utility.runCommand("cp -u '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
        }
    }

    public static void deleteFile(String absoluteFilePath) {
        Utility.deleteFile(absoluteFilePath, true, true);
    }

    public static void deleteFile(String absoluteFilePath, boolean showOutput, boolean showCommand) {
        Utility.runCommand("rm -f '" + absoluteFilePath + "'", showOutput, showCommand);
    }

    public static boolean isCommandPresent(String command) {
        String[] paths;
        String pathEnv = System.getenv("PATH");
        if (pathEnv == null || pathEnv.isEmpty()) {
            return false;
        }
        for (String path : paths = pathEnv.split(File.pathSeparator)) {
            File commandFile = new File(path, command);
            if (!commandFile.isFile() || !commandFile.canExecute()) continue;
            return true;
        }
        return false;
    }

    public static boolean isDirectoryPresent(String dirPath) {
        Path path = Paths.get(dirPath, new String[0]);
        try {
            return Files.exists(path, new LinkOption[0]) && Files.isDirectory(path, new LinkOption[0]);
        }
        catch (Exception e) {
            return false;
        }
    }

    public static boolean isFileOlderThan(String filePath, int hours) {
        long thresholdMillis;
        File file = new File(filePath);
        long lastModified = file.lastModified();
        if (lastModified == 0L) {
            return false;
        }
        long now = System.currentTimeMillis();
        long diffMillis = now - lastModified;
        return diffMillis > (thresholdMillis = (long)hours * 3600000L);
    }

    public static boolean isFilePresent(String filePath) {
        File file = new File(filePath);
        return file.exists() && file.isFile();
    }

    public static boolean isRootOwnedFile(String filePath) {
        try {
            return Objects.equals(Files.getOwner(Paths.get(filePath, new String[0]), new LinkOption[0]).getName(), "root");
        }
        catch (IOException e) {
            return false;
        }
    }

    public static boolean isSystemEfi() {
        File efiMountPoint = new File("/sys/firmware/efi");
        return efiMountPoint.exists();
    }

    public static boolean isTimeFormatNotValid(String timeString) {
        try {
            LocalDateTime.parse(timeString, TIME_FORMATTER);
            return false;
        }
        catch (DateTimeParseException e) {
            return true;
        }
    }

    public static boolean isValidUUID(String uuidString) {
        if (uuidString == null || uuidString.isEmpty()) {
            return false;
        }
        try {
            UUID uuid = UUID.fromString(uuidString);
            return uuidString.equals(uuid.toString());
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean ensureDirectoryExists(String dirPath) {
        File dir = new File(dirPath);
        if (!dir.exists()) {
            if (dir.mkdirs()) {
                Utility.infoMessage("Directory created:", dir.getPath());
                return true;
            }
            Utility.errorMessage("Failed to create directory:", dir.getPath());
            return false;
        }
        return true;
    }

    public static void ensurePathExists(String filePath) {
        Path path = Paths.get(filePath, new String[0]);
        if (Files.exists(path, new LinkOption[0])) {
            return;
        }
        Utility.ensureDirectoryExists(path.getParent().toString());
    }

    public static void infoMessage(String message) {
        Utility.infoMessage(message, null);
    }

    public static void infoMessage(String message, String output) {
        if (Config.QUIET) {
            return;
        }
        if (output == null || output.isEmpty()) {
            System.out.println(String.valueOf((Object)ConsoleColor.WHITE_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
        } else {
            System.out.println(String.valueOf((Object)ConsoleColor.GREEN_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
        }
    }

    public static void warnMessage(String message) {
        Utility.warnMessage(message, null);
    }

    public static void warnMessage(String message, String output) {
        if (output == null || output.isEmpty()) {
            System.out.println(String.valueOf((Object)ConsoleColor.YELLOW_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
            Utility.loggerSend(message, LogLevel.WARNING);
        } else {
            System.out.println(String.valueOf((Object)ConsoleColor.YELLOW_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
            Utility.loggerSend(message + " " + output, LogLevel.WARNING);
        }
    }

    public static void errorMessage(String message) {
        Utility.errorMessage(message, null);
    }

    public static void errorMessage(String message, String output) {
        if (output == null || output.isEmpty()) {
            System.err.println(String.valueOf((Object)ConsoleColor.RED_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
            Utility.loggerSend(message, LogLevel.ERROR);
        } else {
            System.err.println(String.valueOf((Object)ConsoleColor.RED_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
            Utility.loggerSend(message + " " + output, LogLevel.ERROR);
        }
    }

    public static void exitWithError(String message) {
        Utility.exitWithError(message, null, false);
    }

    public static void exitWithError(String message, boolean hold) {
        Utility.exitWithError(message, null, hold);
    }

    public static void exitWithError(String message, String output) {
        Utility.exitWithError(message, output, false);
    }

    public static void exitWithError(String message, String output, boolean hold) {
        Utility.errorMessage(message, output);
        if (hold || Config.IS_TIME_FOR_RESTORE) {
            Utility.infoMessage("You can review, copy or take a screenshot of this error output before the terminal closes.", null);
            Utility.deleteFile("/tmp/limine-snapper-restore.lock", false, false);
            while (!Utility.askUser("Do you want to close this terminal? (yes|no) If no, just ignore this question in loop.")) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        System.exit(1);
    }

    public static String extractParentDirectoryPath(String path) {
        int lastIndex = path.lastIndexOf(47);
        if (lastIndex != -1) {
            return path.substring(0, lastIndex) + "/";
        }
        return "";
    }

    public static String extractLastWord(String line) {
        int lastSpaceIndex = line.lastIndexOf(32);
        if (lastSpaceIndex != -1) {
            return line.substring(lastSpaceIndex + 1).trim();
        }
        return line;
    }

    public static int findLatestSnapperNumber(String directoryPath) {
        Path path = Paths.get(directoryPath, new String[0]);
        int maxNumber = -1;
        Pattern pattern = Pattern.compile("\\d+");
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path);){
            for (Path entry : stream) {
                int number;
                String fileName = entry.getFileName().toString();
                if (!pattern.matcher(fileName).matches() || (number = Integer.parseInt(fileName)) <= maxNumber) continue;
                maxNumber = number;
            }
        }
        catch (IOException ignored) {
            String errMessage = "The latest Snapper snapshot number cannot be found in the directory: " + directoryPath;
            Utility.errorMessage(errMessage);
        }
        return maxNumber;
    }

    public static String generateRandomMachineID() {
        SecureRandom random = new SecureRandom();
        StringBuilder newMachineID = new StringBuilder(32);
        for (int i = 0; i < 32; ++i) {
            int nextInt = random.nextInt(16);
            newMachineID.append(Integer.toHexString(nextInt));
        }
        try {
            Files.writeString(Paths.get(MACHINE_ID_FILE_PATH, new String[0]), (CharSequence)newMachineID, new OpenOption[0]);
            Utility.infoMessage("New machine-id generated:", newMachineID.toString());
            Utility.infoMessage("This machine-id saved:", MACHINE_ID_FILE_PATH);
            Utility.changeFilePermissions(MACHINE_ID_FILE_PATH, "r--r--r--");
        }
        catch (IOException e) {
            Utility.exitWithError("New machine ID generator failed. Error:", e.getMessage());
        }
        return newMachineID.toString();
    }

    public static String getCurrentIsoFormattedTime() {
        return LocalDateTime.now().withNano(0).format(ISO_FORMATTER);
    }

    public static String getCurrentUTCTime() {
        return LocalDateTime.now(ZoneOffset.UTC).format(TIME_FORMATTER);
    }

    public static String getLeadingSpaces(String input) {
        if (input == null) {
            return "";
        }
        StringBuilder leadingSpaces = new StringBuilder();
        int length = input.length();
        for (int i = 0; i < length && input.charAt(i) == ' '; ++i) {
            leadingSpaces.append(' ');
        }
        return leadingSpaces.toString();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String getSubvolFromProcMounts(String mount) {
        Path procPath = Path.of("/proc/mounts", new String[0]);
        String prefix = "subvol=";
        try (BufferedReader reader = Files.newBufferedReader(procPath);){
            String options;
            int idx;
            String[] parts;
            String line;
            do {
                if ((line = reader.readLine()) == null) return null;
            } while (!line.contains(" " + mount + " ") || (parts = line.split(" ")).length < 4 || !parts[1].equals(mount) || (idx = (options = parts[3]).indexOf(prefix)) == -1);
            int start = idx + prefix.length();
            int end = options.indexOf(44, start);
            String string = end == -1 ? options.substring(start) : options.substring(start, end);
            return string;
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return null;
    }

    public static Output getTextFromCommand(String commandLine, boolean exitIfError, String errorMessage) {
        ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", commandLine).redirectErrorStream(true);
        try {
            Process process = processBuilder.start();
            ArrayList<String> textOutput = new ArrayList<String>();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                reader.lines().forEach(textOutput::add);
            }
            int errorCode = process.waitFor();
            if (errorCode == 0) {
                return new Output(textOutput, null, errorCode, true);
            }
            if (errorMessage != null) {
                if (!errorMessage.isEmpty()) {
                    Utility.errorMessage(errorMessage);
                }
            } else {
                Utility.errorMessage("Command failed:", commandLine);
                if (!textOutput.isEmpty() && !((String)textOutput.get(0)).trim().isEmpty()) {
                    Utility.errorMessage("Output:", Arrays.toString(textOutput.toArray()));
                }
            }
            if (exitIfError) {
                Utility.exitWithError("Exit code:", String.valueOf(errorCode));
            }
            return new Output(null, textOutput, errorCode, false);
        }
        catch (IOException | InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public static double getUsagePercent(String mountPoint) {
        Path path = Paths.get(mountPoint, new String[0]);
        try {
            FileStore store = Files.getFileStore(path);
            long total = store.getTotalSpace();
            long usable = store.getUsableSpace();
            long used = total - usable;
            double percent = (double)used / (double)total * 100.0;
            return new BigDecimal(percent).setScale(2, RoundingMode.HALF_UP).doubleValue();
        }
        catch (IOException e) {
            Utility.errorMessage("Error when checking the usage of " + mountPoint + ":", e.getMessage());
            return -1.0;
        }
    }

    public static String getTotalSpace(String mountPoint) {
        Path path = Paths.get(mountPoint, new String[0]);
        try {
            FileStore store = Files.getFileStore(path);
            double totalGB = (double)store.getTotalSpace() / 1.073741824E9;
            return String.format("%.2f GiB", totalGB);
        }
        catch (IOException e) {
            Utility.errorMessage("Error when checking the total space of " + mountPoint + ": " + e.getMessage());
            return "-- GiB";
        }
    }

    public static String getUuid() {
        Output output = Utility.getTextFromCommand("findmnt --mountpoint / -no UUID", true, null);
        if (output.isSuccess() && !output.text().isEmpty() && !output.text().get(0).isEmpty()) {
            return output.text().get(0);
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String getVersion() {
        try (InputStream manifestStream = Utility.class.getResourceAsStream("/META-INF/MANIFEST.MF");){
            if (manifestStream == null) return "Unknown";
            Manifest manifest = new Manifest(manifestStream);
            Attributes attributes = manifest.getMainAttributes();
            String string = attributes.getValue("Implementation-Version");
            return string;
        }
        catch (Exception exception) {
            // empty catch block
        }
        return "Unknown";
    }

    public static boolean doesPartitionHaveSpace(double limitUsagePercent, String mountPoint, Set<String> filePaths, Set<String> deletedFilePaths) {
        try {
            long newFilesSize = Utility.calculateFilesSize(filePaths, deletedFilePaths);
            FileStore store = Files.getFileStore(Paths.get(mountPoint, new String[0]));
            long totalSize = store.getTotalSpace();
            long currentUsage = totalSize - store.getUsableSpace();
            long combinedUsage = currentUsage + newFilesSize + 0x400000L;
            double usagePercent = (double)combinedUsage / (double)totalSize * 100.0;
            return usagePercent < limitUsagePercent;
        }
        catch (IOException e) {
            Utility.exitWithError(e.getMessage());
            return false;
        }
    }

    private static long calculateFilesSize(Set<String> filePaths, Set<String> deletedFilePaths) {
        File file;
        long newFilesSize = 0L;
        for (String filePath : filePaths) {
            file = new File(filePath);
            if (!file.exists() || !file.isFile()) continue;
            newFilesSize += file.length();
        }
        for (String filePath : deletedFilePaths) {
            file = new File(filePath);
            if (!file.exists() || !file.isFile()) continue;
            newFilesSize -= file.length();
        }
        return newFilesSize;
    }

    public static synchronized void loggerSend(String message, LogLevel logLevel) {
        try {
            ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", "logger -t limine-snapper-sync -p " + logLevel.level + " \"" + message + "\"");
            processBuilder.start();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static void moveFile(String sourceFilePath, String targetFilePath, boolean force, boolean showOutput, boolean showCommand) {
        if (force) {
            Utility.runCommand("mv -f '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
        } else {
            Utility.runCommand("mv -u '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
        }
    }

    public static void moveFile(String sourceFilePath, String targetFilePath, boolean force) {
        Utility.moveFile(sourceFilePath, targetFilePath, force, true, true);
    }

    public static void printTable(List<String[]> data, int columnWidth1, int columnWidth2, int columnWidth3) {
        String colSep = "\u2502";
        String rowColSep = "\u253c";
        String rowSep = "\u2500";
        int[] columnWidths = new int[]{columnWidth1, columnWidth2, columnWidth3};
        String header = String.format(" %1$-" + columnWidths[0] + "s " + colSep + " %2$-" + columnWidths[1] + "s " + colSep + " %3$-" + columnWidths[2] + "s ", "ID", "Date", "Description");
        String rowSeparator = rowSep.repeat(columnWidths[0] + 2) + rowColSep + rowSep.repeat(columnWidths[1] + 2) + rowColSep + rowSep.repeat(columnWidths[2] + 2);
        System.out.println(header);
        System.out.println(rowSeparator);
        for (String[] row : data) {
            System.out.printf(" %1$-" + columnWidths[0] + "s " + colSep + " %2$-" + columnWidths[1] + "s " + colSep + " %3$-" + columnWidths[2] + "s %n", row[0], row[1], row[2]);
        }
    }

    public static String readMachineID() {
        String machineID;
        try {
            machineID = new String(Files.readAllBytes(Paths.get(MACHINE_ID_FILE_PATH, new String[0]))).trim();
        }
        catch (IOException e) {
            machineID = Utility.generateRandomMachineID();
        }
        return machineID;
    }

    public static String removeSurroundingSlashes(String path) {
        int start;
        int end = path.length();
        for (start = 0; start < end && path.charAt(start) == '/'; ++start) {
        }
        while (end > start && path.charAt(end - 1) == '/') {
            --end;
        }
        return path.substring(start, end).trim();
    }

    public static String removeSurrounding(String str, String delimiter) {
        if (str != null && str.length() >= 2 && str.startsWith(delimiter) && str.endsWith(delimiter)) {
            return str.substring(delimiter.length(), str.length() - delimiter.length());
        }
        return str;
    }

    public static boolean runCommand(String command, boolean showOutput, boolean showCommand) {
        return Utility.runCommands(List.of(command), true, showOutput, showCommand);
    }

    public static boolean runCommands(List<String> commands, boolean stopNextCommandOnError, boolean showOutput, boolean showCommand) {
        boolean isSuccess = true;
        try {
            for (String command : commands) {
                ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", command);
                Process process = processBuilder.start();
                CompletableFuture<Void> outputFuture = null;
                CompletableFuture<Void> errorFuture = null;
                if (showOutput) {
                    outputFuture = CompletableFuture.runAsync(() -> Utility.printStream(process.getInputStream(), System.out::println));
                    errorFuture = CompletableFuture.runAsync(() -> Utility.printStream(process.getErrorStream(), System.err::println));
                }
                int errorCode = process.waitFor();
                if (showOutput) {
                    outputFuture.join();
                    errorFuture.join();
                }
                if (errorCode != 0) {
                    isSuccess = false;
                    Utility.errorMessage("Command failed:", "\"" + command + "\" with exit code: " + errorCode);
                    if (!stopNextCommandOnError) continue;
                    break;
                }
                if (!showCommand) continue;
                Utility.infoMessage("Command succeeded:", command);
            }
            return isSuccess;
        }
        catch (IOException | InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private static void printStream(InputStream inputStream, Consumer<String> consumer) {
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));){
            bufferedReader.lines().forEach(consumer);
        }
        catch (IOException ignored) {
            Utility.errorMessage("Error reading process stream");
        }
    }

    public static String snapperConfigName() {
        Output output = Utility.getTextFromCommand("snapper --no-headers --utc --machine-readable csv list-configs", false, null);
        if (output.isSuccess()) {
            for (String name : output.text()) {
                String[] columns;
                if (!name.trim().endsWith("/") || (columns = name.split(",")).length != 2 || !"/".equals(columns[1].trim())) continue;
                return columns[0].trim();
            }
        }
        return null;
    }

    public static String userInput(String message) {
        System.out.println(message);
        while (true) {
            System.out.print(String.valueOf((Object)ConsoleColor.YELLOW) + "\nEnter your input: " + String.valueOf((Object)ConsoleColor.RESET));
            String userInput = INPUT_SCANNER.nextLine().trim();
            if (Utility.isValidInput(userInput)) {
                return userInput;
            }
            System.err.println(String.valueOf((Object)ConsoleColor.RED) + "Invalid input. Please avoid using escape sequences and characters." + String.valueOf((Object)ConsoleColor.RESET));
        }
    }

    private static boolean isValidInput(String input) {
        return !FORBIDDEN_PATTERN.matcher(input).find();
    }

    public static String userSelect(String ask) {
        System.out.print(ask + String.valueOf((Object)ConsoleColor.YELLOW) + "\nEnter your choice: " + String.valueOf((Object)ConsoleColor.RESET));
        String input = INPUT_SCANNER.nextLine();
        if (!input.trim().isEmpty() && "cancel".startsWith(input.toLowerCase())) {
            System.out.println("The process is aborted.");
            System.exit(1);
        }
        return input;
    }

    public static int validateAndGetSnapshotID() {
        String line;
        try {
            BufferedReader reader = new BufferedReader(new FileReader("/proc/cmdline"));
            line = reader.readLine();
        }
        catch (IOException e) {
            line = null;
        }
        int snapshotID = -1;
        boolean isRootFlagsFound = false;
        if (line != null) {
            String[] kernelParameters;
            for (String parameter : kernelParameters = line.split(" ")) {
                String[] rootflags = Utility.readRootFlags(parameter);
                if (rootflags == null) continue;
                for (String flag : rootflags) {
                    if (!flag.startsWith("subvol=") || !flag.endsWith("/snapshot")) continue;
                    snapshotID = Utility.extractSnapshotID(flag);
                    break;
                }
                isRootFlagsFound = true;
                break;
            }
        } else {
            String errMessage = "No kernel cmdline from /proc/cmdline";
            Utility.errorMessage(errMessage);
            return -1;
        }
        if (!isRootFlagsFound) {
            return -1;
        }
        return snapshotID;
    }

    private static int extractSnapshotID(String flag) {
        try {
            String[] dirs = flag.split("/");
            String snapshotIDStr = dirs[dirs.length - 2];
            if (snapshotIDStr != null) {
                return Integer.parseInt(snapshotIDStr);
            }
        }
        catch (NumberFormatException dirs) {
            // empty catch block
        }
        String errMessage = "Failed to extract snapshot ID from the flag: " + flag;
        Utility.exitWithError(errMessage);
        return -1;
    }

    public static String[] readRootFlags(String rootFlags) {
        if (rootFlags.startsWith("rootflags=")) {
            String rootflagsPart = rootFlags.substring("rootflags=".length());
            return rootflagsPart.split(",");
        }
        return null;
    }
}

