/*
 * Decompiled with CFR 0.152.
 */
package org.limine.entry.tool.processes;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.limine.entry.tool.formats.limine8.Limine;
import org.limine.entry.tool.objects.Config;
import org.limine.entry.tool.objects.ConsoleColor;
import org.limine.entry.tool.objects.LogLevel;
import org.limine.entry.tool.objects.Output;

public class Utility {
    private static final int MACHINE_ID_LENGTH = 32;
    private static final String MACHINE_ID_FILE_PATH = "/etc/machine-id";

    public static void ensureDirectoryExists(String dirPath) {
        File dir = new File(dirPath);
        if (!dir.exists()) {
            if (dir.mkdirs()) {
                Utility.infoMessage("Directory created:", dir.getPath());
            } else {
                Utility.errorMessage("Failed to create directory:", dir.getPath());
            }
        }
    }

    public static String readMachineID() {
        String machineID;
        try {
            machineID = new String(Files.readAllBytes(Paths.get(MACHINE_ID_FILE_PATH, new String[0]))).trim();
        }
        catch (IOException e) {
            machineID = Utility.generateRandomMachineID();
        }
        if (machineID.length() != 32) {
            Utility.exitWithError("Invalid machine-ID length: " + machineID.length() + ". The length should be 32");
        }
        return machineID;
    }

    public static boolean isFilePresent(String filePath) {
        File file = new File(filePath);
        return file.exists() && file.isFile();
    }

    public static boolean isDirectoryPresent(String dirPath) {
        Path path = Paths.get(dirPath, new String[0]);
        try {
            return Files.exists(path, new LinkOption[0]) && Files.isDirectory(path, new LinkOption[0]);
        }
        catch (Exception e) {
            return false;
        }
    }

    public static String generateRandomMachineID() {
        SecureRandom random = new SecureRandom();
        StringBuilder newMachineID = new StringBuilder(32);
        for (int i = 0; i < 32; ++i) {
            int nextInt = random.nextInt(16);
            newMachineID.append(Integer.toHexString(nextInt));
        }
        try {
            Files.writeString(Paths.get(MACHINE_ID_FILE_PATH, new String[0]), (CharSequence)newMachineID, new OpenOption[0]);
            Utility.infoMessage("New machine-id generated:", String.valueOf(newMachineID));
            Utility.infoMessage("This machine-id saved:", MACHINE_ID_FILE_PATH);
            Utility.changeFilePermissions(MACHINE_ID_FILE_PATH, "r--r--r--");
        }
        catch (IOException e) {
            Utility.exitWithError("New machine ID generator failed. Error:", e.getMessage());
        }
        return newMachineID.toString();
    }

    public static void changeFilePermissions(String filePath, String permissions) {
        Path path = Paths.get(filePath, new String[0]);
        Set<PosixFilePermission> perms = PosixFilePermissions.fromString(permissions);
        try {
            Files.setPosixFilePermissions(path, perms);
            Utility.infoMessage("Permissions " + permissions + " changed for file:", filePath);
        }
        catch (IOException e) {
            Utility.errorMessage("Failed to change permissions " + permissions + " for file: ", filePath);
        }
    }

    private static String extractHash(String outputLine) {
        char c;
        StringBuilder hash = new StringBuilder();
        outputLine = outputLine.trim();
        for (int i = 0; i < outputLine.length() && (c = outputLine.charAt(i)) != ' '; ++i) {
            hash.append(c);
        }
        return hash.toString();
    }

    public static String computeBlake2(String absoluteFilePath) {
        return Utility.blake2IntegrityCheck(absoluteFilePath, 2);
    }

    private static String blake2sum(String absoluteFilePath) {
        Output output = Utility.getTextFromCommand("b2sum --binary " + absoluteFilePath, false, true);
        if (output.isSuccess()) {
            return Utility.extractHash(output.text().get(0));
        }
        return "";
    }

    public static String blake2IntegrityCheck(String absoluteFilePath, int repeatIfCorrupted) {
        for (int attempts = 0; attempts <= repeatIfCorrupted; ++attempts) {
            try {
                CompletableFuture<String> hashFuture1 = CompletableFuture.supplyAsync(() -> Utility.blake2sum(absoluteFilePath));
                CompletableFuture<String> hashFuture2 = CompletableFuture.supplyAsync(() -> Utility.blake2sum(absoluteFilePath));
                String hashResult1 = hashFuture1.get();
                String hashResult2 = hashFuture2.get();
                if (hashResult1.equals(hashResult2)) {
                    return hashResult1;
                }
                String errMessage = String.format("Hash mismatch detected! This is a hardware issue (RAM or CPU is faulty). File: %s (Attempt %d/%d).", absoluteFilePath, attempts + 1, repeatIfCorrupted + 1);
                Utility.errorMessage(errMessage);
                if (attempts != repeatIfCorrupted) continue;
                errMessage = "Max retry attempts reached. Returning a corrupted hash. Do NOT trust this system and hardware!";
                Utility.errorMessage(errMessage);
                return hashResult1;
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException("Error occurred while calculating the hash.", e);
            }
        }
        throw new IllegalStateException("Unexpected state in hash calculation.");
    }

    public static boolean copyFile(String sourceFilePath, String targetFilePath, boolean force, boolean showOutput, boolean showCommand) {
        if (sourceFilePath == null || sourceFilePath.isBlank()) {
            return false;
        }
        if (Utility.isFilePresent(sourceFilePath)) {
            if (force) {
                return Utility.runCommand("cp -f '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
            }
            return Utility.runCommand("cp -u '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
        }
        return false;
    }

    public static void copyFile(String sourceFilePath, String targetFilePath, boolean force) {
        if (Utility.copyFile(sourceFilePath, targetFilePath, force, true, false)) {
            Utility.infoMessage("Copied:", sourceFilePath + " -> " + targetFilePath);
        } else {
            Utility.errorMessage("Failed to copy:", sourceFilePath + " -> " + targetFilePath);
        }
    }

    public static void moveFile(String sourceFilePath, String targetFilePath, boolean force, boolean showOutput, boolean showCommand) {
        if (sourceFilePath == null || sourceFilePath.isBlank()) {
            return;
        }
        if (Utility.isFilePresent(sourceFilePath)) {
            if (force) {
                Utility.runCommand("mv -f '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
            } else {
                Utility.runCommand("mv -u '" + sourceFilePath + "' '" + targetFilePath + "'", showOutput, showCommand);
            }
        }
    }

    public static void deleteDirectory(String dirPath) {
        if (dirPath == null || dirPath.isBlank() || "/".equals(dirPath.trim()) || dirPath.trim().startsWith("/home")) {
            return;
        }
        if (Utility.isDirectoryPresent(dirPath)) {
            if (Utility.runCommand("rm -r '" + dirPath + "'", true, false)) {
                Utility.infoMessage("Deleted:", dirPath);
            } else {
                Utility.errorMessage("Failed to delete:", dirPath);
            }
        }
    }

    public static void deleteFile(String absoluteFilePath) {
        if (absoluteFilePath == null || absoluteFilePath.isBlank()) {
            return;
        }
        if (Utility.isFilePresent(absoluteFilePath)) {
            if (Utility.runCommand("rm '" + absoluteFilePath + "'", true, false)) {
                Utility.infoMessage("Deleted:", absoluteFilePath);
            } else {
                Utility.errorMessage("Failed to delete:", absoluteFilePath);
            }
        }
    }

    public static Output getTextFromCommand(String commandLine, boolean exitIfError, boolean displayErrorMessage) {
        ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", commandLine).redirectErrorStream(true);
        try {
            Process process = processBuilder.start();
            ArrayList<String> textOutput = new ArrayList<String>();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                reader.lines().forEach(textOutput::add);
            }
            int errorCode = process.waitFor();
            if (errorCode == 0) {
                return new Output(textOutput, null, errorCode, true);
            }
            if (displayErrorMessage) {
                Utility.errorMessage("Command failed:", commandLine);
                if (!textOutput.isEmpty() && !((String)textOutput.get(0)).trim().isEmpty()) {
                    Utility.errorMessage("Output:", Arrays.toString(textOutput.toArray()));
                }
            }
            if (exitIfError) {
                Utility.exitWithError("Exit code:", String.valueOf(errorCode));
            }
            return new Output(null, textOutput, errorCode, false);
        }
        catch (IOException | InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public static boolean runCommands(List<String> commands, boolean stopNextCommandOnError, boolean showOutput, boolean showCommand) {
        int errorCode = -1;
        try {
            for (String command : commands) {
                ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", command);
                Process process = processBuilder.start();
                CompletableFuture<Void> outputFuture = null;
                CompletableFuture<Void> errorFuture = null;
                if (showOutput) {
                    outputFuture = CompletableFuture.runAsync(() -> Utility.printStream(process.getInputStream(), System.out::println));
                    errorFuture = CompletableFuture.runAsync(() -> Utility.printStream(process.getErrorStream(), System.err::println));
                }
                errorCode = process.waitFor();
                if (showOutput) {
                    outputFuture.join();
                    errorFuture.join();
                }
                if (errorCode != 0) {
                    Utility.errorMessage("Command failed:", "\"" + command + "\" with exit code: " + errorCode);
                    if (!stopNextCommandOnError) continue;
                    break;
                }
                if (!showCommand) continue;
                Utility.infoMessage("Command:", command);
            }
        }
        catch (IOException | InterruptedException e) {
            throw new RuntimeException(e);
        }
        return errorCode == 0;
    }

    private static void printStream(InputStream inputStream, Consumer<String> consumer) {
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));){
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                consumer.accept(line);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static boolean runCommand(String command, boolean showOutput, boolean showCommand) {
        return Utility.runCommands(List.of(command), true, showOutput, showCommand);
    }

    public static String extractFileNameFromPath(String path) {
        int lastIndex = path.lastIndexOf(47);
        if (lastIndex != -1) {
            return path.substring(lastIndex + 1);
        }
        return path;
    }

    public static void loggerSend(String message, LogLevel logLevel) {
        try {
            ProcessBuilder processBuilder = new ProcessBuilder("bash", "-c", "logger -t limine-entry-tool -p " + logLevel.level + " \"" + message + "\"");
            processBuilder.start();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static String readKernelCmdline() {
        String[] possiblePaths;
        for (String pathString : possiblePaths = new String[]{"/etc/kernel/cmdline", "/proc/cmdline"}) {
            Path path = Paths.get(pathString, new String[0]);
            if (!Files.exists(path, new LinkOption[0])) continue;
            try {
                String line = new String(Files.readAllBytes(path));
                return line.replaceAll("[\\r\\n]+$", "").trim();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return null;
    }

    public static String getLinuxDistroName() {
        Path osReleasePath = Paths.get("/etc/os-release", new String[0]);
        try {
            if (Files.exists(osReleasePath, new LinkOption[0])) {
                List<String> lines = Files.readAllLines(osReleasePath);
                for (String line : lines) {
                    if (!line.startsWith("PRETTY_NAME=")) continue;
                    return line.split("=")[1].replaceAll("\"", "");
                }
            }
            Path debianVersionPath = Paths.get("/etc/debian_version", new String[0]);
            Path redhatVersionPath = Paths.get("/etc/redhat-release", new String[0]);
            if (Files.exists(Paths.get("/etc/lsb-release", new String[0]), new LinkOption[0])) {
                return Utility.getLinuxDistroNameFromLsbReleaseFile();
            }
            if (Files.exists(debianVersionPath, new LinkOption[0])) {
                return "Debian " + Files.readString(debianVersionPath).trim();
            }
            if (Files.exists(redhatVersionPath, new LinkOption[0])) {
                return Files.readString(redhatVersionPath).trim();
            }
        }
        catch (IOException e) {
            Utility.errorMessage("Error reading Linux Distro information:", e.getMessage());
        }
        return null;
    }

    private static String getLinuxDistroNameFromLsbReleaseFile() throws IOException {
        List<String> lines = Files.readAllLines(Paths.get("/etc/lsb-release", new String[0]));
        for (String line : lines) {
            if (!line.startsWith("DISTRIB_DESCRIPTION=")) continue;
            return line.split("=")[1].replaceAll("\"", "").trim();
        }
        return null;
    }

    public static String cleanName(String name) {
        int i;
        if (name == null || name.isEmpty()) {
            return name;
        }
        name = name.trim();
        boolean isColonFound = false;
        for (i = 0; i < name.length() && name.startsWith(Limine.ENTRY_KEY.toString(), i); ++i) {
            isColonFound = true;
        }
        if (isColonFound && name.startsWith(Limine.EXPAND.toString(), i)) {
            ++i;
        }
        return name.substring(i);
    }

    public static String buildPath(String ... parts) {
        String path = Paths.get("", parts).toString();
        if (path.startsWith(File.separator)) {
            return path;
        }
        return File.separator + path;
    }

    public static boolean isCommandPresent(String command) {
        String[] paths;
        String pathEnv = System.getenv("PATH");
        if (pathEnv == null || pathEnv.isEmpty()) {
            return false;
        }
        for (String path : paths = pathEnv.split(File.pathSeparator)) {
            File commandFile = new File(path, command);
            if (!commandFile.isFile() || !commandFile.canExecute()) continue;
            return true;
        }
        return false;
    }

    public static boolean isSystemEfi() {
        return new File("/sys/firmware/efi").exists();
    }

    public static boolean isSystemAmd64() {
        String arch = System.getProperty("os.arch").toLowerCase();
        return arch.equals("x86_64") || arch.equals("amd64");
    }

    public static boolean validateFatPartition(String path) {
        File bootPath = new File(path);
        if (!bootPath.exists() || !bootPath.isDirectory()) {
            Utility.errorMessage("The specified ESP path '" + path + "' does not exist or is not a directory.");
            return false;
        }
        try {
            String fileSystemType = Files.getFileStore(Paths.get(path, new String[0])).type();
            if (!"vfat".equalsIgnoreCase(fileSystemType)) {
                Utility.errorMessage("The ESP path '" + path + "' does not have a FAT32 (vfat) filesystem.");
                return false;
            }
        }
        catch (IOException e) {
            Utility.errorMessage("Error checking filesystem type of the ESP path '" + path + "' :", e.getMessage());
            return false;
        }
        return true;
    }

    public static String correctCaseSensitivePath(String path) {
        File file = new File(path);
        if (!file.exists()) {
            return null;
        }
        StringBuilder correctedPath = new StringBuilder();
        File current = file;
        while (current != null) {
            File parent = current.getParentFile();
            if (parent != null) {
                File[] siblings = parent.listFiles();
                if (siblings != null) {
                    for (File sibling : siblings) {
                        if (!sibling.getName().equalsIgnoreCase(current.getName())) continue;
                        correctedPath.insert(0, File.separator + sibling.getName());
                        break;
                    }
                }
            } else {
                correctedPath.insert(0, current.getName());
            }
            current = parent;
        }
        return correctedPath.toString();
    }

    public static boolean isValidFat32Name(String name) {
        Pattern FAT32_VALID_NAME_PATTERN = Pattern.compile("^(?![.\\-\\s])[a-zA-Z0-9._-]+(?<![.\\-\\s])$");
        if (name == null || name.isBlank() || name.length() > 99) {
            return false;
        }
        return FAT32_VALID_NAME_PATTERN.matcher(name).matches();
    }

    public static boolean isFileOlderThan(String filePath, int hours) {
        long thresholdMillis;
        File file = new File(filePath);
        long lastModified = file.lastModified();
        if (lastModified == 0L) {
            return false;
        }
        long now = System.currentTimeMillis();
        long diffMillis = now - lastModified;
        return diffMillis > (thresholdMillis = (long)hours * 3600000L);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String getVersion() {
        try (InputStream manifestStream = Utility.class.getResourceAsStream("/META-INF/MANIFEST.MF");){
            if (manifestStream == null) return "Unknown";
            Manifest manifest = new Manifest(manifestStream);
            Attributes attributes = manifest.getMainAttributes();
            String string = attributes.getValue("Implementation-Version");
            return string;
        }
        catch (Exception exception) {
            // empty catch block
        }
        return "Unknown";
    }

    public static String getMountPoint(String partUUID) {
        Output pathOutput = Utility.getTextFromCommand("findmnt -no TARGET -S PARTUUID=" + partUUID, false, false);
        if (pathOutput.isSuccess()) {
            return pathOutput.text().get(0);
        }
        return "";
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static String getMountPointIfFat32(String filePath) {
        Path realPath = new File(filePath).getAbsoluteFile().toPath().normalize();
        File mountFile = new File("/proc/mounts");
        if (!mountFile.exists()) {
            return "";
        }
        try (Stream<String> lines = Files.lines(mountFile.toPath());){
            String mountPoint;
            block12: {
                Iterator iterator = ((Iterable)lines::iterator).iterator();
                while (iterator.hasNext()) {
                    String line = (String)iterator.next();
                    String[] parts = line.split(" ");
                    if (parts.length < 3) continue;
                    mountPoint = parts[1];
                    String fsType = parts[2];
                    if (!realPath.startsWith(Paths.get(mountPoint, new String[0])) || !"vfat".equalsIgnoreCase(fsType)) {
                        continue;
                    }
                    break block12;
                }
                return "";
            }
            String string = mountPoint;
            return string;
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return "";
    }

    public static String removeSurroundingSlashes(String path) {
        if (path == null || path.isEmpty()) {
            return path;
        }
        return path.replaceAll("^/+", "").replaceAll("/+$", "").trim();
    }

    public static String getFileSystemType(String mountPoint, boolean checkDirectory) {
        File path = new File(mountPoint);
        if (!path.exists()) {
            Utility.errorMessage("The specified path '" + String.valueOf(path) + "' does not exist.");
            return "";
        }
        if (checkDirectory && !path.isDirectory()) {
            Utility.errorMessage("The specified path '" + String.valueOf(path) + "' is not a directory.");
            return "";
        }
        try {
            FileStore store = Files.getFileStore(Paths.get(mountPoint, new String[0]));
            return store.type();
        }
        catch (IOException e) {
            Utility.errorMessage("Error checking filesystem type of the path '" + mountPoint + "' : " + e.getMessage());
            return "";
        }
    }

    public static String getPartitionUUID(String mountPoint) {
        Output uuidOutput = Utility.getTextFromCommand("findmnt -no PARTUUID --mountpoint " + mountPoint, true, true);
        if (uuidOutput.isSuccess()) {
            return uuidOutput.text().get(0);
        }
        return "";
    }

    public static void infoMessage(String message) {
        Utility.infoMessage(message, null);
    }

    public static void infoMessage(String message, String output) {
        if (output == null || output.isEmpty()) {
            System.out.println(String.valueOf((Object)ConsoleColor.WHITE_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
        } else if (!Config.QUIET) {
            System.out.println(String.valueOf((Object)ConsoleColor.GREEN_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
        }
    }

    public static void warnMessage(String message) {
        Utility.warnMessage(message, null);
    }

    public static void warnMessage(String message, String output) {
        if (output == null || output.isEmpty()) {
            System.out.println(String.valueOf((Object)ConsoleColor.YELLOW_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
        } else {
            System.out.println(String.valueOf((Object)ConsoleColor.YELLOW_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
        }
    }

    public static void errorMessage(String message) {
        Utility.errorMessage(message, null);
    }

    public static void errorMessage(String message, String output) {
        if (output == null || output.isEmpty()) {
            System.err.println(String.valueOf((Object)ConsoleColor.RED_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET));
            Utility.loggerSend(message, LogLevel.ERROR);
        } else {
            System.err.println(String.valueOf((Object)ConsoleColor.RED_BOLD) + message + String.valueOf((Object)ConsoleColor.RESET) + " " + output);
            Utility.loggerSend(message + " " + output, LogLevel.ERROR);
        }
    }

    public static void exitWithError(String message) {
        Utility.exitWithError(message, null);
    }

    public static void exitWithError(String message, String output) {
        Utility.errorMessage(message, output);
        System.exit(1);
    }

    public static String removeSurrounding(String str, String delimiter) {
        if (str != null && str.length() >= 2 && str.startsWith(delimiter) && str.endsWith(delimiter)) {
            return str.substring(delimiter.length(), str.length() - delimiter.length());
        }
        return str;
    }

    public static boolean startsWithIgnoreCase(String str, String search) {
        if (str == null || search == null) {
            return false;
        }
        return str.toLowerCase().startsWith(search.toLowerCase());
    }
}

