#!/bin/sh

set -e -u

# NOTE: why there are "|| return 1" in functions?
# NOTE: see the result of "set -e; a() { set -e; echo before; false; echo after; }; echo begin_all; a || echo failed; echo after_all"
# NOTE: it is "beginall\nbefore\nafter\nafterall" instead of "begin_all\nbefore\n"

# TODO test: re-download on checksum mismatch

echo "~DEBUG: in deploy script"

EXIT_SUCCESS=0
EXIT_NO_DISK_SPACE=82
EXIT_DOWNLOAD_ERROR=83
EXIT_LAUNCH_ERROR=84
EXIT_SELF_CHECK_ERROR=85
EXIT_JAVA_MISSING_ERROR=86
EXIT_UNSUPPORTED_OS=87
EXIT_UNTAR_ERROR=88
EXIT_TAR_MISSING=89
EXIT_AGENT_UNAVAILABLE=90

# usage: die EXIT_CODE ERROR_MESSAGE
die() {
  echo "~ERROR: $2"
  eval "error_code=\${$1}"
  echo "~EXIT-CODE: $error_code"
  exit $error_code
}

# download_file https://www.google.com google.com.html cd00904a734a97409ea94b3cc1083f05e5c90087c164b9fac40b5a324dc746bc
download_file() {
  case "$1" in
    "http://"*) true ;;
    "https://"*) true ;;
    "inline://"*) return 1 ;; # handled by download_file_inline anyway
    *) die EXIT_DOWNLOAD_ERROR "Unsupported url scheme: $1" ;;
  esac

  if command -v curl >/dev/null 2>&1; then
    curl --fail --progress-bar --show-error --location --output "$2.tmp.$$" "$1" || return 1
    check_sha256 "$1" "$2.tmp.$$" "$3" || {
      die EXIT_DOWNLOAD_ERROR "Checksum checking failed for $1, could not continue"
    }
    mv -f "$2.tmp.$$" "$2" || return 1
  elif command -v wget >/dev/null 2>&1; then
    wget -O "$2.tmp.$$" "$1" || return 1
    check_sha256 "$1" "$2.tmp.$$" "$3" || {
      die EXIT_DOWNLOAD_ERROR "Checksum checking failed for $1, could not continue"
    }
    mv -f "$2.tmp.$$" "$2" || return 1
  else
    echo "~WARN: both 'wget' and 'curl' were not found"
    return 1
  fi
}

untar() {
  if command -v tar >/dev/null 2>&1; then
    mkdir -p "$2"
    tar -C "$2" --strip-components 1 -xzf "$1" || {
      die EXIT_UNTAR_ERROR "Failed to extract file $1, could not continue"
    }
    touch "$2/.extracted"
  else
    die EXIT_TAR_MISSING "'tar' command is required, could not continue"
  fi
}

# usage: check_sha256 SOURCE_MONIKER FILE SHA256CHECKSUM
# $1 SOURCE_MONIKER (e.g. url)
# $2 FILE
# $3 SHA256 hex string
check_sha256() {
  echo "~DEBUG: verifying sha256 of '$1' to be '$3'"
  if command -v shasum >/dev/null 2>&1; then
    echo "$3  $2" | shasum --binary -a 256 --status -c || {
      echo "~WARN: ! Checksum mismatch for $2, file $2 was downloaded from $1"
      return 1
    }
    return 0
  fi

  if command -v sha256sum >/dev/null 2>&1; then
    echo "$3  $2" | sha256sum -w -c || {
      echo "~WARN: ! Checksum mismatch for $2, file $2 was downloaded from $1"
      return 1
    }
    return 0
  fi

  echo "Both 'shasum' and 'sha256sum' utilities are missing. Please install one of them"
  return 1
}

# download_file_inline URL DESTINATION_FILE SHA256
download_file_inline() {
  echo "~STATE: download file $1:::$2.tmp.$$"

  true >"$2.tmp.$$" || return 1 # create/truncate the file
  while IFS= read -r line; do
    case "$line" in
      "~END DOWNLOAD FILE $1"*) break ;;
    esac

    # TODO it's disabled it for now since it's too slow in practice (via pty)
    # let's write binary data with printf
    # printf supports \ooo format to output a single byte
    # printf "$line" >>"$2.tmp.$$" || return 1
  done

  test -s "$2.tmp.$$" || {
    die EXIT_DOWNLOAD_ERROR "File was not found at $2.tmp.$$"
  }
  check_sha256 "$1" "$2.tmp.$$" "$3" || {
    die EXIT_DOWNLOAD_ERROR "Checksum checking failed for $1, could not continue"
  }
  mv -f "$2.tmp.$$" "$2" || return 1
}

check_disk_space() {
  free_mb=$(df -m "$1" | tail -n 1 | awk '{print $4}')
  # min_free_mb is set by deployer
  if [ "$free_mb" -lt "$min_free_mb" ]; then
    die EXIT_NO_DISK_SPACE "Error: Free disk space under '$1' is less than $min_free_mb MB. Only $free_mb MB is free"
  fi
}

source_env_file() {
  if test -s "$1"; then
    echo "~DEBUG: sourcing env override file at $1"
    # shellcheck disable=SC1090
    . "$1"
  else
    echo "~DEBUG: env override file is missing at $1, not sourcing"
  fi
}

# variables which are expected to be set
# cli_version_$OS_$ARCH= # could be version or a checksum (for cli built locally)
# cli_sha256_$OS_$ARCH=
# cli_url_$OS_$ARCH=
# jbr_url_$OS_$ARCH=
# jbr_sha256_$OS_$ARCH=
# jbr_name_$OS_$ARCH= # name should be unique across version, platform, arch, jbr type, e.g. jbr-17.0.8-windows-x64-b1000.8
# min_free_mb=
# should_idle_at_end=

# optional variables
# cli_debug_port=
# cli_debug_suspend=
# agent_debug_port=
# agent_debug_suspend=
# async_profiler_path=
# async_profiler_snapshots_dir=
# toolbox_log_level_option=

# sanitize output just in case
uname_system=$(uname -s | head -n1 | sed 's/[^a-zA-Z_0-9]/_/g')
uname_arch=$(uname -m | head -n1 | sed 's/[^a-zA-Z_0-9]/_/g')

# obtaining cli_url before sourcing env files is intentional. env files can override links to cli
eval cli_url="\${cli_url_${uname_system}_${uname_arch}:-unsupported}"
eval cli_sha256="\${cli_sha256_${uname_system}_${uname_arch}:-unsupported}"
eval cli_version="\${cli_version_${uname_system}_${uname_arch}:-unsupported}"

case "$uname_system" in
  Darwin)
    system_wide_env_override_file="/Library/Application Support/JetBrains/ToolboxSshDeploy/env.sh";;
  Linux)
    system_wide_env_override_file="/etc/xdg/JetBrains/ToolboxSshDeploy/env.sh";;
esac

case "$uname_system" in
  Darwin)
    local_env_override_file="$HOME/Library/Application Support/JetBrains/ToolboxSshDeploy/env.sh";;
  Linux)
    if [ -n "${XDG_CONFIG_HOME:-}" ]; then
      local_env_override_file="$XDG_CONFIG_HOME/JetBrains/ToolboxSshDeploy/env.sh"
    else
      local_env_override_file="$HOME/.config/JetBrains/ToolboxSshDeploy/env.sh"
    fi;;
esac

source_env_file "$local_env_override_file"
source_env_file "$system_wide_env_override_file"

if [ -n "${cache_dir:-}" ]; then
  echo "~DEBUG: cache_dir was overridden: $cache_dir"
else
  case "$uname_system" in
    Darwin) cache_dir="$HOME/Library/Caches/JetBrains/Toolbox-CLI-dist" ;;
    Linux) cache_dir="$HOME/.cache/JetBrains/Toolbox-CLI-dist" ;;
    *) die EXIT_UNSUPPORTED_OS "Unsupported system: $uname_system" ;;
  esac
fi

mkdir -p "$cache_dir"

if [ -n "${cli_dir_path:-}" ]; then
  echo "~DEBUG: cli directory path is already set to $cli_dir_path"
else
  cli_dir_path="$cache_dir/tbcli-${cli_version}"
  cli_archive_path="$cli_dir_path.tar.gz"
  echo "~DEBUG: Checking for already downloaded CLI at $cli_dir_path"
  if test -d "$cli_dir_path" && test -e "$cli_dir_path/.extracted"; then
    ls -la "$cli_dir_path"
    echo "~DEBUG: Using already downloaded Toolbox Agent: $cli_dir_path"
  else
    if test -d "$cli_dir_path"; then
      echo "~DEBUG: cli directory $cli_dir_path exists but is corrupted, removing"
      rm -rf "$cli_dir_path"
    fi
    check_disk_space "$cache_dir"
    if [ -z "${cli_url+x}" ] || [ "$cli_url" = "unsupported" ]; then
      die EXIT_AGENT_UNAVAILABLE "Toolbox Agent build is unavailable"
    fi

    download_file "$cli_url" "$cli_archive_path" "$cli_sha256" || {
      echo "Downloading of $cli_url failed, falling back to inline download"
      download_file_inline "$cli_url" "$cli_archive_path" "$cli_sha256"
    }
    untar "$cli_archive_path" "$cli_dir_path"
    rm -f "$cli_archive_path"
  fi
fi

cli_path="$cli_dir_path/bin/tbcli"

echo "~DEBUG: searching for JVM"

if [ -n "${java_path:-}" ]; then
  echo "~DEBUG: java path is already set to $java_path"
else
  java_path=
  for search_dir in "$cli_dir_path/jre/Contents/Home" "$cli_dir_path/jre" /usr/lib/jvm/*; do
    echo "~DEBUG: searching for JVM in $search_dir"
    test -d "$search_dir" || continue

    # we do not handle paths with spaces here
    # let's say, it's user's problem if they store java in a directory with spaces in the name
    # shellcheck disable=SC2044
    for java_temp_path in $(find "$search_dir" -follow -type d); do
      java_cmd="$java_temp_path/bin/java"
      test -x "$java_cmd" || continue

      echo "~DEBUG: trying $java_cmd"
      TB_JAVA_HOME=$java_temp_path "$cli_path" --structured-logging self-check && {
        java_path="$java_temp_path"
        break
      }
    done

    if [ -n "$java_path" ]; then
      break
    fi
  done
fi

if [ -n "$java_path" ]; then
  echo "~DEBUG: Using JVM from the system: $java_path"
else
  eval jbr_url="\${jbr_url_${uname_system}_${uname_arch}:-unsupported}"
  eval jbr_sha256="\${jbr_sha256_${uname_system}_${uname_arch}:-unsupported}"
  eval jbr_name="\${jbr_name_${uname_system}_${uname_arch}:-unsupported}"

  if [ "$jbr_url" = "unsupported" ] || [ "$jbr_sha256" = "unsupported" ] || [ "$jbr_name" = "unsupported" ]; then
    die EXIT_UNSUPPORTED_OS "Combination of $uname_system and $uname_arch is not supported"
  fi

  check_disk_space "$cache_dir"

  extract_code_version=1 # increment it if you want to invalidate all existing jbr downloads (e.g. download or extract code was changed in incompatible way)
  java_path="$cache_dir/$jbr_name-$extract_code_version"

  case "$uname_system" in
    Darwin) java_exe_path="$java_path/Contents/Home/bin/java" ;;
    Linux) java_exe_path="$java_path/bin/java" ;;
    *) die EXIT_UNSUPPORTED_OS "Unsupported system: $uname_system" ;;
  esac

  if [ -f "$java_path/.extracted" ] && [ -x "$java_exe_path" ]; then
    echo "~DEBUG: Using already downloaded JVM at $java_path"
    case "$uname_system" in
      Darwin) java_path="$java_path/Contents/Home" ;;
    esac
  else
    echo "~DEBUG: Downloading JVM from $jbr_url"

    if [ "${jbr_url#file://}" != "$jbr_url" ]; then
      # jbr_url is a file and it was provided externally (by toolbox)
      archive_path="${jbr_url#file://}"
    else
      # need to download
      archive_path="$cache_dir/$jbr_name-$extract_code_version.download.tmp.$$"
      download_file "$jbr_url" "$archive_path" "$jbr_sha256" || {
        echo "Downloading of $cli_url failed, falling back to inline download"
        download_file_inline "$jbr_url" "$archive_path" "$jbr_sha256"
      }
    fi

    rm -rf "$java_path"
    mkdir -p "$java_path"
    tar -x -C "$java_path" --strip-components 1 -z -f "$archive_path"
    rm -f "$archive_path"
    if [ -x "$java_exe_path" ]; then
      echo "~DEBUG: found java executable at $java_exe_path"
    else
      ls -lR "$java_path"
      die EXIT_JAVA_MISSING_ERROR "Unable to find java executable under $java_path, see available files listing in debug output"
    fi
    touch "$java_path/.extracted"

    # Patch the java directory for MacOS
    case "$uname_system" in
      Darwin) java_path="$java_path/Contents/Home" ;;
    esac
  fi

  TB_JAVA_HOME="$java_path" "$cli_path" self-check || {
    die EXIT_SELF_CHECK_ERROR "Self check failed"
  }
fi

export LC_ALL=C.UTF-8

set -- "$cli_path"

if [ -n "${toolbox_log_level_option:-}" ]; then
  set -- "$@" "--$toolbox_log_level_option"
  export TB_AGENT_LOG_LEVEL="$toolbox_log_level_option"
fi

if [ "${forceful_deploy}" = "true" ]; then
  set -- "$@" "--run-as-main-instance"
fi

if [ "${offline_mode}" = "true" ]; then
  set -- "$@" "--offline-mode"
fi

set -- "$@" "--structured-logging"
set -- "$@" "agent"

echo "~DEBUG: App-level options set: " "$@"

echo "~END DEPLOY SCRIPT"

# Launcher-level environment variables
export TB_JAVA_HOME="$java_path"

# Toolbox-level environment variables
export TB_CLI_PATH="$cli_path"
export TB_CLI_HASH="${cli_sha256:-}"
# We need to distinguish debug port for proxy and debug port for agent
export TB_AGENT_DEBUG_PORT="${agent_debug_port:-}"
export TB_AGENT_DEBUG_SUSPEND="${agent_debug_suspend:-}"
export TB_DEBUG_PORT="${cli_debug_port:-}"
export TB_DEBUG_SUSPEND="${cli_debug_suspend:-}"
# We need to export information about async profiler location to toolbox_cli runs
export TB_ASYNC_PROFILER_PATH="${async_profiler_path:-}"
export TB_ASYNC_PROFILER_SNAPSHOTS_DIR="${async_profiler_snapshots_dir:-}"

case "$uname_system" in
  Darwin)
    # https://stackoverflow.com/questions/36590905/is-setsid-command-missing-on-os-x/
    # https://gist.github.com/negokaz/d561fb3ccf113958862bd5ff4e7b0f6e
    setsid_script="use POSIX setsid;setsid() or die \"setsid failed: \${'$'}!\";exec @ARGV;"
    LC_ALL="C" LANG="C" "/usr/bin/perl" -e "$setsid_script" "$@";;
  Linux) setsid "$@";;
esac

# We need to sleep for 10s which is the time required by the agent to startup and print the connection details
# SSH needs to stay alive until we forward the port - then, SSH will live until forwarding is finished
sleep 10
