/*
 * Copyright (c) 2024, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package sun.security.x509;

import sun.security.util.BitArray;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectInputStream;
import java.io.Serial;
import java.security.InvalidKeyException;
import java.security.KeyRep;
import java.security.NoSuchAlgorithmException;
import java.security.ProviderException;
import java.security.spec.NamedParameterSpec;

/// Represents a public key from an algorithm family that is specialized
/// with a named parameter set.
///
/// This key is generated by either a [sun.security.provider.NamedKeyPairGenerator]
/// or [sun.security.provider.NamedKeyFactory]. Its [#getAlgorithm] method
/// returns the algorithm family name, while its [#getParams()] method returns
/// the parameter set name as a [NamedParameterSpec] object. The algorithm
/// identifier in the X.509 encoding of the key is always a single OID derived
/// from the parameter set name.
///
/// @see sun.security.provider.NamedKeyPairGenerator
public final class NamedX509Key extends X509Key {
    @Serial
    private static final long serialVersionUID = 1L;

    private final String fname;
    private final transient NamedParameterSpec paramSpec;
    private final byte[] rawBytes;

    /// Ctor from family name, parameter set name, raw key bytes.
    /// Key bytes won't be cloned, caller must relinquish ownership
    public NamedX509Key(String fname, String pname, byte[] rawBytes) {
        this.fname = fname;
        this.paramSpec = new NamedParameterSpec(pname);
        try {
            this.algid = AlgorithmId.get(pname);
        } catch (NoSuchAlgorithmException e) {
            throw new ProviderException(e);
        }
        this.rawBytes = rawBytes;

        setKey(new BitArray(rawBytes.length * 8, rawBytes));
    }

    /// Ctor from family name, and X.509 bytes
    public NamedX509Key(String fname, byte[] encoded) throws InvalidKeyException {
        this.fname = fname;
        decode(encoded);
        this.paramSpec = new NamedParameterSpec(algid.getName());
        if (algid.encodedParams != null) {
            throw new InvalidKeyException("algorithm identifier has params");
        }
        this.rawBytes = getKey().toByteArray();
    }

    @Override
    public String toString() {
        // Do not modify: this can be used by earlier JDKs that
        // do not have the getParams() method
        return paramSpec.getName() + " public key";
    }

    /// Returns the reference to the internal key. Caller must not modify
    /// the content or keep a reference.
    public byte[] getRawBytes() {
        return rawBytes;
    }

    @Override
    public NamedParameterSpec getParams() {
        return paramSpec;
    }

    @Override
    public String getAlgorithm() {
        return fname;
    }

    @java.io.Serial
    private Object writeReplace() throws java.io.ObjectStreamException {
        return new KeyRep(KeyRep.Type.PUBLIC, getAlgorithm(), getFormat(),
                getEncoded());
    }

    @java.io.Serial
    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {
        throw new InvalidObjectException(
                "NamedX509Key keys are not directly deserializable");
    }
}
