"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const color_1 = tslib_1.__importDefault(require("@heroku-cli/color"));
const command_1 = require("@heroku-cli/command");
const core_1 = require("@oclif/core");
const base_1 = tslib_1.__importDefault(require("../../../lib/base"));
class Info extends base_1.default {
    static description = 'get current status of a specific AI model resource or all AI model resources attached to an app';
    static examples = [
        'heroku ai:models:info claude-3-5-sonnet-acute-04281 --app example-app ',
        'heroku ai:models:info --app example-app ',
    ];
    static flags = {
        app: command_1.flags.app({ required: true }),
        remote: command_1.flags.remote(),
    };
    static args = {
        model_resource: core_1.Args.string({ description: 'resource ID or alias of model resource ' }),
    };
    async run() {
        const { args, flags } = await this.parse(Info);
        const { app } = flags;
        const { model_resource: modelResource } = args;
        const synthesizedModels = [];
        let listOfProvisionedModels = [];
        const modelInfo = async () => {
            const modelInfoResponse = await this.herokuAI.get(`/models/${this.addon.id}`, {
                headers: { authorization: `Bearer ${this.apiKey}` },
            })
                .catch(error => {
                if (error.statusCode === 404) {
                    core_1.ux.warn(`We can’t find a model resource called ${color_1.default.yellow(modelResource)}.\nRun ${color_1.default.cmd('heroku ai:models:info -a <app>')} to see a list of model resources.`);
                }
                else {
                    throw error;
                }
            });
            return modelInfoResponse;
        };
        const addModelProperties = (alias, resourceId, modelResource = {}) => {
            return Object.assign(modelResource, { model_alias: alias, model_resource_id: resourceId });
        };
        const getModelDetails = async (collectedModels) => {
            if (typeof collectedModels === 'string') {
                const modelResource = collectedModels;
                await this.configureHerokuAIClient(modelResource, app);
                let { body: currentModelResource } = await modelInfo() ?? { body: {} };
                currentModelResource = addModelProperties(this.modelAlias, this.addonResourceId, currentModelResource);
                synthesizedModels.push(currentModelResource);
            }
            else {
                for (const addonModel of collectedModels) {
                    await this.configureHerokuAIClient(addonModel.modelResource, app);
                    let { body: currentModelResource } = await modelInfo() ?? { body: {} };
                    currentModelResource = addModelProperties(this.modelAlias, this.addonResourceId, currentModelResource);
                    synthesizedModels.push(currentModelResource);
                }
            }
            return synthesizedModels;
        };
        if (modelResource) {
            listOfProvisionedModels = await getModelDetails(modelResource);
        }
        else {
            const provisionedModelsInfo = [];
            const inferenceRegex = /inference/;
            const { body: addonsResponse } = await this.heroku.get(`/apps/${app}/addons`, {
                headers: { 'Accept-Expansion': 'plan' },
            });
            for (const addonInfo of addonsResponse) {
                const addonType = addonInfo.addon_service?.name || '';
                const isModelAddon = inferenceRegex.test(addonType);
                if (isModelAddon) {
                    provisionedModelsInfo.push({
                        addonName: addonInfo.addon_service?.name,
                        modelResource: addonInfo.name,
                        modelId: addonInfo.addon_service?.id,
                    });
                }
            }
            listOfProvisionedModels = await getModelDetails(provisionedModelsInfo);
        }
        this.displayModelResource(listOfProvisionedModels);
    }
    displayModelResource(modelResources) {
        for (const modelResource of modelResources) {
            core_1.ux.log();
            core_1.ux.styledHeader(modelResource.model_id);
            core_1.ux.styledObject({
                'Base Model ID': modelResource.model_id,
                'Model Alias': modelResource.model_alias,
                'Model Resource ID': modelResource.model_resource_id,
                Ready: modelResource.ready,
                'Avg Performance': modelResource.avg_performance,
            });
        }
    }
}
exports.default = Info;
