"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const color_1 = tslib_1.__importDefault(require("@heroku-cli/color"));
const command_1 = require("@heroku-cli/command");
const core_1 = require("@oclif/core");
const util_1 = require("../../../lib/ai/models/util");
const base_1 = tslib_1.__importDefault(require("../../../lib/base"));
class Attach extends base_1.default {
    static args = {
        model_resource: core_1.Args.string({
            description: 'resource ID or alias of model resource to attach',
            required: true,
        }),
    };
    static description = 'attach an existing model resource to an app';
    static examples = [
        'heroku ai:models:attach claude-3-5-sonnet-acute-41518 --source-app example-source-app --target-app example-target-app',
        'heroku ai:models:attach claude-3-5-sonnet-acute-41518 --source-app example-source-app --target-app example-target-app --as MY_CS35',
    ];
    static flags = {
        as: command_1.flags.string({ description: 'alias name for model resource' }),
        confirm: command_1.flags.string({ description: 'overwrite existing attached resource with same name' }),
        'source-app': command_1.flags.string({ char: 's', description: 'source app for model resource', required: true }),
        'target-app': command_1.flags.app({ char: 't', description: 'target app for model resource', required: true }),
        remote: command_1.flags.remote({ description: 'git remote of target app' }),
    };
    async run() {
        const { flags, args } = await this.parse(Attach);
        const { model_resource: modelResource } = args;
        const { as, confirm } = flags;
        const sourceApp = flags['source-app'];
        const targetApp = flags['target-app'];
        await this.configureHerokuAIClient(modelResource, sourceApp);
        const attachment = await (0, util_1.trapConfirmationRequired)(targetApp, confirm, (confirmed) => this.createAttachment(targetApp, as, confirmed));
        core_1.ux.action.start(`Setting ${color_1.default.attachment(attachment.name || '')} config vars and restarting ${color_1.default.app(targetApp)}`);
        const { body: releases } = await this.heroku.get(`/apps/${targetApp}/releases`, {
            partial: true, headers: { Range: 'version ..; max=1, order=desc' },
        });
        core_1.ux.action.stop(`done, v${releases[0].version}`);
    }
    async createAttachment(app, as, confirmed) {
        const body = {
            name: as, app: { name: app }, addon: { name: this.addon.name }, confirm: confirmed,
        };
        core_1.ux.action.start(`Attaching ${color_1.default.addon(this.addon.name || '')}${as ? ' as ' + color_1.default.attachment(as) : ''} to ${color_1.default.app(app)}`);
        const { body: attachment } = await this.heroku.post('/addon-attachments', { body }).catch(error => {
            core_1.ux.action.stop('');
            (0, util_1.handlePlatformApiErrors)(error, { as });
        });
        core_1.ux.action.stop();
        return attachment;
    }
}
exports.default = Attach;
