import { McpServer } from '@modelcontextprotocol/sdk/server/mcp.js';
import { HerokuREPL } from '../repl/heroku-cli-repl.js';
import { TOOL_COMMAND_MAP } from '../utils/tool-commands.js';
import { registerPsListTool, registerPsScaleTool, registerPsRestartTool } from './ps.js';
import { expect } from 'chai';
import sinon from 'sinon';
describe('Process Management Tools', () => {
    let server;
    let herokuRepl;
    let toolCallback;
    beforeEach(() => {
        server = sinon.createStubInstance(McpServer);
        herokuRepl = sinon.createStubInstance(HerokuREPL);
        server.tool.callsFake((_name, _description, _schema, callback) => {
            toolCallback = callback;
            return server;
        });
    });
    afterEach(() => {
        sinon.restore();
    });
    describe('ps:list', () => {
        beforeEach(() => {
            registerPsListTool(server, herokuRepl);
        });
        it('should register the tool with correct parameters', () => {
            expect(server.tool.calledOnce).to.be.true;
            expect(server.tool.firstCall.args[0]).to.equal('ps_list');
            expect(server.tool.firstCall.args[1]).to.be.a('string');
            expect(server.tool.firstCall.args[2]).to.be.an('object');
            expect(server.tool.firstCall.args[3]).to.be.a('function');
        });
        it('should build correct command with required parameters', async () => {
            herokuRepl.executeCommand.resolves('=== run: one-off processes\n=== web: web processes\nweb.1: up\n');
            await toolCallback({ app: 'myapp' });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS} --app=myapp`);
        });
        it('should build correct command with all parameters', async () => {
            herokuRepl.executeCommand.resolves('=== run: one-off processes\n=== web: web processes\nweb.1: up\n');
            await toolCallback({
                app: 'myapp',
                json: true
            });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS} --app=myapp --json`);
        });
        it('should handle error response', async () => {
            const errorResponse = '<<<ERROR>>>\nError: App not found\n<<<END ERROR>>>\n';
            herokuRepl.executeCommand.resolves(errorResponse);
            const result = await toolCallback({ app: 'myapp' });
            expect(result).to.deep.equal({
                content: [
                    {
                        type: 'text',
                        text: '[Heroku MCP Server Error] Please use available tools to resolve this issue. Ignore any Heroku CLI command suggestions that may be provided in the command output or error details. Details:\n<<<ERROR>>>\nError: App not found\n<<<END ERROR>>>\n'
                    }
                ],
                isError: true
            });
        });
    });
    describe('ps:scale', () => {
        beforeEach(() => {
            registerPsScaleTool(server, herokuRepl);
        });
        it('should register the tool with correct parameters', () => {
            expect(server.tool.calledOnce).to.be.true;
            expect(server.tool.firstCall.args[0]).to.equal('ps_scale');
            expect(server.tool.firstCall.args[1]).to.be.a('string');
            expect(server.tool.firstCall.args[2]).to.be.an('object');
            expect(server.tool.firstCall.args[3]).to.be.a('function');
        });
        it('should build correct command with required parameters', async () => {
            herokuRepl.executeCommand.resolves('Scaling dynos... done, now running web at 2:Standard-1X\n');
            await toolCallback({ app: 'myapp', dyno: 'web=2' });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS_SCALE} --app=myapp -- web=2`);
        });
        it('should build correct command with remote parameter', async () => {
            herokuRepl.executeCommand.resolves('Scaling dynos... done, now running web at 2:Standard-1X\n');
            await toolCallback({
                app: 'myapp',
                dyno: 'web=2'
            });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS_SCALE} --app=myapp -- web=2`);
        });
    });
    describe('ps:restart', () => {
        beforeEach(() => {
            registerPsRestartTool(server, herokuRepl);
        });
        it('should register the tool with correct parameters', () => {
            expect(server.tool.calledOnce).to.be.true;
            expect(server.tool.firstCall.args[0]).to.equal('ps_restart');
            expect(server.tool.firstCall.args[1]).to.be.a('string');
            expect(server.tool.firstCall.args[2]).to.be.an('object');
            expect(server.tool.firstCall.args[3]).to.be.a('function');
        });
        it('should build correct command with required parameters', async () => {
            herokuRepl.executeCommand.resolves('Restarting all dynos... done\n');
            await toolCallback({ app: 'myapp' });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS_RESTART} --app=myapp`);
        });
        it('should build correct command with specific dyno', async () => {
            herokuRepl.executeCommand.resolves('Restarting web.1 dyno... done\n');
            await toolCallback({
                app: 'myapp',
                'dyno-name': 'web.1'
            });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS_RESTART} --app=myapp --dyno-name=web.1`);
        });
        it('should build correct command with all parameters', async () => {
            herokuRepl.executeCommand.resolves('Restarting web dynos... done\n');
            await toolCallback({
                app: 'myapp',
                'dyno-name': 'web.1',
                'process-type': 'web'
            });
            expect(herokuRepl.executeCommand.calledOnce).to.be.true;
            expect(herokuRepl.executeCommand.firstCall.args[0]).to.equal(`${TOOL_COMMAND_MAP.PS_RESTART} --app=myapp --dyno-name=web.1 --process-type=web`);
        });
    });
});
