import { z } from 'zod';
import { handleCliOutput } from '../utils/handle-cli-output.js';
import { CommandBuilder } from '../utils/command-builder.js';
import { TOOL_COMMAND_MAP } from '../utils/tool-commands.js';
/**
 * Schema for creating a new pipeline.
 */
export const pipelinesCreateOptionsSchema = z.object({
    name: z.string().describe('Pipeline name'),
    stage: z.enum(['development', 'staging', 'production']).describe('Initial pipeline stage'),
    app: z.string().optional().describe('App to add to pipeline'),
    team: z.string().optional().describe('Team owning the pipeline')
});
/**
 * Schema for promoting apps through a pipeline.
 */
export const pipelinesPromoteOptionsSchema = z.object({
    app: z.string().describe('Source app for promotion'),
    to: z.string().optional().describe('Target apps for promotion (comma-separated)')
});
/**
 * Schema for listing pipelines.
 */
export const pipelinesListOptionsSchema = z.object({
    json: z.boolean().optional().describe('Enable JSON output')
});
/**
 * Schema for getting pipeline information.
 */
export const pipelinesInfoOptionsSchema = z.object({
    pipeline: z.string().describe('Target pipeline name'),
    json: z.boolean().optional().describe('Enable JSON output')
});
/**
 * Schema for the base pipelines command.
 */
export const pipelinesOptionsSchema = z.object({
    json: z.boolean().optional().describe('Enable JSON output')
});
/**
 * Registers the pipelines:create tool with the MCP server.
 *
 * @param server - The MCP server instance.
 * @param herokuRepl - The Heroku REPL instance.
 */
export const registerPipelinesCreateTool = (server, herokuRepl) => {
    server.tool('pipelines_create', 'Creates new Heroku deployment pipeline with configurable stages, apps, and team settings', pipelinesCreateOptionsSchema.shape, async (options) => {
        const command = new CommandBuilder(TOOL_COMMAND_MAP.PIPELINES_CREATE)
            .addFlags({
            stage: options.stage,
            app: options.app,
            team: options.team
        })
            .addPositionalArguments({ name: options.name })
            .build();
        const output = await herokuRepl.executeCommand(command);
        return handleCliOutput(output);
    });
};
/**
 * Registers the pipelines:promote tool with the MCP server.
 *
 * @param server - The MCP server instance.
 * @param herokuRepl - The Heroku REPL instance.
 */
export const registerPipelinesPromoteTool = (server, herokuRepl) => {
    server.tool('pipelines_promote', 'Promotes apps between pipeline stages with configurable target applications', pipelinesPromoteOptionsSchema.shape, async (options) => {
        const command = new CommandBuilder(TOOL_COMMAND_MAP.PIPELINES_PROMOTE)
            .addFlags({
            app: options.app,
            to: options.to
        })
            .build();
        const output = await herokuRepl.executeCommand(command);
        return handleCliOutput(output);
    });
};
/**
 * Registers the pipelines:list tool with the MCP server.
 *
 * @param server - The MCP server instance.
 * @param herokuRepl - The Heroku REPL instance.
 */
export const registerPipelinesListTool = (server, herokuRepl) => {
    server.tool('pipelines_list', 'Lists accessible Heroku pipelines with ownership and configuration details', pipelinesListOptionsSchema.shape, async (options) => {
        const command = new CommandBuilder(TOOL_COMMAND_MAP.PIPELINES)
            .addFlags({
            json: options.json
        })
            .build();
        const output = await herokuRepl.executeCommand(command);
        return handleCliOutput(output);
    });
};
/**
 * Registers the pipelines:info tool with the MCP server.
 *
 * @param server - The MCP server instance.
 * @param herokuRepl - The Heroku REPL instance.
 */
export const registerPipelinesInfoTool = (server, herokuRepl) => {
    server.tool('pipelines_info', 'Displays detailed pipeline configuration, stages, and connected applications', pipelinesInfoOptionsSchema.shape, async (options) => {
        const command = new CommandBuilder(TOOL_COMMAND_MAP.PIPELINES_INFO)
            .addFlags({
            json: options.json
        })
            .addPositionalArguments({ pipeline: options.pipeline })
            .build();
        const output = await herokuRepl.executeCommand(command);
        return handleCliOutput(output);
    });
};
/**
 * Registers the base pipelines tool with the MCP server.
 *
 * @param server - The MCP server instance.
 * @param herokuRepl - The Heroku REPL instance.
 */
export const registerPipelinesTool = (server, herokuRepl) => {
    server.tool('pipelines', 'Lists and manages Heroku pipelines with configuration visibility', pipelinesOptionsSchema.shape, async (options) => {
        const command = new CommandBuilder('pipelines')
            .addFlags({
            json: options.json
        })
            .build();
        const output = await herokuRepl.executeCommand(command);
        return handleCliOutput(output);
    });
};
