/**
 * [Heroku Platform API - Dyno](https://devcenter.heroku.com/articles/platform-api-reference#dyno)
 * Dynos encapsulate running processes of an app on Heroku. Detailed information about dyno sizes can be found at: [https://devcenter.heroku.com/articles/dyno-types](https://devcenter.heroku.com/articles/dyno-types).
 */
export default class DynoService {
    endpoint;
    /**
     * @param endpoint The endpoint to the API.
     */
    constructor(endpoint) {
        this.endpoint = endpoint;
    }
    /**
     * Create a new dyno.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param payload Object to send to the endpoint.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async create(appIdentity, payload, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos`, {
            ...requestInit,
            body: JSON.stringify(payload, null, 2),
            method: 'POST',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Restart dyno.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param dynoIdentity unique identifier of this dyno or the name of this process on this dyno.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async restart(appIdentity, dynoIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos/${dynoIdentity}`, {
            ...requestInit,
            method: 'DELETE',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Restart dynos of a given formation type.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param dynoFormationType the formation type of this process on this dyno
     * @example "run".
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async restartFormation(appIdentity, dynoFormationType, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/formations/${dynoFormationType}`, {
            ...requestInit,
            method: 'DELETE',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Restart all dynos.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async restartAll(appIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos`, {
            ...requestInit,
            method: 'DELETE',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Stop dyno.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param dynoIdentity unique identifier of this dyno or the name of this process on this dyno.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async stop(appIdentity, dynoIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos/${dynoIdentity}/actions/stop`, {
            ...requestInit,
            method: 'POST',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Stop dynos of a given formation type.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param dynoFormationType the formation type of this process on this dyno
     * @example "run".
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async stopFormation(appIdentity, dynoFormationType, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/formations/${dynoFormationType}/actions/stop`, {
            ...requestInit,
            method: 'POST',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Info for existing dyno.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param dynoIdentity unique identifier of this dyno or the name of this process on this dyno.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async info(appIdentity, dynoIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos/${dynoIdentity}`, {
            ...requestInit,
            method: 'GET',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * List existing dynos.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param requestInit The initializer for the request.
     * @returns The response from the API.
     */
    async list(appIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/dynos`, {
            ...requestInit,
            method: 'GET',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
}
