import { APIClient } from '@heroku-cli/command';
import type { ExtendedAddonAttachment } from '../../types/pg/data-api';
import type { ConnectionDetails, ConnectionDetailsWithAttachment } from '../../types/pg/tunnel';
import { fetchBastionConfig } from './bastion';
import { getConfig } from './config-vars';
export default class DatabaseResolver {
    private readonly heroku;
    private readonly getConfigFn;
    private readonly fetchBastionConfigFn;
    private readonly addonAttachmentResolver;
    private readonly attachmentHeaders;
    constructor(heroku: APIClient, getConfigFn?: typeof getConfig, fetchBastionConfigFn?: typeof fetchBastionConfig);
    /**
     * Resolves a database attachment based on the provided database identifier
     * (attachment name, id, or config var name) and namespace (credential).
     *
     * @param appId - The ID of the app to get the attachment for
     * @param attachmentId - The database identifier (defaults to 'DATABASE_URL')
     * @param namespace - Optional namespace/credential for the attachment
     * @returns Promise resolving to the database attachment
     * @throws {Error} When no databases exist or when database identifier is unknown
     * @throws {AmbiguousError} When multiple matching attachments are found
     */
    getAttachment(appId: string, attachmentId?: string, namespace?: string): Promise<ExtendedAddonAttachment>;
    /**
     * Returns the connection details for a database attachment resolved through the identifiers passed as
     * arguments: appId, attachmentId and namespace (credential).
     *
     * @param appId - The ID of the app containing the database
     * @param attachmentId - Optional database identifier (defaults to 'DATABASE_URL')
     * @param namespace - Optional namespace/credential for the attachment
     * @returns Promise resolving to connection details with attachment information
     */
    getDatabase(appId: string, attachmentId?: string, namespace?: string): Promise<ConnectionDetailsWithAttachment>;
    /**
     * Parses a PostgreSQL connection string (or a local database name) into a ConnectionDetails object.
     *
     * @param connStringOrDbName - PostgreSQL connection string or local database name
     * @returns Connection details object with parsed connection information
     */
    static parsePostgresConnectionString(connStringOrDbName: string): ConnectionDetails;
    /**
     * Fetches all Heroku PostgreSQL add-on attachments for a given app.
     *
     * This is used internally by the `getAttachment` function to get all valid Heroku PostgreSQL add-on attachments
     * to generate a list of possible valid attachments when the user passes a database name that doesn't match any
     * attachments.
     *
     * @param appId - The ID of the app to get the attachments for
     * @returns Promise resolving to array of PostgreSQL add-on attachments
     */
    private allPostgresAttachments;
    /**
     * Returns the connection details for a database attachment according to the app config vars.
     *
     * @param attachment - The attachment to get the connection details for
     * @param config - The record of app config vars with their values
     * @returns Connection details with attachment information
     */
    getConnectionDetails(attachment: ExtendedAddonAttachment, config?: Record<string, string>): ConnectionDetailsWithAttachment;
    /**
     * Helper function that attempts to find a single addon attachment matching the given database identifier
     * (attachment name, id, or config var name).
     *
     * This is used internally by the `getAttachment` function to handle the lookup of addon attachments.
     * It returns either a single match, multiple matches (for ambiguous cases), or null if no matches are found.
     *
     * The AddonAttachmentResolver uses the Platform API add-on attachment resolver endpoint to get the attachment.
     *
     * @param appId - The ID of the app to search for attachments
     * @param attachmentId - The database identifier to match
     * @param namespace - Optional namespace/credential filter
     * @returns Promise resolving to either a single match, multiple matches with error, or no matches with error
     */
    private matchesHelper;
}
