import type { APIClient } from '@heroku-cli/command';
import { Server } from 'node:net';
import { ExtendedAddonAttachment } from '../../types/pg/data-api';
import { BastionConfig, ConnectionDetails, TunnelConfig } from '../../types/pg/tunnel';
/**
 * Determines whether the attachment belongs to an add-on installed onto a non-shield Private Space.
 * If true, the bastion information needs to be fetched from the Data API.
 * For add-ons installed onto a Shield Private Space, the bastion information should be fetched from config vars.
 *
 * @param attachment - The add-on attachment to check
 * @returns True if the attachment belongs to a non-shield Private Space, false otherwise
 */
export declare function bastionKeyPlan(attachment: ExtendedAddonAttachment): boolean;
/**
 * Fetches the bastion configuration from the Data API (only relevant for add-ons installed onto a
 * non-shield Private Space).
 * For add-ons installed onto a Shield Private Space, the bastion information is stored in the config vars.
 *
 * @param heroku - The Heroku API client
 * @param addon - The add-on information
 * @returns Promise that resolves to the bastion configuration
 */
export declare function fetchBastionConfig(heroku: APIClient, addon: ExtendedAddonAttachment['addon']): Promise<BastionConfig>;
/**
 * Returns the bastion configuration from the config vars for add-ons installed onto Shield
 * Private Spaces.
 *
 * If there are bastions, extracts a host and a key from the config vars.
 * If there are no bastions, returns an empty Object.
 *
 * We assert that _BASTIONS and _BASTION_KEY always exist together.
 * If either is falsy, pretend neither exist.
 *
 * @param config - The configuration variables object
 * @param baseName - The base name for the configuration variables
 * @returns The bastion configuration object
 */
export declare const getBastionConfig: (config: Record<string, string>, baseName: string) => BastionConfig;
/**
 * Returns both the required environment variables to effect the psql command execution and the tunnel
 * configuration according to the database connection details.
 *
 * @param connectionDetails - The database connection details with attachment information
 * @returns Object containing database environment variables and tunnel configuration
 */
export declare function getPsqlConfigs(connectionDetails: ConnectionDetails): {
    dbEnv: NodeJS.ProcessEnv;
    dbTunnelConfig: TunnelConfig;
};
export type PsqlConfigs = ReturnType<typeof getPsqlConfigs>;
/**
 * Establishes an SSH tunnel to the database using the provided configuration.
 *
 * @param connectionDetails - The database connection details with attachment information
 * @param dbTunnelConfig - The tunnel configuration object
 * @param timeout - The timeout in milliseconds (default: 10000)
 * @param createSSHTunnel - The function to create the SSH tunnel
 * @returns Promise that resolves to the tunnel server or null if no bastion key is provided
 * @throws Error if unable to establish the tunnel
 */
export declare function sshTunnel(connectionDetails: ConnectionDetails, dbTunnelConfig: TunnelConfig, timeout?: number, createSSHTunnel?: typeof createSSHTunnelAdapter): Promise<Server | void>;
/**
 * Adapter for tunnel-ssh v5 API. Translates our TunnelConfig into the v5
 * createTunnel(tunnelOptions, serverOptions, sshOptions, forwardOptions) call
 * and returns the created local Server.
 *
 * @param config - The tunnel configuration to translate for v5 API
 * @returns Promise that resolves to the created local TCP Server
 */
declare function createSSHTunnelAdapter(config: TunnelConfig): Promise<Server>;
export {};
