"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const command_1 = require("@heroku-cli/command");
const core_1 = require("@oclif/core");
const kolkrabbi_api_1 = require("../../lib/pipelines/kolkrabbi-api");
class ReviewappsEnable extends command_1.Command {
    async run() {
        const { flags } = await this.parse(ReviewappsEnable);
        if (flags.app) {
            // remove app & remote flags when Review Apps 1.0 is deprecated
            this.warn('Specifying an app via --app or --remote is no longer needed with Review Apps');
        }
        const settings = {
            automatic_review_apps: undefined,
            destroy_stale_apps: undefined,
            wait_for_ci: undefined,
            pipeline: undefined,
            repo: undefined,
        };
        if (flags.autodeploy) {
            this.log('Enabling auto deployment...');
            settings.automatic_review_apps = true;
        }
        if (flags.autodestroy) {
            this.log('Enabling auto destroy...');
            settings.destroy_stale_apps = true;
        }
        if (flags['wait-for-ci']) {
            this.log('Enabling wait for CI...');
            settings.wait_for_ci = true;
        }
        const kolkrabbi = new kolkrabbi_api_1.default(this.config.userAgent, () => this.heroku.auth);
        core_1.ux.action.start('Configuring pipeline');
        const { body: pipeline } = await this.heroku.get(`/pipelines/${flags.pipeline}`);
        settings.pipeline = pipeline.id;
        try {
            const { body: feature } = await this.heroku.get('/account/features/dashboard-repositories-api');
            if (feature.enabled) {
                const { body: repo } = await this.heroku.get(`/pipelines/${pipeline.id}/repo`, {
                    headers: { Accept: 'application/vnd.heroku+json; version=3.repositories-api' },
                });
                settings.repo = repo.full_name;
            }
        }
        catch (_a) {
            const { repository } = await kolkrabbi.getPipelineRepository(pipeline.id);
            settings.repo = repository.name;
        }
        if (flags.autodeploy || flags.autodestroy || flags['wait-for-ci']) {
            await this.heroku.patch(`/pipelines/${pipeline.id}/review-app-config`, {
                body: settings,
                headers: { Accept: 'application/vnd.heroku+json; version=3.review-apps' },
            });
        }
        else {
            // if no flags are passed then the user is enabling review apps
            await this.heroku.post(`/pipelines/${pipeline.id}/review-app-config`, {
                body: settings,
                headers: { Accept: 'application/vnd.heroku+json; version=3.review-apps' },
            });
        }
        core_1.ux.action.stop();
    }
}
exports.default = ReviewappsEnable;
ReviewappsEnable.description = 'enable review apps and/or settings on an existing pipeline';
ReviewappsEnable.examples = [
    '$ heroku reviewapps:enable -p my-pipeline -a my-app --autodeploy --autodestroy',
];
ReviewappsEnable.flags = {
    app: command_1.flags.app({
        description: 'parent app used by review apps',
    }),
    remote: command_1.flags.remote(),
    pipeline: command_1.flags.string({
        char: 'p',
        description: 'name of pipeline',
        required: true,
    }),
    autodeploy: command_1.flags.boolean({
        description: 'autodeploy the review app',
    }),
    autodestroy: command_1.flags.boolean({
        description: 'autodestroy the review app',
    }),
    'wait-for-ci': command_1.flags.boolean({
        description: 'wait for CI to pass before deploying',
    }),
};
