"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tunnel = void 0;
const tslib_1 = require("tslib");
const debug_1 = tslib_1.__importDefault(require("debug"));
const node_child_process_1 = require("node:child_process");
const node_events_1 = require("node:events");
const node_stream_1 = require("node:stream");
const promises_1 = require("node:stream/promises");
const bastion_1 = require("./bastion");
const pgDebug = (0, debug_1.default)('pg');
/**
 * A small wrapper around tunnel-ssh so that other code doesn't have to worry about whether there is or is not a tunnel.
 */
class Tunnel {
    /**
     * Creates a new Tunnel instance.
     *
     * @param bastionTunnel - The SSH tunnel server or void if no tunnel is needed
     */
    constructor(bastionTunnel) {
        this.bastionTunnel = bastionTunnel;
        // eslint-disable-next-line unicorn/prefer-event-target
        this.events = new node_events_1.EventEmitter();
    }
    /**
     * Creates and connects to an SSH tunnel.
     *
     * @param connectionDetails - The database connection details with attachment information
     * @param tunnelConfig - The tunnel configuration object
     * @param tunnelFn - The function to create the SSH tunnel (default: sshTunnel)
     * @returns Promise that resolves to a new Tunnel instance
     */
    static async connect(connectionDetails, tunnelConfig, tunnelFn) {
        const tunnel = await tunnelFn(connectionDetails, tunnelConfig);
        return new Tunnel(tunnel);
    }
    /**
     * Closes the tunnel if it exists, or emits a fake close event if no tunnel is needed.
     *
     * @returns void
     */
    close() {
        if (this.bastionTunnel) {
            pgDebug('close tunnel');
            this.bastionTunnel.close();
        }
        else {
            pgDebug('no tunnel necessary; sending fake close event');
            this.events.emit('close', 0);
        }
    }
    /**
     * Waits for the tunnel to close.
     *
     * @returns Promise that resolves when the tunnel closes
     * @throws Error if the secure tunnel fails
     */
    async waitForClose() {
        if (this.bastionTunnel) {
            try {
                pgDebug('wait for tunnel close');
                await (0, node_events_1.once)(this.bastionTunnel, 'close');
                pgDebug('tunnel closed');
            }
            catch (error) {
                pgDebug('tunnel close error', error);
                throw new Error('Secure tunnel to your database failed');
            }
        }
        else {
            pgDebug('no tunnel required; waiting for fake close event');
            await (0, node_events_1.once)(this.events, 'close');
        }
    }
}
exports.Tunnel = Tunnel;
class PsqlService {
    constructor(connectionDetails, getPsqlConfigsFn = bastion_1.getPsqlConfigs, spawnFn = node_child_process_1.spawn, tunnelFn = bastion_1.sshTunnel) {
        this.connectionDetails = connectionDetails;
        this.getPsqlConfigsFn = getPsqlConfigsFn;
        this.spawnFn = spawnFn;
        this.tunnelFn = tunnelFn;
    }
    /**
     * Executes a PostgreSQL query using the instance's database connection details.
     * It uses the `getPsqlConfigs` function to get the configuration for the database and the tunnel,
     * and then calls the `runWithTunnel` function to execute the query.
     *
     * @param query - The SQL query to execute
     * @param psqlCmdArgs - Additional command-line arguments for psql (default: [])
     * @returns Promise that resolves to the query result as a string
     */
    async execQuery(query, psqlCmdArgs = []) {
        const configs = this.getPsqlConfigsFn(this.connectionDetails);
        const options = this.psqlQueryOptions(query, configs.dbEnv, psqlCmdArgs);
        return this.runWithTunnel(configs.dbTunnelConfig, options);
    }
    /**
     * Consumes a stream and returns its content as a string.
     *
     * @param inputStream - The input stream to consume
     * @returns Promise that resolves to the stream content as a string
     */
    consumeStream(inputStream) {
        let result = '';
        const throughStream = new node_stream_1.Stream.PassThrough();
        // eslint-disable-next-line no-async-promise-executor
        const promise = new Promise(async (resolve, reject) => {
            try {
                await (0, promises_1.finished)(throughStream);
                resolve(result);
            }
            catch (error) {
                reject(error);
            }
        });
        // eslint-disable-next-line no-return-assign
        throughStream.on('data', chunk => result += chunk.toString());
        inputStream.pipe(throughStream);
        return promise;
    }
    /**
     * Kills a child process if it hasn't been killed already.
     * According to node.js docs, sending a kill to a process won't cause an error
     * but could have unintended consequences if the PID gets reassigned.
     * To be on the safe side, check if the process was already killed before sending the signal.
     *
     * @param childProcess - The child process to kill
     * @param signal - The signal to send to the process
     * @returns void
     */
    kill(childProcess, signal) {
        if (!childProcess.killed) {
            pgDebug('killing psql child process');
            childProcess.kill(signal);
        }
    }
    /**
     * Creates the options for spawning the psql process.
     *
     * @param query - The SQL query to execute
     * @param dbEnv - The database environment variables
     * @param psqlCmdArgs - Additional command-line arguments for psql (default: [])
     * @returns Object containing child process options, database environment, and psql arguments
     */
    psqlQueryOptions(query, dbEnv, psqlCmdArgs = []) {
        pgDebug('Running query: %s', query.trim());
        const psqlArgs = ['-c', query, '--set', 'sslmode=require', ...psqlCmdArgs];
        const childProcessOptions = {
            stdio: ['ignore', 'pipe', 'inherit'],
        };
        return {
            childProcessOptions,
            dbEnv,
            psqlArgs,
        };
    }
    /**
     * Runs the psql command with tunnel support.
     *
     * @param tunnelConfig - The tunnel configuration object
     * @param options - The options for spawning the psql process
     * @returns Promise that resolves to the query result as a string
     */
    // eslint-disable-next-line perfectionist/sort-classes
    async runWithTunnel(tunnelConfig, options) {
        const tunnel = await Tunnel.connect(this.connectionDetails, tunnelConfig, this.tunnelFn);
        pgDebug('after create tunnel');
        const psql = this.spawnPsql(options);
        // Note: In non-interactive mode, psql.stdout is available for capturing output.
        // In interactive mode, stdio: 'inherit' would make psql.stdout null.
        // Return a string for consistency but ideally we should return the child process from this function
        // and let the caller decide what to do with stdin/stdout/stderr
        const stdoutPromise = psql.stdout ? this.consumeStream(psql.stdout) : Promise.resolve('');
        const cleanupSignalTraps = this.trapAndForwardSignalsToChildProcess(psql);
        try {
            pgDebug('waiting for psql or tunnel to exit');
            // wait for either psql or tunnel to exit;
            // the important bit is that we ensure both processes are
            // always cleaned up in the `finally` block below
            await Promise.race([
                this.waitForPSQLExit(psql),
                tunnel.waitForClose(),
            ]);
        }
        catch (error) {
            pgDebug('wait for psql or tunnel error', error);
            throw error;
        }
        finally {
            pgDebug('begin tunnel cleanup');
            cleanupSignalTraps();
            tunnel.close();
            this.kill(psql, 'SIGKILL');
            pgDebug('end tunnel cleanup');
        }
        return stdoutPromise;
    }
    /**
     * Spawns the psql process with the given options.
     *
     * @param options - The options for spawning the psql process
     * @returns The spawned child process
     */
    spawnPsql(options) {
        const { childProcessOptions, dbEnv, psqlArgs } = options;
        const spawnOptions = Object.assign({ env: dbEnv }, childProcessOptions);
        pgDebug('opening psql process');
        const psql = this.spawnFn('psql', psqlArgs, spawnOptions);
        psql.once('spawn', () => pgDebug('psql process spawned'));
        return psql;
    }
    /**
     * Traps SIGINT so that ctrl+c can be used by psql without killing the parent node process.
     * You can use ctrl+c in psql to kill running queries while keeping the psql process open.
     * This code is to stop the parent node process (heroku CLI) from exiting.
     * If the parent Heroku CLI node process exits, then psql will exit as it is a child process.
     *
     * @param childProcess - The child process to forward signals to
     * @returns Function to restore the original signal handlers
     */
    trapAndForwardSignalsToChildProcess(childProcess) {
        const signalsToTrap = ['SIGINT'];
        const signalTraps = signalsToTrap.map(signal => {
            process.removeAllListeners(signal);
            const listener = () => this.kill(childProcess, signal);
            process.on(signal, listener);
            return [signal, listener];
        });
        // restores the built-in node ctrl+c and other handlers
        return () => {
            for (const [signal, listener] of signalTraps) {
                process.removeListener(signal, listener);
            }
        };
    }
    /**
     * Waits for the psql process to exit and handles any errors.
     *
     * @param psql - The psql process event emitter
     * @throws Error if psql exits with non-zero code or if psql command is not found
     * @returns Promise that resolves to void when psql exits
     */
    async waitForPSQLExit(psql) {
        let errorToThrow = null;
        try {
            const [exitCode] = await (0, node_events_1.once)(psql, 'close');
            pgDebug(`psql exited with code ${exitCode}`);
            if (exitCode > 0) {
                errorToThrow = new Error(`psql exited with code ${exitCode}`);
            }
        }
        catch (error) {
            pgDebug('psql process error', error);
            const { code } = error;
            if (code === 'ENOENT') {
                errorToThrow = new Error('The local psql command could not be located. For help installing psql, see '
                    + 'https://devcenter.heroku.com/articles/heroku-postgresql#local-setup');
            }
        }
        if (errorToThrow) {
            throw errorToThrow;
        }
    }
}
exports.default = PsqlService;
