"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.configVarsByAppIdCache = void 0;
exports.getConfig = getConfig;
exports.getConfigVarName = getConfigVarName;
exports.getConfigVarNameFromAttachment = getConfigVarNameFromAttachment;
const color_1 = require("@heroku-cli/color");
/**
 * Cache of app config vars.
 */
exports.configVarsByAppIdCache = new Map();
/**
 * Returns the app's config vars as a record of key-value pairs, either from the cache or from the API.
 *
 * @param heroku - The Heroku API client
 * @param appId - The ID of the app to get config vars for
 * @returns Promise resolving to a record of config var key-value pairs
 */
async function getConfig(heroku, appId) {
    let promise = exports.configVarsByAppIdCache.get(appId);
    if (!promise) {
        promise = heroku.get(`/apps/${appId}/config-vars`);
        exports.configVarsByAppIdCache.set(appId, promise);
    }
    const { body: config } = await promise;
    return config;
}
/**
 * Returns the attachment's first config var name that has a `_URL` suffix, expected to be the name of the one
 * that contains the database URL connection string.
 *
 * @param configVarNames - Array of config var names from the attachment
 * @returns The first config var name ending with '_URL'
 * @throws {Error} When no config var names end with '_URL'
 */
function getConfigVarName(configVarNames) {
    const urlConfigVarNames = configVarNames.filter(cv => (cv.endsWith('_URL')));
    if (urlConfigVarNames.length === 0)
        throw new Error('Database URL not found for this addon');
    return urlConfigVarNames[0];
}
/**
 * Returns the config var name that contains the database URL connection string for the given
 * attachment, based on the contents of the app's config vars.
 *
 * @param attachment - The addon attachment to get the config var name for
 * @param config - Optional record of app config vars (defaults to empty object)
 * @returns The config var name containing the database URL
 * @throws {Error} When no config vars are found or when they don't contain a database URL
 */
function getConfigVarNameFromAttachment(attachment, config = {}) {
    // Handle the case where no attachment config var names remain after filtering out those that don't contain a
    // database URL connection string in the app's config vars.
    const connStringConfigVarNames = attachment.config_vars
        .filter(cvn => { var _a; return (_a = config[cvn]) === null || _a === void 0 ? void 0 : _a.startsWith('postgres://'); });
    if (connStringConfigVarNames.length === 0) {
        throw new Error(`No config vars found for ${attachment.name}; perhaps they were removed as a side effect of`
            + ` ${color_1.color.cmd('heroku rollback')}? Use ${color_1.color.cmd('heroku addons:attach')} to create a new attachment and `
            + `then ${color_1.color.cmd('heroku addons:detach')} to remove the current attachment.`);
    }
    // Generate the default config var name and return it if it contains a database URL connection string.
    const configVarName = `${attachment.name}_URL`;
    if (connStringConfigVarNames.includes(configVarName) && configVarName in config) {
        return configVarName;
    }
    // Return the first config var name that has a `_URL` suffix. This might not be needed at all anymore.
    return getConfigVarName(connStringConfigVarNames);
}
