import path from 'node:path';
import fs from 'node:fs/promises';
import { Validator } from 'jsonschema';
/**
 * Retrieves the app.json. This file must be in the
 * root of the workspace and must be valid.
 *
 * @param workspaceRootUri The workspace uri to retrieve the file from
 * @param appJsonFile The app.json file to read
 * @returns The typed app.json as an object
 * @throws {Error} If the app.json cannot be read or is invalid
 */
export async function readAppJson(workspaceRootUri, appJsonFile) {
    const appJsonUri = path.join(workspaceRootUri, 'app.json');
    let appJsonBytes = appJsonFile;
    if (!appJsonFile) {
        try {
            await fs.stat(appJsonUri);
        }
        catch {
            throw new Error(`Cannot find app.json file at ${appJsonUri}`);
        }
        appJsonBytes = await fs.readFile(appJsonUri);
    }
    let appJson;
    try {
        appJson = JSON.parse(Buffer.from(appJsonBytes).toString());
    }
    catch (e) {
        throw new Error(`Cannot parse the app.json file: ${e.message}`);
    }
    const validator = new Validator();
    const schema = await import('./app-json.schema.json', { with: { type: 'json' } });
    const result = validator.validate(appJson, schema.default);
    if (!result.valid) {
        throw new Error(`Invalid app.json file: ${result.errors.map((e) => e.message).join(', ')}`);
    }
    return appJson;
}
