/**
 * [Heroku Build API - Build](https://devcenter.heroku.com/articles/platform-api-reference#build)
 * A build represents the process of transforming a code tarball into build artifacts
 */
export default class BuildService {
    endpoint;
    /**
     *
     * @param endpoint The endpoint to use for the build service.
     */
    constructor(endpoint) {
        this.endpoint = endpoint;
    }
    /**
     * Create a new build.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param payload Object to send to the endpoint.
     * @param requestInit The initializer for the request.
     * @returns The build.
     */
    async create(appIdentity, payload, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/builds`, {
            ...requestInit,
            body: JSON.stringify(payload, null, 2),
            method: 'POST',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Info for existing build.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param buildIdentity unique identifier of build.
     * @param requestInit The initializer for the request.
     * @returns The build.
     */
    async info(appIdentity, buildIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/builds/${buildIdentity}`, {
            ...requestInit,
            method: 'GET',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * List existing build.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param requestInit The initializer for the request.
     * @returns The builds.
     */
    async list(appIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/builds`, {
            ...requestInit,
            method: 'GET',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
    /**
     * Destroy a build cache.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param requestInit The initializer for the request.
     */
    async deleteCache(appIdentity, requestInit = {}) {
        await fetch(`/apps/${appIdentity}/build-cache`, {
            ...requestInit,
            method: 'DELETE',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
    }
    /**
     * Cancel running build.
     *
     * @param appIdentity unique identifier of app or unique name of app.
     * @param buildIdentity unique identifier of build.
     * @param requestInit The initializer for the request.
     * @returns The build.
     */
    async cancel(appIdentity, buildIdentity, requestInit = {}) {
        const response = await fetch(`${this.endpoint}/apps/${appIdentity}/builds/${buildIdentity}`, {
            ...requestInit,
            method: 'DELETE',
            headers: {
                ...requestInit?.headers,
                Accept: 'application/vnd.heroku+json; version=3.sdk',
                'Content-Type': 'application/json'
            }
        });
        if (response.ok) {
            return (await response.json());
        }
        let message = response.statusText;
        try {
            ({ message } = (await response.json()));
        }
        catch {
            // no-op
        }
        throw new Error(`${response.status}: ${message}`, { cause: response });
    }
}
