// Copyright 2016-2022 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.goide.documentation;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.psi.PsiElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * This Extension Point offers a facility to provide extra texts documentation in the Quick Documentation popups.
 * It can be used to provide information to the end user based on the element hovered and the surrounding PSI tree.
 * Whenever the users hovers code or invokes Quick Documentation, the Go plugin computes a suitable text from
 * available GoDoc and built in data sources (e.g. for keywords).
 * Then, all AdditionalDocumentationProviders are queried and resulting texts are appended to the internally generated texts.
 */
public interface GoAdditionalDocumentationProvider {
  ExtensionPointName<GoAdditionalDocumentationProvider> EP_NAME =
    ExtensionPointName.create("com.goide.documentation.additionalDocumentationProvider");

  /**
   * Generates the auxiliary documentation to be appended to the Quick Documentation popups.
   *
   * @param element         is the PsiElement being documented in the Quick Documentation popup. This is not necessarily the hovered element,
   *                        i.e. might be the element referenced by the element hovered
   * @param originalElement is the PsiElement the Quick Documentation popup was requested for, i.e. the element hovered or under the caret
   *                        if Quick Documentation is invoked from the menu / via hotkey
   * @return additional documentation in HTML format. Use empty string if you have nothing to append.
   */
  @NotNull String provideAdditionalDocumentation(@NotNull PsiElement element, @Nullable PsiElement originalElement);
}
