// Copyright 2016-2022 JetBrains s.r.o.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package com.goide.codeInsight.imports;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.UserDataHolder;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Provides weights for files (and therefore their packages) for list of imports in import package quickfix.
 * These weights are applied after checking for recently optimized imports in a file and imports usage
 * statistics.
 */
public interface GoImportsWeigher {

  ExtensionPointName<GoImportsWeigher> EP_NAME = ExtensionPointName.create("com.goide.imports.weigher");

  /**
   * Should return true if weigher is applicable to the given project.
   */
  boolean isApplicable(@NotNull Project project);

  /**
   * Should return weight for the given file in the provided context or null if the weight
   * cannot be computed. Implementation can cache data in the provided dataHolder. The same
   * dataHolder is used for all imports in import list.
   */
  @Nullable
  Integer weight(@NotNull Project project,
                 @Nullable PsiElement context,
                 @Nullable PsiFile fileToWeight,
                 @NotNull UserDataHolder dataHolder);

  static int getWeight(@NotNull Project project,
                       @Nullable PsiElement context,
                       @Nullable PsiFile fileToWeight,
                       @NotNull UserDataHolder dataHolder) {
    for (GoImportsWeigher weigher : EP_NAME.getExtensionList()) {
      if (weigher.isApplicable(project)) {
        Integer weight = weigher.weight(project, context, fileToWeight, dataHolder);
        if (weight != null) {
          return weight;
        }
      }
    }
    return 0;
  }
}
