# SPDX-License-Identifier: LGPL-2.1-or-later

# ***************************************************************************
# *   Copyright (c) 2009, 2010 Yorik van Havre <yorik@uncreated.net>        *
# *   Copyright (c) 2009, 2010 Ken Cline <cline@frii.com>                   *
# *   Copyright (c) 2020 FreeCAD Developers                                 *
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU Lesser General Public License (LGPL)    *
# *   as published by the Free Software Foundation; either version 2 of     *
# *   the License, or (at your option) any later version.                   *
# *   for detail see the LICENCE text file.                                 *
# *                                                                         *
# *   This program is distributed in the hope that it will be useful,       *
# *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
# *   GNU Library General Public License for more details.                  *
# *                                                                         *
# *   You should have received a copy of the GNU Library General Public     *
# *   License along with this program; if not, write to the Free Software   *
# *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  *
# *   USA                                                                   *
# *                                                                         *
# ***************************************************************************
"""Provides functions to repair certain objects created with old versions."""
## @package heal
# \ingroup draftfunctions
# \brief Provides functions to repair certain objects from old versions.

## \addtogroup draftfunctions
# @{
import FreeCAD as App
import draftutils.utils as utils

from draftmake.make_copy import make_copy


def heal(objlist=None, delete=True, reparent=True):
    """heal([objlist],[delete],[reparent])

    Recreate Draft objects that are damaged, for example if created from an
    earlier version. If ran without arguments, all the objects in the document
    will be healed if they are damaged.

    Parameters
    ----------
    objlist : list

    delete : Base.Vector or list of Base.Vector
        If delete is True, the damaged objects are deleted (default).

    reparent : bool
        If reparent is True (default), new objects go at the very same place
        in the tree than their original.
    """

    auto = False

    if not objlist:
        objlist = App.ActiveDocument.Objects
        print("Automatic mode: Healing whole document…")
        auto = True
    else:
        print("Manual mode: Force-healing selected objects…")

    if not isinstance(objlist, list):
        objlist = [objlist]

    dellist = []
    got = False

    for obj in objlist:
        dtype = utils.get_type(obj)
        ftype = obj.TypeId
        if ftype in ["Part::FeaturePython", "App::FeaturePython", "Part::Part2DObjectPython"]:
            proxy = obj.Proxy
            if hasattr(obj, "ViewObject"):
                if hasattr(obj.ViewObject, "Proxy"):
                    proxy = obj.ViewObject.Proxy
            if (proxy == 1) or (dtype in ["Unknown", "Part"]) or (not auto):
                got = True
                dellist.append(obj.Name)
                props = obj.PropertiesList
                if ("Dimline" in props) and ("Start" in props):
                    print("Healing " + obj.Name + " of type Dimension")
                    nobj = make_copy(obj, force="Dimension", reparent=reparent)
                elif ("Height" in props) and ("Length" in props):
                    print("Healing " + obj.Name + " of type Rectangle")
                    nobj = make_copy(obj, force="Rectangle", reparent=reparent)
                elif ("Points" in props) and ("Closed" in props):
                    if "BSpline" in obj.Name:
                        print("Healing " + obj.Name + " of type BSpline")
                        nobj = make_copy(obj, force="BSpline", reparent=reparent)
                    else:
                        print("Healing " + obj.Name + " of type Wire")
                        nobj = make_copy(obj, force="Wire", reparent=reparent)
                elif ("Radius" in props) and ("FirstAngle" in props):
                    print("Healing " + obj.Name + " of type Circle")
                    nobj = make_copy(obj, force="Circle", reparent=reparent)
                elif ("DrawMode" in props) and ("FacesNumber" in props):
                    print("Healing " + obj.Name + " of type Polygon")
                    nobj = make_copy(obj, force="Polygon", reparent=reparent)
                else:
                    dellist.pop()
                    print("Object " + obj.Name + " is not healable")

    if not got:
        print("No object seems to need healing")
    else:
        print("Healed ", len(dellist), " objects")

    if dellist and delete:
        for n in dellist:
            App.ActiveDocument.removeObject(n)


## @}
