# SPDX-License-Identifier: LGPL-2.1-or-later

# ***************************************************************************
# *                                                                         *
# *   Copyright (c) 2023 Yorik van Havre <yorik@uncreated.net>              *
# *                                                                         *
# *   This file is part of FreeCAD.                                         *
# *                                                                         *
# *   FreeCAD is free software: you can redistribute it and/or modify it    *
# *   under the terms of the GNU Lesser General Public License as           *
# *   published by the Free Software Foundation, either version 2.1 of the  *
# *   License, or (at your option) any later version.                       *
# *                                                                         *
# *   FreeCAD is distributed in the hope that it will be useful, but        *
# *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
# *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
# *   Lesser General Public License for more details.                       *
# *                                                                         *
# *   You should have received a copy of the GNU Lesser General Public      *
# *   License along with FreeCAD. If not, see                               *
# *   <https://www.gnu.org/licenses/>.                                      *
# *                                                                         *
# ***************************************************************************

"""Unit test for the Native IFC module"""

import difflib
import os
import tempfile
import unittest

import ifcopenshell
from ifcopenshell.util import element

import FreeCAD
import Arch
import Draft

from . import ifc_import
from . import ifc_tools
from . import ifc_geometry
from . import ifc_materials
from . import ifc_layers
from . import ifc_psets
from . import ifc_objects
from . import ifc_generator


IFC_FILE_PATH = None  # downloaded IFC file path
FCSTD_FILE_PATH = None  # saved FreeCAD file
PARAMS = FreeCAD.ParamGet("User parameter:BaseApp/Preferences/Mod/NativeIFC")
SINGLEDOC = False  # This allows one to force singledoc mode for all tests
SDU = int(SINGLEDOC)  # number of objects is different in singledoc
# See also IFCFILECONTENT at the end of this file, retrieved from
# https://github.com/aothms/IfcOpenHouse/IfcOpenHouse_IFC4.ifc

"""
unit tests for the NativeIFC functionality. To run the tests, either:
- in terminal mode: FreeCAD -t ifc_selftest
- in the FreeCAD UI: Switch to Test Framework workbench, press "Self test" and
  choose nativeifc.ifc_selftest in the list
"""


def getIfcFilePath():
    global IFC_FILE_PATH
    if not IFC_FILE_PATH:
        path = tempfile.mkstemp(suffix=".ifc")[1]
        with open(path, "w") as f:
            f.write(IFCFILECONTENT)
        IFC_FILE_PATH = path
    return IFC_FILE_PATH


def clearObjects():
    names = [o.Name for o in FreeCAD.getDocument("IfcTest").Objects]
    for n in names:
        FreeCAD.getDocument("IfcTest").removeObject(n)


def compare(file1, file2):
    with open(file1) as f1:
        f1_text = f1.readlines()
    with open(file2) as f2:
        f2_text = f2.readlines()
    res = [
        l for l in difflib.unified_diff(f1_text, f2_text, fromfile=file1, tofile=file2, lineterm="")
    ]
    res = [l for l in res if l.startswith("+") or l.startswith("-")]
    res = [l for l in res if not l.startswith("+++") and not l.startswith("---")]
    return res


class NativeIFCTest(unittest.TestCase):

    def setUp(self):
        # setting a new document to hold the tests
        if FreeCAD.ActiveDocument:
            if FreeCAD.ActiveDocument.Name != "IfcTest":
                FreeCAD.newDocument("IfcTest")
        else:
            FreeCAD.newDocument("IfcTest")
        FreeCAD.setActiveDocument("IfcTest")

    def tearDown(self):
        FreeCAD.closeDocument("IfcTest")
        pass

    def test01_ImportCoinSingle(self):
        FreeCAD.Console.PrintMessage("NativeIFC 01: Importing single object, coin mode...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=0,
            shapemode=1,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        self.assertTrue(fco == 1 - SDU, "ImportCoinSingle failed")

    def test02_ImportCoinStructure(self):
        FreeCAD.Console.PrintMessage("NativeIFC 02: Importing model structure, coin mode...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=1,
            shapemode=1,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        self.assertTrue(fco == 4 - SDU, "ImportCoinStructure failed")

    def test03_ImportCoinFull(self):
        global FCSTD_FILE_PATH
        FreeCAD.Console.PrintMessage("NativeIFC 03: Importing full model, coin mode...")
        clearObjects()
        fp = getIfcFilePath()
        d = ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=1,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        path = tempfile.mkstemp(suffix=".FCStd")[1]
        d.saveAs(path)
        FCSTD_FILE_PATH = path
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        self.assertTrue(fco > 4 - SDU, "ImportCoinFull failed")

    def test04_ImportShapeFull(self):
        FreeCAD.Console.PrintMessage("NativeIFC 04: Importing full model, shape mode...")
        clearObjects()
        fp = getIfcFilePath()
        d = ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        self.assertTrue(fco > 4 - SDU, "ImportShapeFull failed")

    def test05_ImportFreeCAD(self):
        FreeCAD.Console.PrintMessage("NativeIFC 05: FreeCAD import of NativeIFC coin file...")
        clearObjects()
        doc = FreeCAD.open(FCSTD_FILE_PATH)
        obj = doc.Objects[-1]
        proj = ifc_tools.get_project(obj)
        ifcfile = ifc_tools.get_ifcfile(proj)
        print(ifcfile)
        self.assertTrue(ifcfile, "ImportFreeCAD failed")

    def test06_ModifyObjects(self):
        FreeCAD.Console.PrintMessage("NativeIFC 06: Modifying IFC document...")
        doc = FreeCAD.open(FCSTD_FILE_PATH)
        obj = doc.Objects[-1]
        obj.Label = "Modified name"
        proj = ifc_tools.get_project(obj)
        proj.IfcFilePath = proj.IfcFilePath[:-4] + "_modified.ifc"
        ifc_tools.save_ifc(proj)
        ifc_diff = compare(IFC_FILE_PATH, proj.IfcFilePath)
        obj.ShapeMode = 0
        obj.Proxy.execute(obj)
        self.assertTrue(obj.Shape.Volume > 2 and len(ifc_diff) <= 5, "ModifyObjects failed")

    def test07_CreateDocument(self):
        FreeCAD.Console.PrintMessage("NativeIFC 07: Creating new IFC document...")
        doc = FreeCAD.ActiveDocument
        ifc_tools.create_document(doc, silent=True)
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        print(FreeCAD.getDocument("IfcTest").Objects)
        self.assertTrue(fco == 1 - SDU, "CreateDocument failed")

    def test08_ChangeIFCSchema(self):
        FreeCAD.Console.PrintMessage("NativeIFC 08: Changing IFC schema...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=1,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        obj = FreeCAD.getDocument("IfcTest").Objects[-1]
        proj = ifc_tools.get_project(obj)
        oldid = obj.StepId
        proj.Proxy.silent = True
        proj.Schema = "IFC2X3"
        FreeCAD.getDocument("IfcTest").recompute()
        self.assertTrue(obj.StepId != oldid, "ChangeIFCSchema failed")

    def test09_CreateBIMObjects(self):
        FreeCAD.Console.PrintMessage("NativeIFC 09: Creating BIM objects...")
        doc = FreeCAD.ActiveDocument
        proj = ifc_tools.create_document(doc, silent=True)
        site = Arch.makeSite()
        site = ifc_tools.aggregate(site, proj)
        bldg = Arch.makeBuilding()
        bldg = ifc_tools.aggregate(bldg, site)
        storey = Arch.makeFloor()
        storey = ifc_tools.aggregate(storey, bldg)
        wall = Arch.makeWall(None, 200, 400, 20)
        wall = ifc_tools.aggregate(wall, storey)
        column = Arch.makeStructure(None, 20, 20, 200)
        column.IfcType = "Column"
        column = ifc_tools.aggregate(column, storey)
        beam = Arch.makeStructure(None, 20, 200, 20)
        beam.IfcType = "Beam"
        beam = ifc_tools.aggregate(beam, storey)
        rect = Draft.makeRectangle(200, 200)
        slab = Arch.makeStructure(rect, height=20)
        slab.IfcType = "Slab"
        slab = ifc_tools.aggregate(slab, storey)
        # TODO create door, window
        fco = len(FreeCAD.getDocument("IfcTest").Objects)
        ifco = len(proj.Proxy.ifcfile.by_type("IfcRoot"))
        print(ifco, "IFC objects created")
        self.assertTrue(fco == 8 - SDU and ifco == 12, "CreateDocument failed")

    def test10_ChangePlacement(self):
        FreeCAD.Console.PrintMessage("NativeIFC 10: Changing Placement...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=1,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        obj = FreeCAD.getDocument("IfcTest").getObject("IfcObject00" + str(4 - SDU))
        elem = ifc_tools.get_ifc_element(obj)
        obj.Placement.move(FreeCAD.Vector(100, 200, 300))
        new_plac = ifcopenshell.util.placement.get_local_placement(elem.ObjectPlacement)
        new_plac = str(new_plac).replace(" ", "").replace("\n", "")
        target = "[[1.0.0.100.][0.1.0.200.][0.0.1.300.][0.0.0.1.]]"
        self.assertTrue(new_plac == target, "ChangePlacement failed")

    def test11_ChangeGeometry(self):
        FreeCAD.Console.PrintMessage("NativeIFC 11: Changing Geometry...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        obj = FreeCAD.getDocument("IfcTest").getObject("IfcObject004")
        ifc_geometry.add_geom_properties(obj)
        obj.ExtrusionDepth = "6000 mm"
        FreeCAD.getDocument("IfcTest").recompute()
        self.assertTrue(obj.Shape.Volume > 1500000, "ChangeGeometry failed")

    def test12_RemoveObject(self):
        from . import ifc_observer

        ifc_observer.add_observer()
        FreeCAD.Console.PrintMessage("NativeIFC 12: Remove object...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        ifcfile = ifc_tools.get_ifcfile(FreeCAD.getDocument("IfcTest").Objects[-1])
        count1 = len(ifcfile.by_type("IfcProduct"))
        FreeCAD.getDocument("IfcTest").removeObject("IfcObject004")
        count2 = len(ifcfile.by_type("IfcProduct"))
        self.assertTrue(count2 < count1, "RemoveObject failed")

    def test13_Materials(self):
        FreeCAD.Console.PrintMessage("NativeIFC 13: Materials...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        proj = FreeCAD.getDocument("IfcTest").Objects[0]
        ifc_materials.load_materials(proj)
        prod = FreeCAD.getDocument("IfcTest").getObject("IfcObject006")
        ifcfile = ifc_tools.get_ifcfile(prod)
        mats_before = ifcfile.by_type("IfcMaterialDefinition")
        mat = Arch.makeMaterial("Red")
        ifc_materials.set_material(mat, prod)
        elem = ifc_tools.get_ifc_element(prod)
        res = ifcopenshell.util.element.get_material(elem)
        mats_after = ifcfile.by_type("IfcMaterialDefinition")
        self.assertTrue(len(mats_after) == len(mats_before) + 1, "Materials failed")

    def test14_Layers(self):
        FreeCAD.Console.PrintMessage("NativeIFC 14: Layers...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        proj = FreeCAD.getDocument("IfcTest").Objects[0]
        ifcfile = ifc_tools.get_ifcfile(proj)
        lays_before = ifcfile.by_type("IfcPresentationLayerAssignment")
        layer = ifc_layers.create_layer("My Layer", proj)
        prod = FreeCAD.getDocument("IfcTest").getObject("IfcObject006")
        ifc_layers.add_to_layer(prod, layer)
        lays_after = ifcfile.by_type("IfcPresentationLayerAssignment")
        self.assertTrue(len(lays_after) == len(lays_before) + 1, "Layers failed")

    def test15_Psets(self):
        FreeCAD.Console.PrintMessage("NativeIFC 15: Psets...")
        clearObjects()
        fp = getIfcFilePath()
        ifc_import.insert(
            fp,
            "IfcTest",
            strategy=2,
            shapemode=0,
            switchwb=0,
            silent=True,
            singledoc=SINGLEDOC,
        )
        obj = FreeCAD.getDocument("IfcTest").getObject("IfcObject004")
        ifcfile = ifc_tools.get_ifcfile(obj)
        pset = ifc_psets.add_pset(obj, "Pset_Custom")
        ifc_psets.add_property(ifcfile, pset, "MyMessageToTheWorld", "Hello, World!")
        self.assertTrue(ifc_psets.has_psets(obj), "Psets failed")


IFCFILECONTENT = """ISO-10303-21;
HEADER;
FILE_DESCRIPTION(('ViewDefinition [CoordinationView]'),'2;1');
FILE_NAME('IfcOpenHouse.ifc','2014-05-05T15:42:14',(''),('',''),'IfcOpenShell 0.5.0-dev','IfcOpenShell 0.5.0-dev','');
FILE_SCHEMA(('IFC4'));
ENDSEC;
DATA;
#1=IFCPERSON($,$,'',$,$,$,$,$);
#2=IFCORGANIZATION($,'IfcOpenShell',$,$,$);
#3=IFCPERSONANDORGANIZATION(#1,#2,$);
#4=IFCAPPLICATION(#2,'0.5.0-dev','IfcOpenShell','IfcOpenShell');
#5=IFCOWNERHISTORY(#3,#4,$,.ADDED.,1399297334,#3,#4,1399297334);
#6=IFCDIRECTION((0.,1.));
#7=IFCDIRECTION((1.,0.,0.));
#8=IFCDIRECTION((0.,0.,1.));
#9=IFCCARTESIANPOINT((0.,0.,0.));
#10=IFCAXIS2PLACEMENT3D(#9,#8,#7);
#11=IFCGEOMETRICREPRESENTATIONCONTEXT('Plan','Model',3,1.E-005,#10,#6);
#12=IFCDIMENSIONALEXPONENTS(0,0,0,0,0,0,0);
#13=IFCSIUNIT(*,.LENGTHUNIT.,.MILLI.,.METRE.);
#14=IFCSIUNIT(*,.PLANEANGLEUNIT.,$,.RADIAN.);
#15=IFCMEASUREWITHUNIT(IFCPLANEANGLEMEASURE(0.0174533),#14);
#16=IFCCONVERSIONBASEDUNIT(#12,.PLANEANGLEUNIT.,'Degrees',#15);
#17=IFCUNITASSIGNMENT((#13,#16));
#18=IFCPROJECT('2Iicv0RnfAVPda6Sg4SE78',#5,'IfcOpenHouse',$,$,$,$,(#11),#17);
#19=IFCDIRECTION((1.,0.,0.));
#20=IFCDIRECTION((0.,0.,1.));
#21=IFCCARTESIANPOINT((0.,0.,0.));
#22=IFCAXIS2PLACEMENT3D(#21,#20,#19);
#23=IFCLOCALPLACEMENT($,#22);
#24=IFCSITE('3bqf5ZY_nDpfb1hfmOkxFk',#5,$,$,$,#23,#2325,$,.ELEMENT.,$,$,$,$,$);
#25=IFCRELAGGREGATES('3d0apJSKnCWhbktmj33dJ8',#5,$,$,#18,(#24));
#26=IFCDIRECTION((1.,0.,0.));
#27=IFCDIRECTION((0.,0.,1.));
#28=IFCCARTESIANPOINT((0.,0.,0.));
#29=IFCAXIS2PLACEMENT3D(#28,#27,#26);
#30=IFCLOCALPLACEMENT(#23,#29);
#31=IFCBUILDING('3FweM$L1L56fABBUNXlIbJ',#5,$,$,$,#30,$,$,.ELEMENT.,$,$,$);
#32=IFCRELAGGREGATES('03$UXM4xjFZwZ__XhAtJCL',#5,$,$,#24,(#31));
#33=IFCDIRECTION((1.,0.,0.));
#34=IFCDIRECTION((0.,0.,1.));
#35=IFCCARTESIANPOINT((0.,0.,0.));
#36=IFCAXIS2PLACEMENT3D(#35,#34,#33);
#37=IFCLOCALPLACEMENT(#30,#36);
#38=IFCBUILDINGSTOREY('38aOKO8_DDkBd1FHm_lVXz',#5,$,$,$,#37,$,$,.ELEMENT.,$);
#39=IFCRELAGGREGATES('30qDZxPF51V9OcLHYaYYkO',#5,$,$,#31,(#38));
#40=IFCWALLSTANDARDCASE('3g46_woBL6sugXeY5_WP6n',#5,'South wall',$,$,#65,#42,$,.STANDARD.);
#41=IFCRELCONTAINEDINSPATIALSTRUCTURE('3xqBbvJO14iQBO0t1Vhr9P',#5,$,$,(#40),#38);
#42=IFCPRODUCTDEFINITIONSHAPE($,$,(#43,#44));
#43=IFCSHAPEREPRESENTATION(#11,'Axis','Curve2D',(#60));
#44=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#57));
#45=IFCCARTESIANPOINT((-5000.,-180.));
#46=IFCCARTESIANPOINT((5000.,-180.));
#47=IFCCARTESIANPOINT((5000.,180.));
#48=IFCCARTESIANPOINT((-5000.,180.));
#49=IFCCARTESIANPOINT((-5000.,-180.));
#50=IFCDIRECTION((0.,0.,1.));
#51=IFCDIRECTION((1.,0.,0.));
#52=IFCDIRECTION((0.,0.,1.));
#53=IFCCARTESIANPOINT((0.,0.,0.));
#54=IFCAXIS2PLACEMENT3D(#53,#52,#51);
#55=IFCPOLYLINE((#45,#46,#47,#48,#49,#45));
#56=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#55);
#57=IFCEXTRUDEDAREASOLID(#56,#54,#50,3000.);
#58=IFCCARTESIANPOINT((-5000.,0.));
#59=IFCCARTESIANPOINT((5000.,0.));
#60=IFCPOLYLINE((#58,#59));
#61=IFCDIRECTION((1.,0.,0.));
#62=IFCDIRECTION((0.,0.,1.));
#63=IFCCARTESIANPOINT((0.,0.,0.));
#64=IFCAXIS2PLACEMENT3D(#63,#62,#61);
#65=IFCLOCALPLACEMENT(#37,#64);
#66=IFCCOLOURRGB($,0.75,0.73,0.68);
#67=IFCSURFACESTYLERENDERING(#66,$,$,$,$,$,$,$,.FLAT.);
#68=IFCSURFACESTYLE($,.BOTH.,(#67));
#69=IFCPRESENTATIONSTYLEASSIGNMENT((#68));
#70=IFCSTYLEDITEM(#60,(#69),$);
#71=IFCSTYLEDITEM(#57,(#69),$);
#72=IFCFOOTING('2W4bl$KK1Brv_UEOkAThqU',#5,'Footing',$,$,#93,#75,$,.STRIP_FOOTING.);
#73=IFCRELCONTAINEDINSPATIALSTRUCTURE('0h_u6IeJHFqO2TNY7iSwx5',#5,$,$,(#72),#38);
#74=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#88));
#75=IFCPRODUCTDEFINITIONSHAPE($,$,(#74));
#76=IFCCARTESIANPOINT((-5050.,-2730.));
#77=IFCCARTESIANPOINT((5050.,-2730.));
#78=IFCCARTESIANPOINT((5050.,2730.));
#79=IFCCARTESIANPOINT((-5050.,2730.));
#80=IFCCARTESIANPOINT((-5050.,-2730.));
#81=IFCDIRECTION((0.,0.,1.));
#82=IFCDIRECTION((1.,0.,0.));
#83=IFCDIRECTION((0.,0.,1.));
#84=IFCCARTESIANPOINT((0.,0.,0.));
#85=IFCAXIS2PLACEMENT3D(#84,#83,#82);
#86=IFCPOLYLINE((#76,#77,#78,#79,#80,#76));
#87=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#86);
#88=IFCEXTRUDEDAREASOLID(#87,#85,#81,2000.);
#89=IFCDIRECTION((1.,0.,0.));
#90=IFCDIRECTION((0.,0.,1.));
#91=IFCCARTESIANPOINT((0.,2500.,-2000.));
#92=IFCAXIS2PLACEMENT3D(#91,#90,#89);
#93=IFCLOCALPLACEMENT(#37,#92);
#94=IFCCOLOURRGB($,0.26,0.22,0.18);
#95=IFCSURFACESTYLERENDERING(#94,$,$,$,$,$,$,$,.FLAT.);
#96=IFCSURFACESTYLE($,.BOTH.,(#95));
#97=IFCPRESENTATIONSTYLEASSIGNMENT((#96));
#98=IFCSTYLEDITEM(#88,(#97),$);
#99=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#113));
#100=IFCPRODUCTDEFINITIONSHAPE($,$,(#99));
#101=IFCCARTESIANPOINT((-3000.,-1815.));
#102=IFCCARTESIANPOINT((3000.,-1815.));
#103=IFCCARTESIANPOINT((3000.,1815.));
#104=IFCCARTESIANPOINT((-3000.,1815.));
#105=IFCCARTESIANPOINT((-3000.,-1815.));
#106=IFCDIRECTION((0.,0.,1.));
#107=IFCDIRECTION((1.,0.,0.));
#108=IFCDIRECTION((0.,0.,1.));
#109=IFCCARTESIANPOINT((0.,0.,0.));
#110=IFCAXIS2PLACEMENT3D(#109,#108,#107);
#111=IFCPOLYLINE((#101,#102,#103,#104,#105,#101));
#112=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#111);
#113=IFCEXTRUDEDAREASOLID(#112,#110,#106,1600.);
#114=IFCDIRECTION((1.,0.,0.));
#115=IFCDIRECTION((0.,0.,1.));
#116=IFCCARTESIANPOINT((-2500.,0.,400.));
#117=IFCAXIS2PLACEMENT3D(#116,#115,#114);
#118=IFCLOCALPLACEMENT(#65,#117);
#119=IFCOPENINGELEMENT('2SFAmcP6j7ngFXHkheLGEs',#5,$,$,$,#118,#100,$,.OPENING.);
#120=IFCRELVOIDSELEMENT('1oVimvkWPAf90h9gir8Rpv',#5,$,$,#40,#119);
#121=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#135));
#122=IFCPRODUCTDEFINITIONSHAPE($,$,(#121));
#123=IFCCARTESIANPOINT((-930.,-1500.));
#124=IFCCARTESIANPOINT((930.,-1500.));
#125=IFCCARTESIANPOINT((930.,1500.));
#126=IFCCARTESIANPOINT((-930.,1500.));
#127=IFCCARTESIANPOINT((-930.,-1500.));
#128=IFCDIRECTION((0.,0.,1.));
#129=IFCDIRECTION((1.,0.,0.));
#130=IFCDIRECTION((0.,0.,1.));
#131=IFCCARTESIANPOINT((0.,0.,0.));
#132=IFCAXIS2PLACEMENT3D(#131,#130,#129);
#133=IFCPOLYLINE((#123,#124,#125,#126,#127,#123));
#134=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#133);
#135=IFCEXTRUDEDAREASOLID(#134,#132,#128,1600.);
#136=IFCDIRECTION((1.,0.,0.));
#137=IFCDIRECTION((0.,0.,1.));
#138=IFCCARTESIANPOINT((3000.,0.,400.));
#139=IFCAXIS2PLACEMENT3D(#138,#137,#136);
#140=IFCLOCALPLACEMENT(#37,#139);
#141=IFCOPENINGELEMENT('0vU7dryu95R8Tsq6mf0HAl',#5,$,$,$,#140,#122,$,.OPENING.);
#142=IFCRELVOIDSELEMENT('3ylYPA6kvE08JgnMNPAB$m',#5,$,$,#40,#141);
#143=IFCDIRECTION((1.,0.,0.));
#144=IFCDIRECTION((0.,0.,1.));
#145=IFCCARTESIANPOINT((0.,0.,0.));
#146=IFCAXIS2PLACEMENT3D(#145,#144,#143);
#147=IFCLOCALPLACEMENT(#37,#146);
#148=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#161));
#149=IFCDIRECTION((0.,-0.707107,0.707107));
#150=IFCDIRECTION((1.,0.,0.));
#151=IFCDIRECTION((0.,1.,0.));
#152=IFCCARTESIANPOINT((0.,0.,0.));
#153=IFCAXIS2PLACEMENT3D(#152,#151,#150);
#154=IFCCARTESIANPOINT((-5100.,-180.));
#155=IFCCARTESIANPOINT((5100.,-180.));
#156=IFCCARTESIANPOINT((5100.,180.));
#157=IFCCARTESIANPOINT((-5100.,180.));
#158=IFCCARTESIANPOINT((-5100.,-180.));
#159=IFCPOLYLINE((#154,#155,#156,#157,#158,#154));
#160=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#159);
#161=IFCEXTRUDEDAREASOLID(#160,#153,#149,4101.22);
#162=IFCDIRECTION((1.,0.,0.));
#163=IFCDIRECTION((0.,0.,1.));
#164=IFCCARTESIANPOINT((0.,0.,0.));
#165=IFCAXIS2PLACEMENT3D(#164,#163,#162);
#166=IFCREPRESENTATIONMAP(#165,#148);
#167=IFCCARTESIANPOINT((0.,0.,0.));
#168=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#167,$,$);
#169=IFCMAPPEDITEM(#166,#168);
#170=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#169));
#171=IFCPRODUCTDEFINITIONSHAPE($,$,(#170));
#172=IFCDIRECTION((1.,0.,0.));
#173=IFCDIRECTION((0.,0.,1.));
#174=IFCCARTESIANPOINT((0.,-400.,2700.));
#175=IFCAXIS2PLACEMENT3D(#174,#173,#172);
#176=IFCLOCALPLACEMENT(#147,#175);
#177=IFCCARTESIANPOINT((0.,0.,0.));
#178=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#177,$,$);
#179=IFCMAPPEDITEM(#166,#178);
#180=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#179));
#181=IFCPRODUCTDEFINITIONSHAPE($,$,(#180));
#182=IFCDIRECTION((-1.,0.,0.));
#183=IFCDIRECTION((0.,0.,1.));
#184=IFCCARTESIANPOINT((0.,5400.,2700.));
#185=IFCAXIS2PLACEMENT3D(#184,#183,#182);
#186=IFCLOCALPLACEMENT(#147,#185);
#187=IFCRELAGGREGATES('1EKtlXFYHDfOKET4ab3SjY',#5,$,$,#188,(#189,#190));
#188=IFCROOF('1C6JH2QOP0lwj7ou2leSLH',#5,'Roof',$,$,#147,$,$,.GABLE_ROOF.);
#189=IFCSLAB('2KoBar2pfAWBou8q$ldGHy',#5,'South roof',$,$,#176,#171,$,.ROOF.);
#190=IFCSLAB('3lPsczHcDCwepFiJhZqz9q',#5,'North roof',$,$,#186,#181,$,.ROOF.);
#191=IFCRELCONTAINEDINSPATIALSTRUCTURE('2s1GzjfAX94gJtU718BHIG',#5,$,$,(#188),#38);
#192=IFCCOLOURRGB($,0.24,0.08,0.04);
#193=IFCSURFACESTYLERENDERING(#192,$,$,$,$,$,$,$,.FLAT.);
#194=IFCSURFACESTYLE($,.BOTH.,(#193));
#195=IFCPRESENTATIONSTYLEASSIGNMENT((#194));
#196=IFCSTYLEDITEM(#161,(#195),$);
#197=IFCPRODUCTDEFINITIONSHAPE($,$,(#198,#199));
#198=IFCSHAPEREPRESENTATION(#11,'Axis','Curve2D',(#215));
#199=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#212));
#200=IFCCARTESIANPOINT((-5000.,-180.));
#201=IFCCARTESIANPOINT((5000.,-180.));
#202=IFCCARTESIANPOINT((5000.,180.));
#203=IFCCARTESIANPOINT((-5000.,180.));
#204=IFCCARTESIANPOINT((-5000.,-180.));
#205=IFCDIRECTION((0.,0.,1.));
#206=IFCDIRECTION((1.,0.,0.));
#207=IFCDIRECTION((0.,0.,1.));
#208=IFCCARTESIANPOINT((0.,0.,0.));
#209=IFCAXIS2PLACEMENT3D(#208,#207,#206);
#210=IFCPOLYLINE((#200,#201,#202,#203,#204,#200));
#211=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#210);
#212=IFCEXTRUDEDAREASOLID(#211,#209,#205,3000.);
#213=IFCCARTESIANPOINT((-5000.,0.));
#214=IFCCARTESIANPOINT((5000.,0.));
#215=IFCPOLYLINE((#213,#214));
#216=IFCDIRECTION((1.,0.,0.));
#217=IFCDIRECTION((0.,0.,1.));
#218=IFCCARTESIANPOINT((0.,5000.,0.));
#219=IFCAXIS2PLACEMENT3D(#218,#217,#216);
#220=IFCLOCALPLACEMENT(#37,#219);
#221=IFCWALLSTANDARDCASE('3xUPAVO39FGgNkCUQqf4JV',#5,'North wall',$,$,#220,#197,$,.STANDARD.);
#222=IFCRELCONTAINEDINSPATIALSTRUCTURE('2qGX1RUfPBo9AvGfHpx6Ua',#5,$,$,(#221),#38);
#223=IFCSTYLEDITEM(#215,(#69),$);
#224=IFCSTYLEDITEM(#212,(#69),$);
#225=IFCSHAPEREPRESENTATION(#11,'Body','Clipping',(#252));
#226=IFCCARTESIANPOINT((-2500.,-180.));
#227=IFCCARTESIANPOINT((2500.,-180.));
#228=IFCCARTESIANPOINT((2500.,180.));
#229=IFCCARTESIANPOINT((-2500.,180.));
#230=IFCCARTESIANPOINT((-2500.,-180.));
#231=IFCDIRECTION((0.,0.,1.));
#232=IFCDIRECTION((1.,0.,0.));
#233=IFCDIRECTION((0.,0.,1.));
#234=IFCCARTESIANPOINT((0.,0.,0.));
#235=IFCAXIS2PLACEMENT3D(#234,#233,#232);
#236=IFCPOLYLINE((#226,#227,#228,#229,#230,#226));
#237=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#236);
#238=IFCEXTRUDEDAREASOLID(#237,#235,#231,6000.);
#239=IFCDIRECTION((1.,0.,0.));
#240=IFCDIRECTION((-1.,0.,1.));
#241=IFCCARTESIANPOINT((-2500.,0.,3000.));
#242=IFCAXIS2PLACEMENT3D(#241,#240,#239);
#243=IFCPLANE(#242);
#244=IFCHALFSPACESOLID(#243,.F.);
#245=IFCBOOLEANCLIPPINGRESULT(.DIFFERENCE.,#238,#244);
#246=IFCDIRECTION((1.,0.,0.));
#247=IFCDIRECTION((1.,0.,1.));
#248=IFCCARTESIANPOINT((2500.,0.,3000.));
#249=IFCAXIS2PLACEMENT3D(#248,#247,#246);
#250=IFCPLANE(#249);
#251=IFCHALFSPACESOLID(#250,.F.);
#252=IFCBOOLEANCLIPPINGRESULT(.DIFFERENCE.,#245,#251);
#253=IFCDIRECTION((1.,0.,0.));
#254=IFCDIRECTION((0.,0.,1.));
#255=IFCCARTESIANPOINT((0.,0.,0.));
#256=IFCAXIS2PLACEMENT3D(#255,#254,#253);
#257=IFCREPRESENTATIONMAP(#256,#225);
#258=IFCCARTESIANPOINT((0.,0.,0.));
#259=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#258,$,$);
#260=IFCMAPPEDITEM(#257,#259);
#261=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#260));
#262=IFCPRODUCTDEFINITIONSHAPE($,$,(#261,#283));
#263=IFCDIRECTION((0.,1.,0.));
#264=IFCDIRECTION((0.,0.,1.));
#265=IFCCARTESIANPOINT((4820.,2500.,0.));
#266=IFCAXIS2PLACEMENT3D(#265,#264,#263);
#267=IFCLOCALPLACEMENT(#37,#266);
#268=IFCWALLSTANDARDCASE('3hw7qrktPAl8j6w3qKhwKm',#5,'East wall',$,$,#267,#262,$,.STANDARD.);
#269=IFCRELCONTAINEDINSPATIALSTRUCTURE('2M52LGcrD2OQqyhonWX1xO',#5,$,$,(#268),#38);
#270=IFCSTYLEDITEM(#252,(#69),$);
#271=IFCCARTESIANPOINT((0.,0.,0.));
#272=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#271,$,$);
#273=IFCMAPPEDITEM(#257,#272);
#274=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#273));
#275=IFCPRODUCTDEFINITIONSHAPE($,$,(#274,#287));
#276=IFCDIRECTION((0.,-1.,0.));
#277=IFCDIRECTION((0.,0.,1.));
#278=IFCCARTESIANPOINT((-4820.,2500.,0.));
#279=IFCAXIS2PLACEMENT3D(#278,#277,#276);
#280=IFCLOCALPLACEMENT(#37,#279);
#281=IFCWALLSTANDARDCASE('1hwEPyGUD1vwPpm508N9dQ',#5,'West wall',$,$,#280,#275,$,.STANDARD.);
#282=IFCRELCONTAINEDINSPATIALSTRUCTURE('3wvI53c8TA$evr0ReG0nFI',#5,$,$,(#281),#38);
#283=IFCSHAPEREPRESENTATION(#11,'Axis','Curve2D',(#286));
#284=IFCCARTESIANPOINT((-2500.,0.));
#285=IFCCARTESIANPOINT((2500.,0.));
#286=IFCPOLYLINE((#284,#285));
#287=IFCSHAPEREPRESENTATION(#11,'Axis','Curve2D',(#290));
#288=IFCCARTESIANPOINT((-2500.,0.));
#289=IFCCARTESIANPOINT((2500.,0.));
#290=IFCPOLYLINE((#288,#289));
#291=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#305));
#292=IFCPRODUCTDEFINITIONSHAPE($,$,(#291));
#293=IFCCARTESIANPOINT((-3000.,-1815.));
#294=IFCCARTESIANPOINT((3000.,-1815.));
#295=IFCCARTESIANPOINT((3000.,1815.));
#296=IFCCARTESIANPOINT((-3000.,1815.));
#297=IFCCARTESIANPOINT((-3000.,-1815.));
#298=IFCDIRECTION((0.,0.,1.));
#299=IFCDIRECTION((1.,0.,0.));
#300=IFCDIRECTION((0.,0.,1.));
#301=IFCCARTESIANPOINT((0.,0.,0.));
#302=IFCAXIS2PLACEMENT3D(#301,#300,#299);
#303=IFCPOLYLINE((#293,#294,#295,#296,#297,#293));
#304=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#303);
#305=IFCEXTRUDEDAREASOLID(#304,#302,#298,1600.);
#306=IFCDIRECTION((0.,1.,0.));
#307=IFCDIRECTION((0.,0.,1.));
#308=IFCCARTESIANPOINT((2500.,2320.,400.));
#309=IFCAXIS2PLACEMENT3D(#308,#307,#306);
#310=IFCLOCALPLACEMENT(#280,#309);
#311=IFCOPENINGELEMENT('1RBjKX3ZvFfwXeqVcf3G6M',#5,$,$,$,#310,#292,$,.OPENING.);
#312=IFCRELVOIDSELEMENT('1Icc3JNvz4Wh$epGd3tWdH',#5,$,$,#281,#311);
#313=IFCCARTESIANPOINT((-10000.,-10000.,-4130.));
#314=IFCCARTESIANPOINT((-10000.,10000.,-7130.));
#315=IFCCARTESIANPOINT((-10000.,-6466.6,-4289.81));
#316=IFCCARTESIANPOINT((-10000.,-3224.67,-4743.01));
#317=IFCCARTESIANPOINT((-10000.,742.019,-5623.35));
#318=IFCCARTESIANPOINT((-10000.,5141.19,-6632.33));
#319=IFCCARTESIANPOINT((-10000.,7488.65,-6988.88));
#320=IFCCARTESIANPOINT((10000.,-10000.,-6130.));
#321=IFCCARTESIANPOINT((-2188.73,-10000.,-5250.12));
#322=IFCCARTESIANPOINT((927.846,-10000.,-5637.39));
#323=IFCCARTESIANPOINT((4197.14,-10000.,-5936.56));
#324=IFCCARTESIANPOINT((6849.13,-10000.,-6076.02));
#325=IFCCARTESIANPOINT((10000.,10000.,-8130.));
#326=IFCCARTESIANPOINT((10000.,-3134.81,-4969.51));
#327=IFCCARTESIANPOINT((10000.,0.,-4755.));
#328=IFCCARTESIANPOINT((10000.,2516.45,-4936.97));
#329=IFCCARTESIANPOINT((10000.,4758.24,-5453.27));
#330=IFCCARTESIANPOINT((10000.,6939.51,-6314.74));
#331=IFCCARTESIANPOINT((10000.,8435.27,-7114.84));
#332=IFCCARTESIANPOINT((-6270.29,10000.,-5653.1));
#333=IFCCARTESIANPOINT((-3497.25,10000.,-4894.57));
#334=IFCCARTESIANPOINT((-1173.64,10000.,-4583.59));
#335=IFCCARTESIANPOINT((942.227,10000.,-4629.55));
#336=IFCCARTESIANPOINT((3068.35,10000.,-5014.55));
#337=IFCCARTESIANPOINT((5687.83,10000.,-5907.6));
#338=IFCCARTESIANPOINT((7705.61,10000.,-6849.67));
#339=IFCCARTESIANPOINT((4403.69,1755.24,-160.9));
#340=IFCCARTESIANPOINT((1417.79,1816.7,-247.206));
#341=IFCCARTESIANPOINT((4412.1,909.967,-172.116));
#342=IFCCARTESIANPOINT((1386.65,3817.88,-310.918));
#343=IFCCARTESIANPOINT((-3883.09,9024.79,-3377.23));
#344=IFCCARTESIANPOINT((-5504.42,8187.37,-3253.06));
#345=IFCCARTESIANPOINT((-4094.5,8319.08,-2704.06));
#346=IFCCARTESIANPOINT((-5265.55,8931.27,-3868.));
#347=IFCCARTESIANPOINT((-2695.8,7990.86,-1990.47));
#348=IFCCARTESIANPOINT((-4208.62,7873.25,-2387.03));
#349=IFCCARTESIANPOINT((-2927.21,6389.12,-1254.02));
#350=IFCCARTESIANPOINT((-1593.75,6470.44,-1032.62));
#351=IFCCARTESIANPOINT((7273.54,-1034.03,-962.597));
#352=IFCCARTESIANPOINT((7138.85,-2754.32,-1315.97));
#353=IFCCARTESIANPOINT((8127.22,-2838.4,-1974.78));
#354=IFCCARTESIANPOINT((5742.46,-920.381,-457.885));
#355=IFCCARTESIANPOINT((5611.21,-2669.32,-786.4));
#356=IFCCARTESIANPOINT((8740.36,-3003.84,-2634.34));
#357=IFCCARTESIANPOINT((8118.17,-2933.97,-2000.3));
#358=IFCCARTESIANPOINT((7918.9,-4614.85,-2563.98));
#359=IFCCARTESIANPOINT((8577.88,-4664.23,-3143.56));
#360=IFCCARTESIANPOINT((5786.17,834.561,-315.255));
#361=IFCCARTESIANPOINT((4375.14,-860.721,-296.338));
#362=IFCCARTESIANPOINT((8747.74,-2909.6,-2610.95));
#363=IFCCARTESIANPOINT((-2614.45,8423.71,-2322.12));
#364=IFCCARTESIANPOINT((5776.23,1672.19,-305.031));
#365=IFCCARTESIANPOINT((-1462.1,8059.62,-1785.92));
#366=IFCCARTESIANPOINT((8245.86,-1148.3,-1630.9));
#367=IFCCARTESIANPOINT((9423.47,-3012.48,-3630.84));
#368=IFCCARTESIANPOINT((9419.16,-3104.74,-3649.56));
#369=IFCCARTESIANPOINT((-2463.76,9099.1,-3036.89));
#370=IFCCARTESIANPOINT((1281.89,6175.78,-682.867));
#371=IFCCARTESIANPOINT((-1617.49,6098.38,-931.782));
#372=IFCCARTESIANPOINT((-1722.54,3749.08,-613.703));
#373=IFCCARTESIANPOINT((1365.69,3817.85,-312.326));
#374=IFCCARTESIANPOINT((1301.6,6175.82,-682.169));
#375=IFCCARTESIANPOINT((-1415.83,8484.91,-2128.9));
#376=IFCCARTESIANPOINT((4316.26,3747.3,-268.218));
#377=IFCCARTESIANPOINT((7318.39,688.26,-808.575));
#378=IFCCARTESIANPOINT((1262.95,6547.46,-797.991));
#379=IFCCARTESIANPOINT((-8809.7,-4698.86,-3102.36));
#380=IFCCARTESIANPOINT((-8608.12,-6382.51,-3282.77));
#381=IFCCARTESIANPOINT((1282.37,6547.51,-797.453));
#382=IFCCARTESIANPOINT((8844.48,-1245.95,-2293.46));
#383=IFCCARTESIANPOINT((-1330.1,9142.58,-2871.46));
#384=IFCCARTESIANPOINT((7308.19,1510.03,-801.758));
#385=IFCCARTESIANPOINT((8285.36,539.52,-1481.97));
#386=IFCCARTESIANPOINT((9480.02,-1388.03,-3374.23));
#387=IFCCARTESIANPOINT((5672.84,3648.18,-441.555));
#388=IFCCARTESIANPOINT((8876.68,411.636,-2157.56));
#389=IFCCARTESIANPOINT((-6056.28,-4541.56,-1708.84));
#390=IFCCARTESIANPOINT((-7430.37,-4605.99,-2214.34));
#391=IFCCARTESIANPOINT((-7129.69,-6317.26,-2623.29));
#392=IFCCARTESIANPOINT((1157.95,8125.18,-1636.07));
#393=IFCCARTESIANPOINT((-7646.46,-2936.95,-2052.37));
#394=IFCCARTESIANPOINT((-8961.15,-2970.6,-3096.44));
#395=IFCCARTESIANPOINT((-8954.57,-3063.16,-3092.9));
#396=IFCCARTESIANPOINT((1175.78,8125.22,-1636.35));
#397=IFCCARTESIANPOINT((8276.37,1344.55,-1479.3));
#398=IFCCARTESIANPOINT((-7656.28,-2842.25,-2048.62));
#399=IFCCARTESIANPOINT((4077.44,6101.57,-775.218));
#400=IFCCARTESIANPOINT((-9047.4,-1340.55,-3216.83));
#401=IFCCARTESIANPOINT((-1353.19,-8037.34,-3143.97));
#402=IFCCARTESIANPOINT((-2504.35,-8043.8,-3126.42));
#403=IFCCARTESIANPOINT((1138.32,8543.35,-2010.52));
#404=IFCCARTESIANPOINT((1121.03,8543.31,-2010.));
#405=IFCCARTESIANPOINT((-4582.93,-4492.82,-1389.47));
#406=IFCCARTESIANPOINT((-4301.53,-6238.81,-2005.98));
#407=IFCCARTESIANPOINT((-2762.03,-6214.64,-1863.86));
#408=IFCCARTESIANPOINT((8869.36,1201.98,-2158.74));
#409=IFCCARTESIANPOINT((-6284.77,-2848.68,-1471.23));
#410=IFCCARTESIANPOINT((7202.1,3452.64,-971.763));
#411=IFCCARTESIANPOINT((9498.84,224.712,-3267.93));
#412=IFCCARTESIANPOINT((-6295.16,-2752.44,-1463.97));
#413=IFCCARTESIANPOINT((-7784.93,-1170.03,-2054.6));
#414=IFCCARTESIANPOINT((1052.65,9184.12,-2810.42));
#415=IFCCARTESIANPOINT((1068.9,9184.15,-2811.35));
#416=IFCCARTESIANPOINT((9494.56,993.276,-3275.08));
#417=IFCCARTESIANPOINT((8182.93,3251.87,-1665.83));
#418=IFCCARTESIANPOINT((-2962.61,-4456.82,-1183.51));
#419=IFCCARTESIANPOINT((-1499.78,-6202.65,-1807.));
#420=IFCCARTESIANPOINT((5326.58,6368.06,-1183.46));
#421=IFCCARTESIANPOINT((4023.45,6474.1,-918.882));
#422=IFCCARTESIANPOINT((5390.42,5994.58,-1025.02));
#423=IFCCARTESIANPOINT((-4785.16,-2781.,-1106.92));
#424=IFCCARTESIANPOINT((8793.16,3078.35,-2350.05));
#425=IFCCARTESIANPOINT((-4794.35,-2683.53,-1097.63));
#426=IFCCARTESIANPOINT((-6431.19,-1049.94,-1413.17));
#427=IFCCARTESIANPOINT((6846.79,6157.37,-1800.87));
#428=IFCCARTESIANPOINT((1087.6,-8036.09,-3280.96));
#429=IFCCARTESIANPOINT((1071.07,-8036.04,-3279.56));
#430=IFCCARTESIANPOINT((6912.29,5782.15,-1629.72));
#431=IFCCARTESIANPOINT((9450.02,2823.7,-3461.79));
#432=IFCCARTESIANPOINT((-7827.77,496.777,-2162.92));
#433=IFCCARTESIANPOINT((-1613.88,-4438.11,-1079.49));
#434=IFCCARTESIANPOINT((-7818.03,1291.58,-2241.67));
#435=IFCCARTESIANPOINT((-9069.58,1049.73,-3546.42));
#436=IFCCARTESIANPOINT((-9076.12,278.241,-3423.93));
#437=IFCCARTESIANPOINT((-3106.76,-2729.79,-865.124));
#438=IFCCARTESIANPOINT((4848.25,8409.7,-2641.28));
#439=IFCCARTESIANPOINT((3618.95,8488.97,-2301.74));
#440=IFCCARTESIANPOINT((3724.16,8063.98,-1891.21));
#441=IFCCARTESIANPOINT((4972.66,7974.82,-2217.55));
#442=IFCCARTESIANPOINT((7869.97,5940.16,-2517.94));
#443=IFCCARTESIANPOINT((7927.66,5563.21,-2345.49));
#444=IFCCARTESIANPOINT((-3113.31,-2631.32,-854.22));
#445=IFCCARTESIANPOINT((4617.77,9089.35,-3506.44));
#446=IFCCARTESIANPOINT((3424.05,9145.61,-3154.54));
#447=IFCCARTESIANPOINT((-4914.75,-956.793,-1013.28));
#448=IFCCARTESIANPOINT((8537.98,5752.,-3186.97));
#449=IFCCARTESIANPOINT((8585.03,5373.61,-3019.59));
#450=IFCCARTESIANPOINT((1206.29,-6198.6,-1828.29));
#451=IFCCARTESIANPOINT((1188.,-6198.53,-1827.46));
#452=IFCCARTESIANPOINT((6355.95,8251.83,-3300.83));
#453=IFCCARTESIANPOINT((6483.62,7797.32,-2875.36));
#454=IFCCARTESIANPOINT((-6476.49,651.379,-1464.6));
#455=IFCCARTESIANPOINT((9300.87,5475.42,-4233.5));
#456=IFCCARTESIANPOINT((-1695.89,-2702.04,-729.986));
#457=IFCCARTESIANPOINT((9328.36,5094.94,-4081.15));
#458=IFCCARTESIANPOINT((7550.08,7614.11,-3575.65));
#459=IFCCARTESIANPOINT((-6466.19,1462.94,-1514.07));
#460=IFCCARTESIANPOINT((6119.45,8977.27,-4151.68));
#461=IFCCARTESIANPOINT((-1699.62,-2602.99,-717.605));
#462=IFCCARTESIANPOINT((7437.63,8088.85,-3985.46));
#463=IFCCARTESIANPOINT((8185.46,7947.59,-4586.4));
#464=IFCCARTESIANPOINT((8277.15,7455.32,-4199.38));
#465=IFCCARTESIANPOINT((-7716.69,3177.11,-2496.75));
#466=IFCCARTESIANPOINT((7229.31,8861.55,-4793.76));
#467=IFCCARTESIANPOINT((-3199.13,-884.91,-743.322));
#468=IFCCARTESIANPOINT((8015.6,8761.23,-5343.02));
#469=IFCCARTESIANPOINT((-4954.84,772.184,-1026.66));
#470=IFCCARTESIANPOINT((1279.02,-4428.3,-994.721));
#471=IFCCARTESIANPOINT((9094.81,7739.83,-5483.7));
#472=IFCCARTESIANPOINT((9148.41,7221.81,-5141.57));
#473=IFCCARTESIANPOINT((8995.53,8613.67,-6146.02));
#474=IFCCARTESIANPOINT((1298.67,-4428.38,-994.813));
#475=IFCCARTESIANPOINT((-8816.07,5159.01,-4424.97));
#476=IFCCARTESIANPOINT((-9001.65,2886.18,-3890.59));
#477=IFCCARTESIANPOINT((-8349.17,-8139.49,-3697.07));
#478=IFCCARTESIANPOINT((-9168.43,-8162.24,-3909.));
#479=IFCCARTESIANPOINT((-8197.92,-9054.72,-4015.35));
#480=IFCCARTESIANPOINT((4679.86,-8065.62,-3789.1));
#481=IFCCARTESIANPOINT((3476.55,-8051.4,-3567.21));
#482=IFCCARTESIANPOINT((-7719.37,-7222.36,-3177.91));
#483=IFCCARTESIANPOINT((-6743.42,-8101.96,-3395.87));
#484=IFCCARTESIANPOINT((-4945.72,1597.18,-1054.53));
#485=IFCCARTESIANPOINT((-8019.39,-5491.33,-2730.13));
#486=IFCCARTESIANPOINT((-8774.13,5538.58,-4528.69));
#487=IFCCARTESIANPOINT((-9475.93,-3941.84,-3760.6));
#488=IFCCARTESIANPOINT((-1748.44,-844.698,-583.573));
#489=IFCCARTESIANPOINT((-9304.05,-3649.36,-3529.4));
#490=IFCCARTESIANPOINT((-3036.03,-9005.2,-4025.69));
#491=IFCCARTESIANPOINT((-4232.94,-8691.88,-3689.67));
#492=IFCCARTESIANPOINT((-4371.52,-9013.88,-3977.08));
#493=IFCCARTESIANPOINT((-3940.04,-8057.26,-3151.16));
#494=IFCCARTESIANPOINT((-8245.09,-3818.62,-2536.82));
#495=IFCCARTESIANPOINT((-6359.03,3383.76,-1697.73));
#496=IFCCARTESIANPOINT((-3227.71,866.568,-727.728));
#497=IFCCARTESIANPOINT((-3386.69,-7125.54,-2442.69));
#498=IFCCARTESIANPOINT((-8404.49,-2074.38,-2522.));
#499=IFCCARTESIANPOINT((-9581.25,-267.935,-4242.93));
#500=IFCCARTESIANPOINT((-2035.89,-7111.92,-2389.3));
#501=IFCCARTESIANPOINT((1344.43,-2683.14,-555.737));
#502=IFCCARTESIANPOINT((-131.714,-8997.,-4219.32));
#503=IFCCARTESIANPOINT((-8479.39,-427.498,-2629.76));
#504=IFCCARTESIANPOINT((1365.07,-2683.21,-555.109));
#505=IFCCARTESIANPOINT((-149.026,-7103.92,-2399.27));
#506=IFCCARTESIANPOINT((1347.4,-2583.46,-538.818));
#507=IFCCARTESIANPOINT((-9564.56,1806.67,-4659.7));
#508=IFCCARTESIANPOINT((2131.63,-9000.98,-4447.55));
#509=IFCCARTESIANPOINT((1368.09,-2583.53,-538.151));
#510=IFCCARTESIANPOINT((-8447.99,2109.29,-2952.78));
#511=IFCCARTESIANPOINT((2403.48,-7113.01,-2569.55));
#512=IFCCARTESIANPOINT((3809.81,-6223.7,-2080.23));
#513=IFCCARTESIANPOINT((-8289.13,4186.02,-3348.86));
#514=IFCCARTESIANPOINT((-7439.88,5474.58,-2993.42));
#515=IFCCARTESIANPOINT((5188.95,-9020.53,-4882.53));
#516=IFCCARTESIANPOINT((6183.16,-8089.95,-4164.56));
#517=IFCCARTESIANPOINT((2614.5,-5317.96,-1415.07));
#518=IFCCARTESIANPOINT((4069.23,-4460.86,-1155.09));
#519=IFCCARTESIANPOINT((-3221.21,1702.5,-739.507));
#520=IFCCARTESIANPOINT((-9348.95,6229.88,-5697.98));
#521=IFCCARTESIANPOINT((-8541.57,7274.08,-5075.45));
#522=IFCCARTESIANPOINT((5644.13,-7161.33,-3195.07));
#523=IFCCARTESIANPOINT((5073.95,-6248.08,-2340.18));
#524=IFCCARTESIANPOINT((6587.55,-6290.61,-2847.21));
#525=IFCCARTESIANPOINT((2773.25,-3566.,-748.638));
#526=IFCCARTESIANPOINT((-7377.31,5851.51,-3105.22));
#527=IFCCARTESIANPOINT((4255.67,-2722.78,-609.321));
#528=IFCCARTESIANPOINT((4264.14,-2623.53,-586.585));
#529=IFCCARTESIANPOINT((-7954.42,6564.45,-4028.72));
#530=IFCCARTESIANPOINT((-7030.42,7537.59,-3774.97));
#531=IFCCARTESIANPOINT((-9189.52,8095.52,-6173.19));
#532=IFCCARTESIANPOINT((-4850.89,3546.3,-1180.26));
#533=IFCCARTESIANPOINT((-8459.82,7786.17,-5264.87));
#534=IFCCARTESIANPOINT((-8308.37,8646.47,-5620.06));
#535=IFCCARTESIANPOINT((-9069.6,9270.79,-6486.5));
#536=IFCCARTESIANPOINT((-1764.7,920.401,-547.798));
#537=IFCCARTESIANPOINT((8700.19,-8148.38,-5114.77));
#538=IFCCARTESIANPOINT((8061.36,-8131.35,-4829.26));
#539=IFCCARTESIANPOINT((8923.12,-9064.9,-5674.88));
#540=IFCCARTESIANPOINT((9022.27,-8157.56,-5271.53));
#541=IFCCARTESIANPOINT((-7314.96,9375.59,-5466.92));
#542=IFCCARTESIANPOINT((-6682.57,8812.83,-4558.42));
#543=IFCCARTESIANPOINT((9139.65,-6984.56,-4797.61));
#544=IFCCARTESIANPOINT((8351.8,-6364.28,-3861.82));
#545=IFCCARTESIANPOINT((7285.42,-8112.71,-4524.92));
#546=IFCCARTESIANPOINT((9557.14,-7294.87,-5258.77));
#547=IFCCARTESIANPOINT((-4369.33,9513.2,-4250.08));
#548=IFCCARTESIANPOINT((-1761.,1762.95,-550.41));
#549=IFCCARTESIANPOINT((-6066.33,5699.08,-2133.81));
#550=IFCCARTESIANPOINT((9700.84,-3917.32,-4411.87));
#551=IFCCARTESIANPOINT((-608.959,7591.2,-1373.02));
#552=IFCCARTESIANPOINT((1386.35,-812.591,-336.314));
#553=IFCCARTESIANPOINT((-159.866,7343.93,-1200.71));
#554=IFCCARTESIANPOINT((1407.62,-812.632,-335.082));
#555=IFCCARTESIANPOINT((302.815,7087.61,-1048.6));
#556=IFCCARTESIANPOINT((-1791.01,9592.41,-3692.89));
#557=IFCCARTESIANPOINT((2154.41,7092.13,-1022.11));
#558=IFCCARTESIANPOINT((5380.72,-4494.82,-1397.64));
#559=IFCCARTESIANPOINT((-143.035,8860.18,-2378.65));
#560=IFCCARTESIANPOINT((-6000.17,6074.18,-2243.36));
#561=IFCCARTESIANPOINT((4893.49,4899.26,-536.954));
#562=IFCCARTESIANPOINT((2568.8,7350.03,-1182.95));
#563=IFCCARTESIANPOINT((-3153.61,3674.66,-822.59));
#564=IFCCARTESIANPOINT((-130.374,9620.14,-3585.2));
#565=IFCCARTESIANPOINT((6634.09,4319.65,-896.975));
#566=IFCCARTESIANPOINT((2968.39,7597.97,-1379.87));
#567=IFCCARTESIANPOINT((6335.1,4747.47,-894.888));
#568=IFCCARTESIANPOINT((6027.4,5170.03,-915.608));
#569=IFCCARTESIANPOINT((4527.87,7251.61,-1461.3));
#570=IFCCARTESIANPOINT((-6908.47,8020.53,-4035.72));
#571=IFCCARTESIANPOINT((7589.09,4531.,-1544.96));
#572=IFCCARTESIANPOINT((7641.62,-6330.95,-3383.55));
#573=IFCCARTESIANPOINT((2119.05,9622.25,-3762.61));
#574=IFCCARTESIANPOINT((-4591.83,5876.12,-1555.64));
#575=IFCCARTESIANPOINT((5929.18,7106.7,-1908.47));
#576=IFCCARTESIANPOINT((9711.3,3799.37,-4383.14));
#577=IFCCARTESIANPOINT((1399.31,967.078,-257.195));
#578=IFCCARTESIANPOINT((3829.37,9591.52,-4158.95));
#579=IFCCARTESIANPOINT((8828.65,6485.65,-4110.41));
#580=IFCCARTESIANPOINT((1420.78,967.063,-255.651));
#581=IFCCARTESIANPOINT((6472.87,9481.11,-5241.01));
#582=IFCCARTESIANPOINT((8381.7,9351.27,-6385.87));
#583=IFCCARTESIANPOINT((-9036.11,-9529.99,-4166.9));
#584=IFCCARTESIANPOINT((-7347.58,-9359.22,-4156.32));
#585=IFCCARTESIANPOINT((-4533.27,6249.8,-1660.));
#586=IFCCARTESIANPOINT((-9624.,-7298.75,-4000.07));
#587=IFCCARTESIANPOINT((6902.34,-4555.81,-1940.75));
#588=IFCCARTESIANPOINT((-6922.06,-9515.49,-4269.47));
#589=IFCCARTESIANPOINT((5601.19,-2767.86,-811.583));
#590=IFCCARTESIANPOINT((7895.,-9521.79,-5763.77));
#591=IFCCARTESIANPOINT((9455.58,-9072.92,-5815.13));
#592=IFCCARTESIANPOINT((9178.49,-4279.63,-3671.17));
#593=IFCCARTESIANPOINT((9324.19,-4734.88,-4048.89));
#594=IFCCARTESIANPOINT((9830.79,-5157.49,-4966.41));
#595=IFCCARTESIANPOINT((1665.05,9121.39,-2763.35));
#596=IFCCARTESIANPOINT((-5633.36,7725.18,-2956.58));
#597=IFCCARTESIANPOINT((2298.34,8927.96,-2568.54));
#598=IFCCARTESIANPOINT((1396.36,1816.71,-248.793));
#599=IFCCARTESIANPOINT((2848.69,9101.64,-2930.66));
#600=IFCCARTESIANPOINT((9193.94,4599.93,-3582.19));
#601=IFCCARTESIANPOINT((-4420.49,-9750.,-4685.26));
#602=IFCCARTESIANPOINT((9146.72,-6154.35,-4427.54));
#603=IFCCARTESIANPOINT((9150.41,-5744.42,-4248.73));
#604=IFCCARTESIANPOINT((-2968.95,6016.49,-1153.51));
#605=IFCCARTESIANPOINT((8559.11,4091.22,-2363.29));
#606=IFCCARTESIANPOINT((8427.82,4596.53,-2409.77));
#607=IFCCARTESIANPOINT((-5579.36,-8904.65,-3864.47));
#608=IFCCARTESIANPOINT((7128.57,-2851.41,-1342.52));
#609=IFCCARTESIANPOINT((-6504.8,-7752.66,-3159.95));
#610=IFCCARTESIANPOINT((-6386.18,-7578.88,-3041.74));
#611=IFCCARTESIANPOINT((-5706.92,-7128.43,-2675.3));
#612=IFCPOLYLOOP((#339,#340,#341));
#613=IFCFACEOUTERBOUND(#612,.T.);
#614=IFCFACE((#613));
#615=IFCPOLYLOOP((#339,#342,#340));
#616=IFCFACEOUTERBOUND(#615,.T.);
#617=IFCFACE((#616));
#618=IFCPOLYLOOP((#343,#344,#345));
#619=IFCFACEOUTERBOUND(#618,.T.);
#620=IFCFACE((#619));
#621=IFCPOLYLOOP((#343,#346,#344));
#622=IFCFACEOUTERBOUND(#621,.T.);
#623=IFCFACE((#622));
#624=IFCPOLYLOOP((#347,#348,#349));
#625=IFCFACEOUTERBOUND(#624,.T.);
#626=IFCFACE((#625));
#627=IFCPOLYLOOP((#347,#345,#348));
#628=IFCFACEOUTERBOUND(#627,.T.);
#629=IFCFACE((#628));
#630=IFCPOLYLOOP((#347,#349,#350));
#631=IFCFACEOUTERBOUND(#630,.T.);
#632=IFCFACE((#631));
#633=IFCPOLYLOOP((#351,#352,#353));
#634=IFCFACEOUTERBOUND(#633,.T.);
#635=IFCFACE((#634));
#636=IFCPOLYLOOP((#351,#354,#355));
#637=IFCFACEOUTERBOUND(#636,.T.);
#638=IFCFACE((#637));
#639=IFCPOLYLOOP((#351,#355,#352));
#640=IFCFACEOUTERBOUND(#639,.T.);
#641=IFCFACE((#640));
#642=IFCPOLYLOOP((#356,#357,#358));
#643=IFCFACEOUTERBOUND(#642,.T.);
#644=IFCFACE((#643));
#645=IFCPOLYLOOP((#356,#358,#359));
#646=IFCFACEOUTERBOUND(#645,.T.);
#647=IFCFACE((#646));
#648=IFCPOLYLOOP((#360,#361,#354));
#649=IFCFACEOUTERBOUND(#648,.T.);
#650=IFCFACE((#649));
#651=IFCPOLYLOOP((#360,#341,#361));
#652=IFCFACEOUTERBOUND(#651,.T.);
#653=IFCFACE((#652));
#654=IFCPOLYLOOP((#362,#353,#357));
#655=IFCFACEOUTERBOUND(#654,.T.);
#656=IFCFACE((#655));
#657=IFCPOLYLOOP((#362,#357,#356));
#658=IFCFACEOUTERBOUND(#657,.T.);
#659=IFCFACE((#658));
#660=IFCPOLYLOOP((#363,#343,#345));
#661=IFCFACEOUTERBOUND(#660,.T.);
#662=IFCFACE((#661));
#663=IFCPOLYLOOP((#363,#345,#347));
#664=IFCFACEOUTERBOUND(#663,.T.);
#665=IFCFACE((#664));
#666=IFCPOLYLOOP((#364,#339,#341));
#667=IFCFACEOUTERBOUND(#666,.T.);
#668=IFCFACE((#667));
#669=IFCPOLYLOOP((#364,#341,#360));
#670=IFCFACEOUTERBOUND(#669,.T.);
#671=IFCFACE((#670));
#672=IFCPOLYLOOP((#365,#347,#350));
#673=IFCFACEOUTERBOUND(#672,.T.);
#674=IFCFACE((#673));
#675=IFCPOLYLOOP((#365,#363,#347));
#676=IFCFACEOUTERBOUND(#675,.T.);
#677=IFCFACE((#676));
#678=IFCPOLYLOOP((#366,#351,#353));
#679=IFCFACEOUTERBOUND(#678,.T.);
#680=IFCFACE((#679));
#681=IFCPOLYLOOP((#367,#356,#368));
#682=IFCFACEOUTERBOUND(#681,.T.);
#683=IFCFACE((#682));
#684=IFCPOLYLOOP((#367,#362,#356));
#685=IFCFACEOUTERBOUND(#684,.T.);
#686=IFCFACE((#685));
#687=IFCPOLYLOOP((#367,#368,#326));
#688=IFCFACEOUTERBOUND(#687,.T.);
#689=IFCFACE((#688));
#690=IFCPOLYLOOP((#369,#333,#343));
#691=IFCFACEOUTERBOUND(#690,.T.);
#692=IFCFACE((#691));
#693=IFCPOLYLOOP((#369,#343,#363));
#694=IFCFACEOUTERBOUND(#693,.T.);
#695=IFCFACE((#694));
#696=IFCPOLYLOOP((#370,#371,#372));
#697=IFCFACEOUTERBOUND(#696,.T.);
#698=IFCFACE((#697));
#699=IFCPOLYLOOP((#370,#372,#373));
#700=IFCFACEOUTERBOUND(#699,.T.);
#701=IFCFACE((#700));
#702=IFCPOLYLOOP((#374,#370,#373));
#703=IFCFACEOUTERBOUND(#702,.T.);
#704=IFCFACE((#703));
#705=IFCPOLYLOOP((#374,#373,#342));
#706=IFCFACEOUTERBOUND(#705,.T.);
#707=IFCFACE((#706));
#708=IFCPOLYLOOP((#375,#363,#365));
#709=IFCFACEOUTERBOUND(#708,.T.);
#710=IFCFACE((#709));
#711=IFCPOLYLOOP((#375,#369,#363));
#712=IFCFACEOUTERBOUND(#711,.T.);
#713=IFCFACE((#712));
#714=IFCPOLYLOOP((#376,#339,#364));
#715=IFCFACEOUTERBOUND(#714,.T.);
#716=IFCFACE((#715));
#717=IFCPOLYLOOP((#376,#342,#339));
#718=IFCFACEOUTERBOUND(#717,.T.);
#719=IFCFACE((#718));
#720=IFCPOLYLOOP((#376,#374,#342));
#721=IFCFACEOUTERBOUND(#720,.T.);
#722=IFCFACE((#721));
#723=IFCPOLYLOOP((#377,#360,#354));
#724=IFCFACEOUTERBOUND(#723,.T.);
#725=IFCFACE((#724));
#726=IFCPOLYLOOP((#377,#354,#351));
#727=IFCFACEOUTERBOUND(#726,.T.);
#728=IFCFACE((#727));
#729=IFCPOLYLOOP((#378,#370,#374));
#730=IFCFACEOUTERBOUND(#729,.T.);
#731=IFCFACE((#730));
#732=IFCPOLYLOOP((#379,#315,#380));
#733=IFCFACEOUTERBOUND(#732,.T.);
#734=IFCFACE((#733));
#735=IFCPOLYLOOP((#381,#378,#374));
#736=IFCFACEOUTERBOUND(#735,.T.);
#737=IFCFACE((#736));
#738=IFCPOLYLOOP((#382,#362,#367));
#739=IFCFACEOUTERBOUND(#738,.T.);
#740=IFCFACE((#739));
#741=IFCPOLYLOOP((#382,#353,#362));
#742=IFCFACEOUTERBOUND(#741,.T.);
#743=IFCFACE((#742));
#744=IFCPOLYLOOP((#382,#366,#353));
#745=IFCFACEOUTERBOUND(#744,.T.);
#746=IFCFACE((#745));
#747=IFCPOLYLOOP((#383,#369,#375));
#748=IFCFACEOUTERBOUND(#747,.T.);
#749=IFCFACE((#748));
#750=IFCPOLYLOOP((#384,#364,#360));
#751=IFCFACEOUTERBOUND(#750,.T.);
#752=IFCFACE((#751));
#753=IFCPOLYLOOP((#384,#360,#377));
#754=IFCFACEOUTERBOUND(#753,.T.);
#755=IFCFACE((#754));
#756=IFCPOLYLOOP((#385,#351,#366));
#757=IFCFACEOUTERBOUND(#756,.T.);
#758=IFCFACE((#757));
#759=IFCPOLYLOOP((#385,#377,#351));
#760=IFCFACEOUTERBOUND(#759,.T.);
#761=IFCFACE((#760));
#762=IFCPOLYLOOP((#386,#326,#327));
#763=IFCFACEOUTERBOUND(#762,.T.);
#764=IFCFACE((#763));
#765=IFCPOLYLOOP((#386,#367,#326));
#766=IFCFACEOUTERBOUND(#765,.T.);
#767=IFCFACE((#766));
#768=IFCPOLYLOOP((#386,#382,#367));
#769=IFCFACEOUTERBOUND(#768,.T.);
#770=IFCFACE((#769));
#771=IFCPOLYLOOP((#387,#376,#364));
#772=IFCFACEOUTERBOUND(#771,.T.);
#773=IFCFACE((#772));
#774=IFCPOLYLOOP((#387,#364,#384));
#775=IFCFACEOUTERBOUND(#774,.T.);
#776=IFCFACE((#775));
#777=IFCPOLYLOOP((#388,#366,#382));
#778=IFCFACEOUTERBOUND(#777,.T.);
#779=IFCFACE((#778));
#780=IFCPOLYLOOP((#388,#385,#366));
#781=IFCFACEOUTERBOUND(#780,.T.);
#782=IFCFACE((#781));
#783=IFCPOLYLOOP((#389,#390,#391));
#784=IFCFACEOUTERBOUND(#783,.T.);
#785=IFCFACE((#784));
#786=IFCPOLYLOOP((#392,#378,#381));
#787=IFCFACEOUTERBOUND(#786,.T.);
#788=IFCFACE((#787));
#789=IFCPOLYLOOP((#393,#390,#389));
#790=IFCFACEOUTERBOUND(#789,.T.);
#791=IFCFACE((#790));
#792=IFCPOLYLOOP((#393,#394,#395));
#793=IFCFACEOUTERBOUND(#792,.T.);
#794=IFCFACE((#793));
#795=IFCPOLYLOOP((#396,#392,#381));
#796=IFCFACEOUTERBOUND(#795,.T.);
#797=IFCFACE((#796));
#798=IFCPOLYLOOP((#397,#384,#377));
#799=IFCFACEOUTERBOUND(#798,.T.);
#800=IFCFACE((#799));
#801=IFCPOLYLOOP((#397,#377,#385));
#802=IFCFACEOUTERBOUND(#801,.T.);
#803=IFCFACE((#802));
#804=IFCPOLYLOOP((#398,#394,#393));
#805=IFCFACEOUTERBOUND(#804,.T.);
#806=IFCFACE((#805));
#807=IFCPOLYLOOP((#399,#374,#376));
#808=IFCFACEOUTERBOUND(#807,.T.);
#809=IFCFACE((#808));
#810=IFCPOLYLOOP((#400,#316,#394));
#811=IFCFACEOUTERBOUND(#810,.T.);
#812=IFCFACE((#811));
#813=IFCPOLYLOOP((#401,#402,#321));
#814=IFCFACEOUTERBOUND(#813,.T.);
#815=IFCFACE((#814));
#816=IFCPOLYLOOP((#403,#392,#396));
#817=IFCFACEOUTERBOUND(#816,.T.);
#818=IFCFACE((#817));
#819=IFCPOLYLOOP((#403,#404,#392));
#820=IFCFACEOUTERBOUND(#819,.T.);
#821=IFCFACE((#820));
#822=IFCPOLYLOOP((#405,#406,#407));
#823=IFCFACEOUTERBOUND(#822,.T.);
#824=IFCFACE((#823));
#825=IFCPOLYLOOP((#405,#389,#406));
#826=IFCFACEOUTERBOUND(#825,.T.);
#827=IFCFACE((#826));
#828=IFCPOLYLOOP((#408,#385,#388));
#829=IFCFACEOUTERBOUND(#828,.T.);
#830=IFCFACE((#829));
#831=IFCPOLYLOOP((#408,#397,#385));
#832=IFCFACEOUTERBOUND(#831,.T.);
#833=IFCFACE((#832));
#834=IFCPOLYLOOP((#409,#393,#389));
#835=IFCFACEOUTERBOUND(#834,.T.);
#836=IFCFACE((#835));
#837=IFCPOLYLOOP((#409,#389,#405));
#838=IFCFACEOUTERBOUND(#837,.T.);
#839=IFCFACE((#838));
#840=IFCPOLYLOOP((#410,#384,#397));
#841=IFCFACEOUTERBOUND(#840,.T.);
#842=IFCFACE((#841));
#843=IFCPOLYLOOP((#410,#387,#384));
#844=IFCFACEOUTERBOUND(#843,.T.);
#845=IFCFACE((#844));
#846=IFCPOLYLOOP((#411,#386,#327));
#847=IFCFACEOUTERBOUND(#846,.T.);
#848=IFCFACE((#847));
#849=IFCPOLYLOOP((#411,#382,#386));
#850=IFCFACEOUTERBOUND(#849,.T.);
#851=IFCFACE((#850));
#852=IFCPOLYLOOP((#411,#388,#382));
#853=IFCFACEOUTERBOUND(#852,.T.);
#854=IFCFACE((#853));
#855=IFCPOLYLOOP((#412,#398,#393));
#856=IFCFACEOUTERBOUND(#855,.T.);
#857=IFCFACE((#856));
#858=IFCPOLYLOOP((#412,#393,#409));
#859=IFCFACEOUTERBOUND(#858,.T.);
#860=IFCFACE((#859));
#861=IFCPOLYLOOP((#413,#398,#412));
#862=IFCFACEOUTERBOUND(#861,.T.);
#863=IFCFACE((#862));
#864=IFCPOLYLOOP((#414,#404,#403));
#865=IFCFACEOUTERBOUND(#864,.T.);
#866=IFCFACE((#865));
#867=IFCPOLYLOOP((#415,#414,#403));
#868=IFCFACEOUTERBOUND(#867,.T.);
#869=IFCFACE((#868));
#870=IFCPOLYLOOP((#415,#335,#414));
#871=IFCFACEOUTERBOUND(#870,.T.);
#872=IFCFACE((#871));
#873=IFCPOLYLOOP((#416,#408,#388));
#874=IFCFACEOUTERBOUND(#873,.T.);
#875=IFCFACE((#874));
#876=IFCPOLYLOOP((#416,#327,#328));
#877=IFCFACEOUTERBOUND(#876,.T.);
#878=IFCFACE((#877));
#879=IFCPOLYLOOP((#416,#411,#327));
#880=IFCFACEOUTERBOUND(#879,.T.);
#881=IFCFACE((#880));
#882=IFCPOLYLOOP((#416,#388,#411));
#883=IFCFACEOUTERBOUND(#882,.T.);
#884=IFCFACE((#883));
#885=IFCPOLYLOOP((#417,#397,#408));
#886=IFCFACEOUTERBOUND(#885,.T.);
#887=IFCFACE((#886));
#888=IFCPOLYLOOP((#417,#410,#397));
#889=IFCFACEOUTERBOUND(#888,.T.);
#890=IFCFACE((#889));
#891=IFCPOLYLOOP((#418,#407,#419));
#892=IFCFACEOUTERBOUND(#891,.T.);
#893=IFCFACE((#892));
#894=IFCPOLYLOOP((#420,#421,#399));
#895=IFCFACEOUTERBOUND(#894,.T.);
#896=IFCFACE((#895));
#897=IFCPOLYLOOP((#420,#399,#422));
#898=IFCFACEOUTERBOUND(#897,.T.);
#899=IFCFACE((#898));
#900=IFCPOLYLOOP((#418,#405,#407));
#901=IFCFACEOUTERBOUND(#900,.T.);
#902=IFCFACE((#901));
#903=IFCPOLYLOOP((#423,#409,#405));
#904=IFCFACEOUTERBOUND(#903,.T.);
#905=IFCFACE((#904));
#906=IFCPOLYLOOP((#423,#405,#418));
#907=IFCFACEOUTERBOUND(#906,.T.);
#908=IFCFACE((#907));
#909=IFCPOLYLOOP((#424,#417,#408));
#910=IFCFACEOUTERBOUND(#909,.T.);
#911=IFCFACE((#910));
#912=IFCPOLYLOOP((#424,#408,#416));
#913=IFCFACEOUTERBOUND(#912,.T.);
#914=IFCFACE((#913));
#915=IFCPOLYLOOP((#425,#409,#423));
#916=IFCFACEOUTERBOUND(#915,.T.);
#917=IFCFACE((#916));
#918=IFCPOLYLOOP((#425,#412,#409));
#919=IFCFACEOUTERBOUND(#918,.T.);
#920=IFCFACE((#919));
#921=IFCPOLYLOOP((#426,#413,#412));
#922=IFCFACEOUTERBOUND(#921,.T.);
#923=IFCFACE((#922));
#924=IFCPOLYLOOP((#426,#412,#425));
#925=IFCFACEOUTERBOUND(#924,.T.);
#926=IFCFACE((#925));
#927=IFCPOLYLOOP((#427,#420,#422));
#928=IFCFACEOUTERBOUND(#927,.T.);
#929=IFCFACE((#928));
#930=IFCPOLYLOOP((#428,#429,#322));
#931=IFCFACEOUTERBOUND(#930,.T.);
#932=IFCFACE((#931));
#933=IFCPOLYLOOP((#427,#422,#430));
#934=IFCFACEOUTERBOUND(#933,.T.);
#935=IFCFACE((#934));
#936=IFCPOLYLOOP((#431,#416,#328));
#937=IFCFACEOUTERBOUND(#936,.T.);
#938=IFCFACE((#937));
#939=IFCPOLYLOOP((#431,#424,#416));
#940=IFCFACEOUTERBOUND(#939,.T.);
#941=IFCFACE((#940));
#942=IFCPOLYLOOP((#432,#413,#426));
#943=IFCFACEOUTERBOUND(#942,.T.);
#944=IFCFACE((#943));
#945=IFCPOLYLOOP((#433,#418,#419));
#946=IFCFACEOUTERBOUND(#945,.T.);
#947=IFCFACE((#946));
#948=IFCPOLYLOOP((#434,#435,#436));
#949=IFCFACEOUTERBOUND(#948,.T.);
#950=IFCFACE((#949));
#951=IFCPOLYLOOP((#434,#436,#432));
#952=IFCFACEOUTERBOUND(#951,.T.);
#953=IFCFACE((#952));
#954=IFCPOLYLOOP((#437,#423,#418));
#955=IFCFACEOUTERBOUND(#954,.T.);
#956=IFCFACE((#955));
#957=IFCPOLYLOOP((#438,#439,#440));
#958=IFCFACEOUTERBOUND(#957,.T.);
#959=IFCFACE((#958));
#960=IFCPOLYLOOP((#437,#425,#423));
#961=IFCFACEOUTERBOUND(#960,.T.);
#962=IFCFACE((#961));
#963=IFCPOLYLOOP((#438,#440,#441));
#964=IFCFACEOUTERBOUND(#963,.T.);
#965=IFCFACE((#964));
#966=IFCPOLYLOOP((#437,#418,#433));
#967=IFCFACEOUTERBOUND(#966,.T.);
#968=IFCFACE((#967));
#969=IFCPOLYLOOP((#442,#427,#430));
#970=IFCFACEOUTERBOUND(#969,.T.);
#971=IFCFACE((#970));
#972=IFCPOLYLOOP((#442,#430,#443));
#973=IFCFACEOUTERBOUND(#972,.T.);
#974=IFCFACE((#973));
#975=IFCPOLYLOOP((#444,#425,#437));
#976=IFCFACEOUTERBOUND(#975,.T.);
#977=IFCFACE((#976));
#978=IFCPOLYLOOP((#445,#446,#439));
#979=IFCFACEOUTERBOUND(#978,.T.);
#980=IFCFACE((#979));
#981=IFCPOLYLOOP((#445,#439,#438));
#982=IFCFACEOUTERBOUND(#981,.T.);
#983=IFCFACE((#982));
#984=IFCPOLYLOOP((#447,#426,#425));
#985=IFCFACEOUTERBOUND(#984,.T.);
#986=IFCFACE((#985));
#987=IFCPOLYLOOP((#447,#425,#444));
#988=IFCFACEOUTERBOUND(#987,.T.);
#989=IFCFACE((#988));
#990=IFCPOLYLOOP((#448,#442,#443));
#991=IFCFACEOUTERBOUND(#990,.T.);
#992=IFCFACE((#991));
#993=IFCPOLYLOOP((#448,#443,#449));
#994=IFCFACEOUTERBOUND(#993,.T.);
#995=IFCFACE((#994));
#996=IFCPOLYLOOP((#450,#451,#429));
#997=IFCFACEOUTERBOUND(#996,.T.);
#998=IFCFACE((#997));
#999=IFCPOLYLOOP((#450,#429,#428));
#1000=IFCFACEOUTERBOUND(#999,.T.);
#1001=IFCFACE((#1000));
#1002=IFCPOLYLOOP((#452,#438,#441));
#1003=IFCFACEOUTERBOUND(#1002,.T.);
#1004=IFCFACE((#1003));
#1005=IFCPOLYLOOP((#452,#441,#453));
#1006=IFCFACEOUTERBOUND(#1005,.T.);
#1007=IFCFACE((#1006));
#1008=IFCPOLYLOOP((#454,#432,#426));
#1009=IFCFACEOUTERBOUND(#1008,.T.);
#1010=IFCFACE((#1009));
#1011=IFCPOLYLOOP((#454,#426,#447));
#1012=IFCFACEOUTERBOUND(#1011,.T.);
#1013=IFCFACE((#1012));
#1014=IFCPOLYLOOP((#455,#329,#330));
#1015=IFCFACEOUTERBOUND(#1014,.T.);
#1016=IFCFACE((#1015));
#1017=IFCPOLYLOOP((#456,#437,#433));
#1018=IFCFACEOUTERBOUND(#1017,.T.);
#1019=IFCFACE((#1018));
#1020=IFCPOLYLOOP((#455,#457,#329));
#1021=IFCFACEOUTERBOUND(#1020,.T.);
#1022=IFCFACE((#1021));
#1023=IFCPOLYLOOP((#456,#444,#437));
#1024=IFCFACEOUTERBOUND(#1023,.T.);
#1025=IFCFACE((#1024));
#1026=IFCPOLYLOOP((#455,#448,#449));
#1027=IFCFACEOUTERBOUND(#1026,.T.);
#1028=IFCFACE((#1027));
#1029=IFCPOLYLOOP((#455,#449,#457));
#1030=IFCFACEOUTERBOUND(#1029,.T.);
#1031=IFCFACE((#1030));
#1032=IFCPOLYLOOP((#458,#427,#442));
#1033=IFCFACEOUTERBOUND(#1032,.T.);
#1034=IFCFACE((#1033));
#1035=IFCPOLYLOOP((#459,#432,#454));
#1036=IFCFACEOUTERBOUND(#1035,.T.);
#1037=IFCFACE((#1036));
#1038=IFCPOLYLOOP((#458,#453,#427));
#1039=IFCFACEOUTERBOUND(#1038,.T.);
#1040=IFCFACE((#1039));
#1041=IFCPOLYLOOP((#459,#434,#432));
#1042=IFCFACEOUTERBOUND(#1041,.T.);
#1043=IFCFACE((#1042));
#1044=IFCPOLYLOOP((#458,#452,#453));
#1045=IFCFACEOUTERBOUND(#1044,.T.);
#1046=IFCFACE((#1045));
#1047=IFCPOLYLOOP((#460,#337,#445));
#1048=IFCFACEOUTERBOUND(#1047,.T.);
#1049=IFCFACE((#1048));
#1050=IFCPOLYLOOP((#460,#445,#438));
#1051=IFCFACEOUTERBOUND(#1050,.T.);
#1052=IFCFACE((#1051));
#1053=IFCPOLYLOOP((#460,#438,#452));
#1054=IFCFACEOUTERBOUND(#1053,.T.);
#1055=IFCFACE((#1054));
#1056=IFCPOLYLOOP((#461,#444,#456));
#1057=IFCFACEOUTERBOUND(#1056,.T.);
#1058=IFCFACE((#1057));
#1059=IFCPOLYLOOP((#462,#452,#458));
#1060=IFCFACEOUTERBOUND(#1059,.T.);
#1061=IFCFACE((#1060));
#1062=IFCPOLYLOOP((#463,#458,#464));
#1063=IFCFACEOUTERBOUND(#1062,.T.);
#1064=IFCFACE((#1063));
#1065=IFCPOLYLOOP((#465,#434,#459));
#1066=IFCFACEOUTERBOUND(#1065,.T.);
#1067=IFCFACE((#1066));
#1068=IFCPOLYLOOP((#463,#462,#458));
#1069=IFCFACEOUTERBOUND(#1068,.T.);
#1070=IFCFACE((#1069));
#1071=IFCPOLYLOOP((#466,#460,#452));
#1072=IFCFACEOUTERBOUND(#1071,.T.);
#1073=IFCFACE((#1072));
#1074=IFCPOLYLOOP((#467,#444,#461));
#1075=IFCFACEOUTERBOUND(#1074,.T.);
#1076=IFCFACE((#1075));
#1077=IFCPOLYLOOP((#466,#452,#462));
#1078=IFCFACEOUTERBOUND(#1077,.T.);
#1079=IFCFACE((#1078));
#1080=IFCPOLYLOOP((#467,#447,#444));
#1081=IFCFACEOUTERBOUND(#1080,.T.);
#1082=IFCFACE((#1081));
#1083=IFCPOLYLOOP((#466,#462,#463));
#1084=IFCFACEOUTERBOUND(#1083,.T.);
#1085=IFCFACE((#1084));
#1086=IFCPOLYLOOP((#468,#466,#463));
#1087=IFCFACEOUTERBOUND(#1086,.T.);
#1088=IFCFACE((#1087));
#1089=IFCPOLYLOOP((#469,#454,#447));
#1090=IFCFACEOUTERBOUND(#1089,.T.);
#1091=IFCFACE((#1090));
#1092=IFCPOLYLOOP((#469,#447,#467));
#1093=IFCFACEOUTERBOUND(#1092,.T.);
#1094=IFCFACE((#1093));
#1095=IFCPOLYLOOP((#468,#338,#466));
#1096=IFCFACEOUTERBOUND(#1095,.T.);
#1097=IFCFACE((#1096));
#1098=IFCPOLYLOOP((#470,#433,#419));
#1099=IFCFACEOUTERBOUND(#1098,.T.);
#1100=IFCFACE((#1099));
#1101=IFCPOLYLOOP((#471,#463,#464));
#1102=IFCFACEOUTERBOUND(#1101,.T.);
#1103=IFCFACE((#1102));
#1104=IFCPOLYLOOP((#471,#472,#330));
#1105=IFCFACEOUTERBOUND(#1104,.T.);
#1106=IFCFACE((#1105));
#1107=IFCPOLYLOOP((#471,#330,#331));
#1108=IFCFACEOUTERBOUND(#1107,.T.);
#1109=IFCFACE((#1108));
#1110=IFCPOLYLOOP((#471,#464,#472));
#1111=IFCFACEOUTERBOUND(#1110,.T.);
#1112=IFCFACE((#1111));
#1113=IFCPOLYLOOP((#470,#419,#451));
#1114=IFCFACEOUTERBOUND(#1113,.T.);
#1115=IFCFACE((#1114));
#1116=IFCPOLYLOOP((#473,#471,#331));
#1117=IFCFACEOUTERBOUND(#1116,.T.);
#1118=IFCFACE((#1117));
#1119=IFCPOLYLOOP((#474,#451,#450));
#1120=IFCFACEOUTERBOUND(#1119,.T.);
#1121=IFCFACE((#1120));
#1122=IFCPOLYLOOP((#473,#331,#325));
#1123=IFCFACEOUTERBOUND(#1122,.T.);
#1124=IFCFACE((#1123));
#1125=IFCPOLYLOOP((#474,#470,#451));
#1126=IFCFACEOUTERBOUND(#1125,.T.);
#1127=IFCFACE((#1126));
#1128=IFCPOLYLOOP((#473,#468,#463));
#1129=IFCFACEOUTERBOUND(#1128,.T.);
#1130=IFCFACE((#1129));
#1131=IFCPOLYLOOP((#473,#463,#471));
#1132=IFCFACEOUTERBOUND(#1131,.T.);
#1133=IFCFACE((#1132));
#1134=IFCPOLYLOOP((#475,#318,#476));
#1135=IFCFACEOUTERBOUND(#1134,.T.);
#1136=IFCFACE((#1135));
#1137=IFCPOLYLOOP((#477,#478,#479));
#1138=IFCFACEOUTERBOUND(#1137,.T.);
#1139=IFCFACE((#1138));
#1140=IFCPOLYLOOP((#480,#481,#323));
#1141=IFCFACEOUTERBOUND(#1140,.T.);
#1142=IFCFACE((#1141));
#1143=IFCPOLYLOOP((#482,#477,#483));
#1144=IFCFACEOUTERBOUND(#1143,.T.);
#1145=IFCFACE((#1144));
#1146=IFCPOLYLOOP((#482,#380,#477));
#1147=IFCFACEOUTERBOUND(#1146,.T.);
#1148=IFCFACE((#1147));
#1149=IFCPOLYLOOP((#484,#459,#454));
#1150=IFCFACEOUTERBOUND(#1149,.T.);
#1151=IFCFACE((#1150));
#1152=IFCPOLYLOOP((#482,#391,#380));
#1153=IFCFACEOUTERBOUND(#1152,.T.);
#1154=IFCFACE((#1153));
#1155=IFCPOLYLOOP((#484,#454,#469));
#1156=IFCFACEOUTERBOUND(#1155,.T.);
#1157=IFCFACE((#1156));
#1158=IFCPOLYLOOP((#485,#391,#390));
#1159=IFCFACEOUTERBOUND(#1158,.T.);
#1160=IFCFACE((#1159));
#1161=IFCPOLYLOOP((#485,#390,#379));
#1162=IFCFACEOUTERBOUND(#1161,.T.);
#1163=IFCFACE((#1162));
#1164=IFCPOLYLOOP((#486,#318,#475));
#1165=IFCFACEOUTERBOUND(#1164,.T.);
#1166=IFCFACE((#1165));
#1167=IFCPOLYLOOP((#485,#380,#391));
#1168=IFCFACEOUTERBOUND(#1167,.T.);
#1169=IFCFACE((#1168));
#1170=IFCPOLYLOOP((#485,#379,#380));
#1171=IFCFACEOUTERBOUND(#1170,.T.);
#1172=IFCFACE((#1171));
#1173=IFCPOLYLOOP((#487,#316,#315));
#1174=IFCFACEOUTERBOUND(#1173,.T.);
#1175=IFCFACE((#1174));
#1176=IFCPOLYLOOP((#487,#315,#379));
#1177=IFCFACEOUTERBOUND(#1176,.T.);
#1178=IFCFACE((#1177));
#1179=IFCPOLYLOOP((#488,#467,#461));
#1180=IFCFACEOUTERBOUND(#1179,.T.);
#1181=IFCFACE((#1180));
#1182=IFCPOLYLOOP((#489,#487,#379));
#1183=IFCFACEOUTERBOUND(#1182,.T.);
#1184=IFCFACE((#1183));
#1185=IFCPOLYLOOP((#489,#395,#394));
#1186=IFCFACEOUTERBOUND(#1185,.T.);
#1187=IFCFACE((#1186));
#1188=IFCPOLYLOOP((#489,#394,#316));
#1189=IFCFACEOUTERBOUND(#1188,.T.);
#1190=IFCFACE((#1189));
#1191=IFCPOLYLOOP((#489,#316,#487));
#1192=IFCFACEOUTERBOUND(#1191,.T.);
#1193=IFCFACE((#1192));
#1194=IFCPOLYLOOP((#490,#491,#492));
#1195=IFCFACEOUTERBOUND(#1194,.T.);
#1196=IFCFACE((#1195));
#1197=IFCPOLYLOOP((#490,#321,#402));
#1198=IFCFACEOUTERBOUND(#1197,.T.);
#1199=IFCFACE((#1198));
#1200=IFCPOLYLOOP((#490,#493,#491));
#1201=IFCFACEOUTERBOUND(#1200,.T.);
#1202=IFCFACE((#1201));
#1203=IFCPOLYLOOP((#490,#402,#493));
#1204=IFCFACEOUTERBOUND(#1203,.T.);
#1205=IFCFACE((#1204));
#1206=IFCPOLYLOOP((#494,#489,#379));
#1207=IFCFACEOUTERBOUND(#1206,.T.);
#1208=IFCFACE((#1207));
#1209=IFCPOLYLOOP((#495,#459,#484));
#1210=IFCFACEOUTERBOUND(#1209,.T.);
#1211=IFCFACE((#1210));
#1212=IFCPOLYLOOP((#494,#379,#390));
#1213=IFCFACEOUTERBOUND(#1212,.T.);
#1214=IFCFACE((#1213));
#1215=IFCPOLYLOOP((#495,#465,#459));
#1216=IFCFACEOUTERBOUND(#1215,.T.);
#1217=IFCFACE((#1216));
#1218=IFCPOLYLOOP((#494,#395,#489));
#1219=IFCFACEOUTERBOUND(#1218,.T.);
#1220=IFCFACE((#1219));
#1221=IFCPOLYLOOP((#494,#390,#393));
#1222=IFCFACEOUTERBOUND(#1221,.T.);
#1223=IFCFACE((#1222));
#1224=IFCPOLYLOOP((#494,#393,#395));
#1225=IFCFACEOUTERBOUND(#1224,.T.);
#1226=IFCFACE((#1225));
#1227=IFCPOLYLOOP((#496,#484,#469));
#1228=IFCFACEOUTERBOUND(#1227,.T.);
#1229=IFCFACE((#1228));
#1230=IFCPOLYLOOP((#497,#407,#406));
#1231=IFCFACEOUTERBOUND(#1230,.T.);
#1232=IFCFACE((#1231));
#1233=IFCPOLYLOOP((#497,#493,#402));
#1234=IFCFACEOUTERBOUND(#1233,.T.);
#1235=IFCFACE((#1234));
#1236=IFCPOLYLOOP((#496,#467,#488));
#1237=IFCFACEOUTERBOUND(#1236,.T.);
#1238=IFCFACE((#1237));
#1239=IFCPOLYLOOP((#497,#406,#493));
#1240=IFCFACEOUTERBOUND(#1239,.T.);
#1241=IFCFACE((#1240));
#1242=IFCPOLYLOOP((#498,#398,#413));
#1243=IFCFACEOUTERBOUND(#1242,.T.);
#1244=IFCFACE((#1243));
#1245=IFCPOLYLOOP((#498,#413,#400));
#1246=IFCFACEOUTERBOUND(#1245,.T.);
#1247=IFCFACE((#1246));
#1248=IFCPOLYLOOP((#498,#394,#398));
#1249=IFCFACEOUTERBOUND(#1248,.T.);
#1250=IFCFACE((#1249));
#1251=IFCPOLYLOOP((#496,#469,#467));
#1252=IFCFACEOUTERBOUND(#1251,.T.);
#1253=IFCFACE((#1252));
#1254=IFCPOLYLOOP((#498,#400,#394));
#1255=IFCFACEOUTERBOUND(#1254,.T.);
#1256=IFCFACE((#1255));
#1257=IFCPOLYLOOP((#499,#317,#316));
#1258=IFCFACEOUTERBOUND(#1257,.T.);
#1259=IFCFACE((#1258));
#1260=IFCPOLYLOOP((#499,#436,#435));
#1261=IFCFACEOUTERBOUND(#1260,.T.);
#1262=IFCFACE((#1261));
#1263=IFCPOLYLOOP((#499,#400,#436));
#1264=IFCFACEOUTERBOUND(#1263,.T.);
#1265=IFCFACE((#1264));
#1266=IFCPOLYLOOP((#499,#316,#400));
#1267=IFCFACEOUTERBOUND(#1266,.T.);
#1268=IFCFACE((#1267));
#1269=IFCPOLYLOOP((#499,#435,#317));
#1270=IFCFACEOUTERBOUND(#1269,.T.);
#1271=IFCFACE((#1270));
#1272=IFCPOLYLOOP((#500,#419,#407));
#1273=IFCFACEOUTERBOUND(#1272,.T.);
#1274=IFCFACE((#1273));
#1275=IFCPOLYLOOP((#501,#433,#470));
#1276=IFCFACEOUTERBOUND(#1275,.T.);
#1277=IFCFACE((#1276));
#1278=IFCPOLYLOOP((#500,#401,#419));
#1279=IFCFACEOUTERBOUND(#1278,.T.);
#1280=IFCFACE((#1279));
#1281=IFCPOLYLOOP((#500,#497,#402));
#1282=IFCFACEOUTERBOUND(#1281,.T.);
#1283=IFCFACE((#1282));
#1284=IFCPOLYLOOP((#501,#456,#433));
#1285=IFCFACEOUTERBOUND(#1284,.T.);
#1286=IFCFACE((#1285));
#1287=IFCPOLYLOOP((#500,#407,#497));
#1288=IFCFACEOUTERBOUND(#1287,.T.);
#1289=IFCFACE((#1288));
#1290=IFCPOLYLOOP((#500,#402,#401));
#1291=IFCFACEOUTERBOUND(#1290,.T.);
#1292=IFCFACE((#1291));
#1293=IFCPOLYLOOP((#502,#321,#322));
#1294=IFCFACEOUTERBOUND(#1293,.T.);
#1295=IFCFACE((#1294));
#1296=IFCPOLYLOOP((#502,#401,#321));
#1297=IFCFACEOUTERBOUND(#1296,.T.);
#1298=IFCFACE((#1297));
#1299=IFCPOLYLOOP((#502,#322,#429));
#1300=IFCFACEOUTERBOUND(#1299,.T.);
#1301=IFCFACE((#1300));
#1302=IFCPOLYLOOP((#503,#400,#413));
#1303=IFCFACEOUTERBOUND(#1302,.T.);
#1304=IFCFACE((#1303));
#1305=IFCPOLYLOOP((#504,#501,#470));
#1306=IFCFACEOUTERBOUND(#1305,.T.);
#1307=IFCFACE((#1306));
#1308=IFCPOLYLOOP((#503,#436,#400));
#1309=IFCFACEOUTERBOUND(#1308,.T.);
#1310=IFCFACE((#1309));
#1311=IFCPOLYLOOP((#503,#413,#432));
#1312=IFCFACEOUTERBOUND(#1311,.T.);
#1313=IFCFACE((#1312));
#1314=IFCPOLYLOOP((#503,#432,#436));
#1315=IFCFACEOUTERBOUND(#1314,.T.);
#1316=IFCFACE((#1315));
#1317=IFCPOLYLOOP((#504,#470,#474));
#1318=IFCFACEOUTERBOUND(#1317,.T.);
#1319=IFCFACE((#1318));
#1320=IFCPOLYLOOP((#505,#419,#401));
#1321=IFCFACEOUTERBOUND(#1320,.T.);
#1322=IFCFACE((#1321));
#1323=IFCPOLYLOOP((#505,#429,#451));
#1324=IFCFACEOUTERBOUND(#1323,.T.);
#1325=IFCFACE((#1324));
#1326=IFCPOLYLOOP((#505,#502,#429));
#1327=IFCFACEOUTERBOUND(#1326,.T.);
#1328=IFCFACE((#1327));
#1329=IFCPOLYLOOP((#505,#401,#502));
#1330=IFCFACEOUTERBOUND(#1329,.T.);
#1331=IFCFACE((#1330));
#1332=IFCPOLYLOOP((#506,#456,#501));
#1333=IFCFACEOUTERBOUND(#1332,.T.);
#1334=IFCFACE((#1333));
#1335=IFCPOLYLOOP((#505,#451,#419));
#1336=IFCFACEOUTERBOUND(#1335,.T.);
#1337=IFCFACE((#1336));
#1338=IFCPOLYLOOP((#507,#318,#317));
#1339=IFCFACEOUTERBOUND(#1338,.T.);
#1340=IFCFACE((#1339));
#1341=IFCPOLYLOOP((#507,#476,#318));
#1342=IFCFACEOUTERBOUND(#1341,.T.);
#1343=IFCFACE((#1342));
#1344=IFCPOLYLOOP((#506,#461,#456));
#1345=IFCFACEOUTERBOUND(#1344,.T.);
#1346=IFCFACE((#1345));
#1347=IFCPOLYLOOP((#507,#317,#435));
#1348=IFCFACEOUTERBOUND(#1347,.T.);
#1349=IFCFACE((#1348));
#1350=IFCPOLYLOOP((#508,#322,#323));
#1351=IFCFACEOUTERBOUND(#1350,.T.);
#1352=IFCFACE((#1351));
#1353=IFCPOLYLOOP((#508,#428,#322));
#1354=IFCFACEOUTERBOUND(#1353,.T.);
#1355=IFCFACE((#1354));
#1356=IFCPOLYLOOP((#509,#501,#504));
#1357=IFCFACEOUTERBOUND(#1356,.T.);
#1358=IFCFACE((#1357));
#1359=IFCPOLYLOOP((#508,#323,#481));
#1360=IFCFACEOUTERBOUND(#1359,.T.);
#1361=IFCFACE((#1360));
#1362=IFCPOLYLOOP((#510,#435,#434));
#1363=IFCFACEOUTERBOUND(#1362,.T.);
#1364=IFCFACE((#1363));
#1365=IFCPOLYLOOP((#509,#506,#501));
#1366=IFCFACEOUTERBOUND(#1365,.T.);
#1367=IFCFACE((#1366));
#1368=IFCPOLYLOOP((#510,#465,#476));
#1369=IFCFACEOUTERBOUND(#1368,.T.);
#1370=IFCFACE((#1369));
#1371=IFCPOLYLOOP((#510,#476,#507));
#1372=IFCFACEOUTERBOUND(#1371,.T.);
#1373=IFCFACE((#1372));
#1374=IFCPOLYLOOP((#510,#507,#435));
#1375=IFCFACEOUTERBOUND(#1374,.T.);
#1376=IFCFACE((#1375));
#1377=IFCPOLYLOOP((#510,#434,#465));
#1378=IFCFACEOUTERBOUND(#1377,.T.);
#1379=IFCFACE((#1378));
#1380=IFCPOLYLOOP((#511,#450,#428));
#1381=IFCFACEOUTERBOUND(#1380,.T.);
#1382=IFCFACE((#1381));
#1383=IFCPOLYLOOP((#511,#428,#508));
#1384=IFCFACEOUTERBOUND(#1383,.T.);
#1385=IFCFACE((#1384));
#1386=IFCPOLYLOOP((#511,#508,#481));
#1387=IFCFACEOUTERBOUND(#1386,.T.);
#1388=IFCFACE((#1387));
#1389=IFCPOLYLOOP((#511,#481,#512));
#1390=IFCFACEOUTERBOUND(#1389,.T.);
#1391=IFCFACE((#1390));
#1392=IFCPOLYLOOP((#513,#465,#514));
#1393=IFCFACEOUTERBOUND(#1392,.T.);
#1394=IFCFACE((#1393));
#1395=IFCPOLYLOOP((#513,#476,#465));
#1396=IFCFACEOUTERBOUND(#1395,.T.);
#1397=IFCFACE((#1396));
#1398=IFCPOLYLOOP((#513,#475,#476));
#1399=IFCFACEOUTERBOUND(#1398,.T.);
#1400=IFCFACE((#1399));
#1401=IFCPOLYLOOP((#513,#514,#475));
#1402=IFCFACEOUTERBOUND(#1401,.T.);
#1403=IFCFACE((#1402));
#1404=IFCPOLYLOOP((#515,#323,#324));
#1405=IFCFACEOUTERBOUND(#1404,.T.);
#1406=IFCFACE((#1405));
#1407=IFCPOLYLOOP((#515,#324,#516));
#1408=IFCFACEOUTERBOUND(#1407,.T.);
#1409=IFCFACE((#1408));
#1410=IFCPOLYLOOP((#515,#516,#480));
#1411=IFCFACEOUTERBOUND(#1410,.T.);
#1412=IFCFACE((#1411));
#1413=IFCPOLYLOOP((#515,#480,#323));
#1414=IFCFACEOUTERBOUND(#1413,.T.);
#1415=IFCFACE((#1414));
#1416=IFCPOLYLOOP((#517,#512,#518));
#1417=IFCFACEOUTERBOUND(#1416,.T.);
#1418=IFCFACE((#1417));
#1419=IFCPOLYLOOP((#517,#450,#511));
#1420=IFCFACEOUTERBOUND(#1419,.T.);
#1421=IFCFACE((#1420));
#1422=IFCPOLYLOOP((#517,#511,#512));
#1423=IFCFACEOUTERBOUND(#1422,.T.);
#1424=IFCFACE((#1423));
#1425=IFCPOLYLOOP((#517,#474,#450));
#1426=IFCFACEOUTERBOUND(#1425,.T.);
#1427=IFCFACE((#1426));
#1428=IFCPOLYLOOP((#519,#484,#496));
#1429=IFCFACEOUTERBOUND(#1428,.T.);
#1430=IFCFACE((#1429));
#1431=IFCPOLYLOOP((#520,#319,#318));
#1432=IFCFACEOUTERBOUND(#1431,.T.);
#1433=IFCFACE((#1432));
#1434=IFCPOLYLOOP((#520,#521,#319));
#1435=IFCFACEOUTERBOUND(#1434,.T.);
#1436=IFCFACE((#1435));
#1437=IFCPOLYLOOP((#520,#318,#486));
#1438=IFCFACEOUTERBOUND(#1437,.T.);
#1439=IFCFACE((#1438));
#1440=IFCPOLYLOOP((#522,#523,#480));
#1441=IFCFACEOUTERBOUND(#1440,.T.);
#1442=IFCFACE((#1441));
#1443=IFCPOLYLOOP((#522,#480,#516));
#1444=IFCFACEOUTERBOUND(#1443,.T.);
#1445=IFCFACE((#1444));
#1446=IFCPOLYLOOP((#522,#524,#523));
#1447=IFCFACEOUTERBOUND(#1446,.T.);
#1448=IFCFACE((#1447));
#1449=IFCPOLYLOOP((#523,#512,#481));
#1450=IFCFACEOUTERBOUND(#1449,.T.);
#1451=IFCFACE((#1450));
#1452=IFCPOLYLOOP((#522,#516,#524));
#1453=IFCFACEOUTERBOUND(#1452,.T.);
#1454=IFCFACE((#1453));
#1455=IFCPOLYLOOP((#525,#509,#504));
#1456=IFCFACEOUTERBOUND(#1455,.T.);
#1457=IFCFACE((#1456));
#1458=IFCPOLYLOOP((#523,#481,#480));
#1459=IFCFACEOUTERBOUND(#1458,.T.);
#1460=IFCFACE((#1459));
#1461=IFCPOLYLOOP((#525,#504,#474));
#1462=IFCFACEOUTERBOUND(#1461,.T.);
#1463=IFCFACE((#1462));
#1464=IFCPOLYLOOP((#526,#486,#475));
#1465=IFCFACEOUTERBOUND(#1464,.T.);
#1466=IFCFACE((#1465));
#1467=IFCPOLYLOOP((#525,#474,#517));
#1468=IFCFACEOUTERBOUND(#1467,.T.);
#1469=IFCFACE((#1468));
#1470=IFCPOLYLOOP((#525,#518,#527));
#1471=IFCFACEOUTERBOUND(#1470,.T.);
#1472=IFCFACE((#1471));
#1473=IFCPOLYLOOP((#525,#517,#518));
#1474=IFCFACEOUTERBOUND(#1473,.T.);
#1475=IFCFACE((#1474));
#1476=IFCPOLYLOOP((#525,#528,#509));
#1477=IFCFACEOUTERBOUND(#1476,.T.);
#1478=IFCFACE((#1477));
#1479=IFCPOLYLOOP((#525,#527,#528));
#1480=IFCFACEOUTERBOUND(#1479,.T.);
#1481=IFCFACE((#1480));
#1482=IFCPOLYLOOP((#526,#475,#514));
#1483=IFCFACEOUTERBOUND(#1482,.T.);
#1484=IFCFACE((#1483));
#1485=IFCPOLYLOOP((#529,#486,#526));
#1486=IFCFACEOUTERBOUND(#1485,.T.);
#1487=IFCFACE((#1486));
#1488=IFCPOLYLOOP((#529,#526,#530));
#1489=IFCFACEOUTERBOUND(#1488,.T.);
#1490=IFCFACE((#1489));
#1491=IFCPOLYLOOP((#529,#530,#521));
#1492=IFCFACEOUTERBOUND(#1491,.T.);
#1493=IFCFACE((#1492));
#1494=IFCPOLYLOOP((#529,#521,#520));
#1495=IFCFACEOUTERBOUND(#1494,.T.);
#1496=IFCFACE((#1495));
#1497=IFCPOLYLOOP((#529,#520,#486));
#1498=IFCFACEOUTERBOUND(#1497,.T.);
#1499=IFCFACE((#1498));
#1500=IFCPOLYLOOP((#531,#319,#521));
#1501=IFCFACEOUTERBOUND(#1500,.T.);
#1502=IFCFACE((#1501));
#1503=IFCPOLYLOOP((#532,#484,#519));
#1504=IFCFACEOUTERBOUND(#1503,.T.);
#1505=IFCFACE((#1504));
#1506=IFCPOLYLOOP((#531,#521,#533));
#1507=IFCFACEOUTERBOUND(#1506,.T.);
#1508=IFCFACE((#1507));
#1509=IFCPOLYLOOP((#532,#495,#484));
#1510=IFCFACEOUTERBOUND(#1509,.T.);
#1511=IFCFACE((#1510));
#1512=IFCPOLYLOOP((#531,#533,#534));
#1513=IFCFACEOUTERBOUND(#1512,.T.);
#1514=IFCFACE((#1513));
#1515=IFCPOLYLOOP((#535,#531,#534));
#1516=IFCFACEOUTERBOUND(#1515,.T.);
#1517=IFCFACE((#1516));
#1518=IFCPOLYLOOP((#536,#496,#488));
#1519=IFCFACEOUTERBOUND(#1518,.T.);
#1520=IFCFACE((#1519));
#1521=IFCPOLYLOOP((#535,#314,#319));
#1522=IFCFACEOUTERBOUND(#1521,.T.);
#1523=IFCFACE((#1522));
#1524=IFCPOLYLOOP((#535,#319,#531));
#1525=IFCFACEOUTERBOUND(#1524,.T.);
#1526=IFCFACE((#1525));
#1527=IFCPOLYLOOP((#536,#519,#496));
#1528=IFCFACEOUTERBOUND(#1527,.T.);
#1529=IFCFACE((#1528));
#1530=IFCPOLYLOOP((#537,#538,#539));
#1531=IFCFACEOUTERBOUND(#1530,.T.);
#1532=IFCFACE((#1531));
#1533=IFCPOLYLOOP((#540,#537,#539));
#1534=IFCFACEOUTERBOUND(#1533,.T.);
#1535=IFCFACE((#1534));
#1536=IFCPOLYLOOP((#541,#314,#535));
#1537=IFCFACEOUTERBOUND(#1536,.T.);
#1538=IFCFACE((#1537));
#1539=IFCPOLYLOOP((#541,#332,#314));
#1540=IFCFACEOUTERBOUND(#1539,.T.);
#1541=IFCFACE((#1540));
#1542=IFCPOLYLOOP((#541,#542,#332));
#1543=IFCFACEOUTERBOUND(#1542,.T.);
#1544=IFCFACE((#1543));
#1545=IFCPOLYLOOP((#541,#535,#534));
#1546=IFCFACEOUTERBOUND(#1545,.T.);
#1547=IFCFACE((#1546));
#1548=IFCPOLYLOOP((#541,#534,#542));
#1549=IFCFACEOUTERBOUND(#1548,.T.);
#1550=IFCFACE((#1549));
#1551=IFCPOLYLOOP((#543,#538,#537));
#1552=IFCFACEOUTERBOUND(#1551,.T.);
#1553=IFCFACE((#1552));
#1554=IFCPOLYLOOP((#543,#537,#540));
#1555=IFCFACEOUTERBOUND(#1554,.T.);
#1556=IFCFACE((#1555));
#1557=IFCPOLYLOOP((#543,#544,#538));
#1558=IFCFACEOUTERBOUND(#1557,.T.);
#1559=IFCFACE((#1558));
#1560=IFCPOLYLOOP((#545,#516,#324));
#1561=IFCFACEOUTERBOUND(#1560,.T.);
#1562=IFCFACE((#1561));
#1563=IFCPOLYLOOP((#546,#543,#540));
#1564=IFCFACEOUTERBOUND(#1563,.T.);
#1565=IFCFACE((#1564));
#1566=IFCPOLYLOOP((#547,#346,#343));
#1567=IFCFACEOUTERBOUND(#1566,.T.);
#1568=IFCFACE((#1567));
#1569=IFCPOLYLOOP((#548,#519,#536));
#1570=IFCFACEOUTERBOUND(#1569,.T.);
#1571=IFCFACE((#1570));
#1572=IFCPOLYLOOP((#547,#333,#332));
#1573=IFCFACEOUTERBOUND(#1572,.T.);
#1574=IFCFACE((#1573));
#1575=IFCPOLYLOOP((#547,#332,#346));
#1576=IFCFACEOUTERBOUND(#1575,.T.);
#1577=IFCFACE((#1576));
#1578=IFCPOLYLOOP((#547,#343,#333));
#1579=IFCFACEOUTERBOUND(#1578,.T.);
#1580=IFCFACE((#1579));
#1581=IFCPOLYLOOP((#549,#465,#495));
#1582=IFCFACEOUTERBOUND(#1581,.T.);
#1583=IFCFACE((#1582));
#1584=IFCPOLYLOOP((#549,#514,#465));
#1585=IFCFACEOUTERBOUND(#1584,.T.);
#1586=IFCFACE((#1585));
#1587=IFCPOLYLOOP((#550,#326,#368));
#1588=IFCFACEOUTERBOUND(#1587,.T.);
#1589=IFCFACE((#1588));
#1590=IFCPOLYLOOP((#551,#365,#350));
#1591=IFCFACEOUTERBOUND(#1590,.T.);
#1592=IFCFACE((#1591));
#1593=IFCPOLYLOOP((#552,#488,#461));
#1594=IFCFACEOUTERBOUND(#1593,.T.);
#1595=IFCFACE((#1594));
#1596=IFCPOLYLOOP((#552,#506,#509));
#1597=IFCFACEOUTERBOUND(#1596,.T.);
#1598=IFCFACE((#1597));
#1599=IFCPOLYLOOP((#553,#392,#551));
#1600=IFCFACEOUTERBOUND(#1599,.T.);
#1601=IFCFACE((#1600));
#1602=IFCPOLYLOOP((#553,#551,#350));
#1603=IFCFACEOUTERBOUND(#1602,.T.);
#1604=IFCFACE((#1603));
#1605=IFCPOLYLOOP((#552,#461,#506));
#1606=IFCFACEOUTERBOUND(#1605,.T.);
#1607=IFCFACE((#1606));
#1608=IFCPOLYLOOP((#554,#552,#509));
#1609=IFCFACEOUTERBOUND(#1608,.T.);
#1610=IFCFACE((#1609));
#1611=IFCPOLYLOOP((#555,#371,#370));
#1612=IFCFACEOUTERBOUND(#1611,.T.);
#1613=IFCFACE((#1612));
#1614=IFCPOLYLOOP((#555,#553,#350));
#1615=IFCFACEOUTERBOUND(#1614,.T.);
#1616=IFCFACE((#1615));
#1617=IFCPOLYLOOP((#555,#370,#378));
#1618=IFCFACEOUTERBOUND(#1617,.T.);
#1619=IFCFACE((#1618));
#1620=IFCPOLYLOOP((#555,#392,#553));
#1621=IFCFACEOUTERBOUND(#1620,.T.);
#1622=IFCFACE((#1621));
#1623=IFCPOLYLOOP((#555,#378,#392));
#1624=IFCFACEOUTERBOUND(#1623,.T.);
#1625=IFCFACE((#1624));
#1626=IFCPOLYLOOP((#555,#350,#371));
#1627=IFCFACEOUTERBOUND(#1626,.T.);
#1628=IFCFACE((#1627));
#1629=IFCPOLYLOOP((#556,#334,#333));
#1630=IFCFACEOUTERBOUND(#1629,.T.);
#1631=IFCFACE((#1630));
#1632=IFCPOLYLOOP((#556,#333,#369));
#1633=IFCFACEOUTERBOUND(#1632,.T.);
#1634=IFCFACE((#1633));
#1635=IFCPOLYLOOP((#556,#383,#334));
#1636=IFCFACEOUTERBOUND(#1635,.T.);
#1637=IFCFACE((#1636));
#1638=IFCPOLYLOOP((#556,#369,#383));
#1639=IFCFACEOUTERBOUND(#1638,.T.);
#1640=IFCFACE((#1639));
#1641=IFCPOLYLOOP((#557,#399,#421));
#1642=IFCFACEOUTERBOUND(#1641,.T.);
#1643=IFCFACE((#1642));
#1644=IFCPOLYLOOP((#558,#518,#512));
#1645=IFCFACEOUTERBOUND(#1644,.T.);
#1646=IFCFACE((#1645));
#1647=IFCPOLYLOOP((#557,#374,#399));
#1648=IFCFACEOUTERBOUND(#1647,.T.);
#1649=IFCFACE((#1648));
#1650=IFCPOLYLOOP((#557,#381,#374));
#1651=IFCFACEOUTERBOUND(#1650,.T.);
#1652=IFCFACE((#1651));
#1653=IFCPOLYLOOP((#558,#512,#523));
#1654=IFCFACEOUTERBOUND(#1653,.T.);
#1655=IFCFACE((#1654));
#1656=IFCPOLYLOOP((#557,#396,#381));
#1657=IFCFACEOUTERBOUND(#1656,.T.);
#1658=IFCFACE((#1657));
#1659=IFCPOLYLOOP((#559,#404,#414));
#1660=IFCFACEOUTERBOUND(#1659,.T.);
#1661=IFCFACE((#1660));
#1662=IFCPOLYLOOP((#559,#551,#392));
#1663=IFCFACEOUTERBOUND(#1662,.T.);
#1664=IFCFACE((#1663));
#1665=IFCPOLYLOOP((#559,#365,#551));
#1666=IFCFACEOUTERBOUND(#1665,.T.);
#1667=IFCFACE((#1666));
#1668=IFCPOLYLOOP((#559,#375,#365));
#1669=IFCFACEOUTERBOUND(#1668,.T.);
#1670=IFCFACE((#1669));
#1671=IFCPOLYLOOP((#560,#526,#514));
#1672=IFCFACEOUTERBOUND(#1671,.T.);
#1673=IFCFACE((#1672));
#1674=IFCPOLYLOOP((#559,#383,#375));
#1675=IFCFACEOUTERBOUND(#1674,.T.);
#1676=IFCFACE((#1675));
#1677=IFCPOLYLOOP((#559,#392,#404));
#1678=IFCFACEOUTERBOUND(#1677,.T.);
#1679=IFCFACE((#1678));
#1680=IFCPOLYLOOP((#560,#514,#549));
#1681=IFCFACEOUTERBOUND(#1680,.T.);
#1682=IFCFACE((#1681));
#1683=IFCPOLYLOOP((#561,#422,#399));
#1684=IFCFACEOUTERBOUND(#1683,.T.);
#1685=IFCFACE((#1684));
#1686=IFCPOLYLOOP((#561,#376,#387));
#1687=IFCFACEOUTERBOUND(#1686,.T.);
#1688=IFCFACE((#1687));
#1689=IFCPOLYLOOP((#561,#399,#376));
#1690=IFCFACEOUTERBOUND(#1689,.T.);
#1691=IFCFACE((#1690));
#1692=IFCPOLYLOOP((#562,#396,#557));
#1693=IFCFACEOUTERBOUND(#1692,.T.);
#1694=IFCFACE((#1693));
#1695=IFCPOLYLOOP((#563,#532,#519));
#1696=IFCFACEOUTERBOUND(#1695,.T.);
#1697=IFCFACE((#1696));
#1698=IFCPOLYLOOP((#562,#557,#421));
#1699=IFCFACEOUTERBOUND(#1698,.T.);
#1700=IFCFACE((#1699));
#1701=IFCPOLYLOOP((#563,#519,#548));
#1702=IFCFACEOUTERBOUND(#1701,.T.);
#1703=IFCFACE((#1702));
#1704=IFCPOLYLOOP((#564,#335,#334));
#1705=IFCFACEOUTERBOUND(#1704,.T.);
#1706=IFCFACE((#1705));
#1707=IFCPOLYLOOP((#564,#383,#559));
#1708=IFCFACEOUTERBOUND(#1707,.T.);
#1709=IFCFACE((#1708));
#1710=IFCPOLYLOOP((#564,#414,#335));
#1711=IFCFACEOUTERBOUND(#1710,.T.);
#1712=IFCFACE((#1711));
#1713=IFCPOLYLOOP((#564,#334,#383));
#1714=IFCFACEOUTERBOUND(#1713,.T.);
#1715=IFCFACE((#1714));
#1716=IFCPOLYLOOP((#564,#559,#414));
#1717=IFCFACEOUTERBOUND(#1716,.T.);
#1718=IFCFACE((#1717));
#1719=IFCPOLYLOOP((#565,#387,#410));
#1720=IFCFACEOUTERBOUND(#1719,.T.);
#1721=IFCFACE((#1720));
#1722=IFCPOLYLOOP((#530,#526,#560));
#1723=IFCFACEOUTERBOUND(#1722,.T.);
#1724=IFCFACE((#1723));
#1725=IFCPOLYLOOP((#566,#396,#562));
#1726=IFCFACEOUTERBOUND(#1725,.T.);
#1727=IFCFACE((#1726));
#1728=IFCPOLYLOOP((#566,#440,#439));
#1729=IFCFACEOUTERBOUND(#1728,.T.);
#1730=IFCFACE((#1729));
#1731=IFCPOLYLOOP((#566,#562,#421));
#1732=IFCFACEOUTERBOUND(#1731,.T.);
#1733=IFCFACE((#1732));
#1734=IFCPOLYLOOP((#567,#561,#387));
#1735=IFCFACEOUTERBOUND(#1734,.T.);
#1736=IFCFACE((#1735));
#1737=IFCPOLYLOOP((#567,#387,#565));
#1738=IFCFACEOUTERBOUND(#1737,.T.);
#1739=IFCFACE((#1738));
#1740=IFCPOLYLOOP((#568,#422,#561));
#1741=IFCFACEOUTERBOUND(#1740,.T.);
#1742=IFCFACE((#1741));
#1743=IFCPOLYLOOP((#568,#561,#567));
#1744=IFCFACEOUTERBOUND(#1743,.T.);
#1745=IFCFACE((#1744));
#1746=IFCPOLYLOOP((#568,#567,#430));
#1747=IFCFACEOUTERBOUND(#1746,.T.);
#1748=IFCFACE((#1747));
#1749=IFCPOLYLOOP((#568,#430,#422));
#1750=IFCFACEOUTERBOUND(#1749,.T.);
#1751=IFCFACE((#1750));
#1752=IFCPOLYLOOP((#569,#421,#420));
#1753=IFCFACEOUTERBOUND(#1752,.T.);
#1754=IFCFACE((#1753));
#1755=IFCPOLYLOOP((#528,#554,#509));
#1756=IFCFACEOUTERBOUND(#1755,.T.);
#1757=IFCFACE((#1756));
#1758=IFCPOLYLOOP((#569,#441,#440));
#1759=IFCFACEOUTERBOUND(#1758,.T.);
#1760=IFCFACE((#1759));
#1761=IFCPOLYLOOP((#569,#566,#421));
#1762=IFCFACEOUTERBOUND(#1761,.T.);
#1763=IFCFACE((#1762));
#1764=IFCPOLYLOOP((#569,#440,#566));
#1765=IFCFACEOUTERBOUND(#1764,.T.);
#1766=IFCFACE((#1765));
#1767=IFCPOLYLOOP((#570,#533,#521));
#1768=IFCFACEOUTERBOUND(#1767,.T.);
#1769=IFCFACE((#1768));
#1770=IFCPOLYLOOP((#571,#565,#410));
#1771=IFCFACEOUTERBOUND(#1770,.T.);
#1772=IFCFACE((#1771));
#1773=IFCPOLYLOOP((#570,#534,#533));
#1774=IFCFACEOUTERBOUND(#1773,.T.);
#1775=IFCFACE((#1774));
#1776=IFCPOLYLOOP((#571,#410,#417));
#1777=IFCFACEOUTERBOUND(#1776,.T.);
#1778=IFCFACE((#1777));
#1779=IFCPOLYLOOP((#570,#521,#530));
#1780=IFCFACEOUTERBOUND(#1779,.T.);
#1781=IFCFACE((#1780));
#1782=IFCPOLYLOOP((#571,#443,#430));
#1783=IFCFACEOUTERBOUND(#1782,.T.);
#1784=IFCFACE((#1783));
#1785=IFCPOLYLOOP((#571,#567,#565));
#1786=IFCFACEOUTERBOUND(#1785,.T.);
#1787=IFCFACE((#1786));
#1788=IFCPOLYLOOP((#572,#516,#545));
#1789=IFCFACEOUTERBOUND(#1788,.T.);
#1790=IFCFACE((#1789));
#1791=IFCPOLYLOOP((#572,#524,#516));
#1792=IFCFACEOUTERBOUND(#1791,.T.);
#1793=IFCFACE((#1792));
#1794=IFCPOLYLOOP((#571,#430,#567));
#1795=IFCFACEOUTERBOUND(#1794,.T.);
#1796=IFCFACE((#1795));
#1797=IFCPOLYLOOP((#573,#336,#335));
#1798=IFCFACEOUTERBOUND(#1797,.T.);
#1799=IFCFACE((#1798));
#1800=IFCPOLYLOOP((#574,#549,#495));
#1801=IFCFACEOUTERBOUND(#1800,.T.);
#1802=IFCFACE((#1801));
#1803=IFCPOLYLOOP((#574,#532,#563));
#1804=IFCFACEOUTERBOUND(#1803,.T.);
#1805=IFCFACE((#1804));
#1806=IFCPOLYLOOP((#575,#569,#420));
#1807=IFCFACEOUTERBOUND(#1806,.T.);
#1808=IFCFACE((#1807));
#1809=IFCPOLYLOOP((#574,#495,#532));
#1810=IFCFACEOUTERBOUND(#1809,.T.);
#1811=IFCFACE((#1810));
#1812=IFCPOLYLOOP((#575,#420,#427));
#1813=IFCFACEOUTERBOUND(#1812,.T.);
#1814=IFCFACE((#1813));
#1815=IFCPOLYLOOP((#575,#441,#569));
#1816=IFCFACEOUTERBOUND(#1815,.T.);
#1817=IFCFACE((#1816));
#1818=IFCPOLYLOOP((#575,#453,#441));
#1819=IFCFACEOUTERBOUND(#1818,.T.);
#1820=IFCFACE((#1819));
#1821=IFCPOLYLOOP((#575,#427,#453));
#1822=IFCFACEOUTERBOUND(#1821,.T.);
#1823=IFCFACE((#1822));
#1824=IFCPOLYLOOP((#576,#328,#329));
#1825=IFCFACEOUTERBOUND(#1824,.T.);
#1826=IFCFACE((#1825));
#1827=IFCPOLYLOOP((#576,#431,#328));
#1828=IFCFACEOUTERBOUND(#1827,.T.);
#1829=IFCFACE((#1828));
#1830=IFCPOLYLOOP((#577,#536,#488));
#1831=IFCFACEOUTERBOUND(#1830,.T.);
#1832=IFCFACE((#1831));
#1833=IFCPOLYLOOP((#577,#552,#554));
#1834=IFCFACEOUTERBOUND(#1833,.T.);
#1835=IFCFACE((#1834));
#1836=IFCPOLYLOOP((#576,#329,#457));
#1837=IFCFACEOUTERBOUND(#1836,.T.);
#1838=IFCFACE((#1837));
#1839=IFCPOLYLOOP((#577,#488,#552));
#1840=IFCFACEOUTERBOUND(#1839,.T.);
#1841=IFCFACE((#1840));
#1842=IFCPOLYLOOP((#578,#337,#336));
#1843=IFCFACEOUTERBOUND(#1842,.T.);
#1844=IFCFACE((#1843));
#1845=IFCPOLYLOOP((#578,#445,#337));
#1846=IFCFACEOUTERBOUND(#1845,.T.);
#1847=IFCFACE((#1846));
#1848=IFCPOLYLOOP((#578,#336,#446));
#1849=IFCFACEOUTERBOUND(#1848,.T.);
#1850=IFCFACE((#1849));
#1851=IFCPOLYLOOP((#578,#446,#445));
#1852=IFCFACEOUTERBOUND(#1851,.T.);
#1853=IFCFACE((#1852));
#1854=IFCPOLYLOOP((#579,#330,#472));
#1855=IFCFACEOUTERBOUND(#1854,.T.);
#1856=IFCFACE((#1855));
#1857=IFCPOLYLOOP((#579,#458,#442));
#1858=IFCFACEOUTERBOUND(#1857,.T.);
#1859=IFCFACE((#1858));
#1860=IFCPOLYLOOP((#579,#455,#330));
#1861=IFCFACEOUTERBOUND(#1860,.T.);
#1862=IFCFACE((#1861));
#1863=IFCPOLYLOOP((#579,#442,#448));
#1864=IFCFACEOUTERBOUND(#1863,.T.);
#1865=IFCFACE((#1864));
#1866=IFCPOLYLOOP((#579,#448,#455));
#1867=IFCFACEOUTERBOUND(#1866,.T.);
#1868=IFCFACE((#1867));
#1869=IFCPOLYLOOP((#579,#464,#458));
#1870=IFCFACEOUTERBOUND(#1869,.T.);
#1871=IFCFACE((#1870));
#1872=IFCPOLYLOOP((#579,#472,#464));
#1873=IFCFACEOUTERBOUND(#1872,.T.);
#1874=IFCFACE((#1873));
#1875=IFCPOLYLOOP((#580,#577,#554));
#1876=IFCFACEOUTERBOUND(#1875,.T.);
#1877=IFCFACE((#1876));
#1878=IFCPOLYLOOP((#581,#338,#337));
#1879=IFCFACEOUTERBOUND(#1878,.T.);
#1880=IFCFACE((#1879));
#1881=IFCPOLYLOOP((#581,#337,#460));
#1882=IFCFACEOUTERBOUND(#1881,.T.);
#1883=IFCFACE((#1882));
#1884=IFCPOLYLOOP((#581,#460,#466));
#1885=IFCFACEOUTERBOUND(#1884,.T.);
#1886=IFCFACE((#1885));
#1887=IFCPOLYLOOP((#581,#466,#338));
#1888=IFCFACEOUTERBOUND(#1887,.T.);
#1889=IFCFACE((#1888));
#1890=IFCPOLYLOOP((#582,#468,#473));
#1891=IFCFACEOUTERBOUND(#1890,.T.);
#1892=IFCFACE((#1891));
#1893=IFCPOLYLOOP((#582,#338,#468));
#1894=IFCFACEOUTERBOUND(#1893,.T.);
#1895=IFCFACE((#1894));
#1896=IFCPOLYLOOP((#582,#473,#325));
#1897=IFCFACEOUTERBOUND(#1896,.T.);
#1898=IFCFACE((#1897));
#1899=IFCPOLYLOOP((#582,#325,#338));
#1900=IFCFACEOUTERBOUND(#1899,.T.);
#1901=IFCFACE((#1900));
#1902=IFCPOLYLOOP((#583,#478,#313));
#1903=IFCFACEOUTERBOUND(#1902,.T.);
#1904=IFCFACE((#1903));
#1905=IFCPOLYLOOP((#583,#479,#478));
#1906=IFCFACEOUTERBOUND(#1905,.T.);
#1907=IFCFACE((#1906));
#1908=IFCPOLYLOOP((#372,#563,#548));
#1909=IFCFACEOUTERBOUND(#1908,.T.);
#1910=IFCFACE((#1909));
#1911=IFCPOLYLOOP((#584,#477,#479));
#1912=IFCFACEOUTERBOUND(#1911,.T.);
#1913=IFCFACE((#1912));
#1914=IFCPOLYLOOP((#584,#483,#477));
#1915=IFCFACEOUTERBOUND(#1914,.T.);
#1916=IFCFACE((#1915));
#1917=IFCPOLYLOOP((#585,#560,#549));
#1918=IFCFACEOUTERBOUND(#1917,.T.);
#1919=IFCFACE((#1918));
#1920=IFCPOLYLOOP((#585,#549,#574));
#1921=IFCFACEOUTERBOUND(#1920,.T.);
#1922=IFCFACE((#1921));
#1923=IFCPOLYLOOP((#584,#479,#583));
#1924=IFCFACEOUTERBOUND(#1923,.T.);
#1925=IFCFACE((#1924));
#1926=IFCPOLYLOOP((#586,#315,#313));
#1927=IFCFACEOUTERBOUND(#1926,.T.);
#1928=IFCFACE((#1927));
#1929=IFCPOLYLOOP((#586,#313,#478));
#1930=IFCFACEOUTERBOUND(#1929,.T.);
#1931=IFCFACE((#1930));
#1932=IFCPOLYLOOP((#586,#478,#477));
#1933=IFCFACEOUTERBOUND(#1932,.T.);
#1934=IFCFACE((#1933));
#1935=IFCPOLYLOOP((#587,#523,#524));
#1936=IFCFACEOUTERBOUND(#1935,.T.);
#1937=IFCFACE((#1936));
#1938=IFCPOLYLOOP((#586,#477,#380));
#1939=IFCFACEOUTERBOUND(#1938,.T.);
#1940=IFCFACE((#1939));
#1941=IFCPOLYLOOP((#586,#380,#315));
#1942=IFCFACEOUTERBOUND(#1941,.T.);
#1943=IFCFACE((#1942));
#1944=IFCPOLYLOOP((#587,#558,#523));
#1945=IFCFACEOUTERBOUND(#1944,.T.);
#1946=IFCFACE((#1945));
#1947=IFCPOLYLOOP((#588,#483,#584));
#1948=IFCFACEOUTERBOUND(#1947,.T.);
#1949=IFCFACE((#1948));
#1950=IFCPOLYLOOP((#588,#584,#583));
#1951=IFCFACEOUTERBOUND(#1950,.T.);
#1952=IFCFACE((#1951));
#1953=IFCPOLYLOOP((#588,#583,#313));
#1954=IFCFACEOUTERBOUND(#1953,.T.);
#1955=IFCFACE((#1954));
#1956=IFCPOLYLOOP((#589,#518,#558));
#1957=IFCFACEOUTERBOUND(#1956,.T.);
#1958=IFCFACE((#1957));
#1959=IFCPOLYLOOP((#590,#324,#320));
#1960=IFCFACEOUTERBOUND(#1959,.T.);
#1961=IFCFACE((#1960));
#1962=IFCPOLYLOOP((#590,#539,#538));
#1963=IFCFACEOUTERBOUND(#1962,.T.);
#1964=IFCFACE((#1963));
#1965=IFCPOLYLOOP((#590,#320,#539));
#1966=IFCFACEOUTERBOUND(#1965,.T.);
#1967=IFCFACE((#1966));
#1968=IFCPOLYLOOP((#589,#527,#518));
#1969=IFCFACEOUTERBOUND(#1968,.T.);
#1970=IFCFACE((#1969));
#1971=IFCPOLYLOOP((#590,#538,#545));
#1972=IFCFACEOUTERBOUND(#1971,.T.);
#1973=IFCFACE((#1972));
#1974=IFCPOLYLOOP((#590,#545,#324));
#1975=IFCFACEOUTERBOUND(#1974,.T.);
#1976=IFCFACE((#1975));
#1977=IFCPOLYLOOP((#591,#540,#539));
#1978=IFCFACEOUTERBOUND(#1977,.T.);
#1979=IFCFACE((#1978));
#1980=IFCPOLYLOOP((#591,#546,#540));
#1981=IFCFACEOUTERBOUND(#1980,.T.);
#1982=IFCFACE((#1981));
#1983=IFCPOLYLOOP((#591,#320,#546));
#1984=IFCFACEOUTERBOUND(#1983,.T.);
#1985=IFCFACE((#1984));
#1986=IFCPOLYLOOP((#355,#527,#589));
#1987=IFCFACEOUTERBOUND(#1986,.T.);
#1988=IFCFACE((#1987));
#1989=IFCPOLYLOOP((#591,#539,#320));
#1990=IFCFACEOUTERBOUND(#1989,.T.);
#1991=IFCFACE((#1990));
#1992=IFCPOLYLOOP((#592,#368,#356));
#1993=IFCFACEOUTERBOUND(#1992,.T.);
#1994=IFCFACE((#1993));
#1995=IFCPOLYLOOP((#592,#356,#359));
#1996=IFCFACEOUTERBOUND(#1995,.T.);
#1997=IFCFACE((#1996));
#1998=IFCPOLYLOOP((#355,#528,#527));
#1999=IFCFACEOUTERBOUND(#1998,.T.);
#2000=IFCFACE((#1999));
#2001=IFCPOLYLOOP((#592,#593,#550));
#2002=IFCFACEOUTERBOUND(#2001,.T.);
#2003=IFCFACE((#2002));
#2004=IFCPOLYLOOP((#544,#572,#545));
#2005=IFCFACEOUTERBOUND(#2004,.T.);
#2006=IFCFACE((#2005));
#2007=IFCPOLYLOOP((#592,#550,#368));
#2008=IFCFACEOUTERBOUND(#2007,.T.);
#2009=IFCFACE((#2008));
#2010=IFCPOLYLOOP((#592,#359,#593));
#2011=IFCFACEOUTERBOUND(#2010,.T.);
#2012=IFCFACE((#2011));
#2013=IFCPOLYLOOP((#594,#320,#326));
#2014=IFCFACEOUTERBOUND(#2013,.T.);
#2015=IFCFACE((#2014));
#2016=IFCPOLYLOOP((#544,#545,#538));
#2017=IFCFACEOUTERBOUND(#2016,.T.);
#2018=IFCFACE((#2017));
#2019=IFCPOLYLOOP((#594,#546,#320));
#2020=IFCFACEOUTERBOUND(#2019,.T.);
#2021=IFCFACE((#2020));
#2022=IFCPOLYLOOP((#542,#534,#570));
#2023=IFCFACEOUTERBOUND(#2022,.T.);
#2024=IFCFACE((#2023));
#2025=IFCPOLYLOOP((#594,#550,#593));
#2026=IFCFACEOUTERBOUND(#2025,.T.);
#2027=IFCFACE((#2026));
#2028=IFCPOLYLOOP((#594,#326,#550));
#2029=IFCFACEOUTERBOUND(#2028,.T.);
#2030=IFCFACE((#2029));
#2031=IFCPOLYLOOP((#595,#335,#415));
#2032=IFCFACEOUTERBOUND(#2031,.T.);
#2033=IFCFACE((#2032));
#2034=IFCPOLYLOOP((#596,#560,#585));
#2035=IFCFACEOUTERBOUND(#2034,.T.);
#2036=IFCFACE((#2035));
#2037=IFCPOLYLOOP((#595,#415,#403));
#2038=IFCFACEOUTERBOUND(#2037,.T.);
#2039=IFCFACE((#2038));
#2040=IFCPOLYLOOP((#595,#573,#335));
#2041=IFCFACEOUTERBOUND(#2040,.T.);
#2042=IFCFACE((#2041));
#2043=IFCPOLYLOOP((#596,#530,#560));
#2044=IFCFACEOUTERBOUND(#2043,.T.);
#2045=IFCFACE((#2044));
#2046=IFCPOLYLOOP((#597,#595,#403));
#2047=IFCFACEOUTERBOUND(#2046,.T.);
#2048=IFCFACE((#2047));
#2049=IFCPOLYLOOP((#597,#403,#396));
#2050=IFCFACEOUTERBOUND(#2049,.T.);
#2051=IFCFACE((#2050));
#2052=IFCPOLYLOOP((#597,#573,#595));
#2053=IFCFACEOUTERBOUND(#2052,.T.);
#2054=IFCFACE((#2053));
#2055=IFCPOLYLOOP((#598,#536,#577));
#2056=IFCFACEOUTERBOUND(#2055,.T.);
#2057=IFCFACE((#2056));
#2058=IFCPOLYLOOP((#597,#396,#566));
#2059=IFCFACEOUTERBOUND(#2058,.T.);
#2060=IFCFACE((#2059));
#2061=IFCPOLYLOOP((#597,#566,#439));
#2062=IFCFACEOUTERBOUND(#2061,.T.);
#2063=IFCFACE((#2062));
#2064=IFCPOLYLOOP((#598,#548,#536));
#2065=IFCFACEOUTERBOUND(#2064,.T.);
#2066=IFCFACE((#2065));
#2067=IFCPOLYLOOP((#599,#336,#573));
#2068=IFCFACEOUTERBOUND(#2067,.T.);
#2069=IFCFACE((#2068));
#2070=IFCPOLYLOOP((#599,#573,#597));
#2071=IFCFACEOUTERBOUND(#2070,.T.);
#2072=IFCFACE((#2071));
#2073=IFCPOLYLOOP((#599,#439,#446));
#2074=IFCFACEOUTERBOUND(#2073,.T.);
#2075=IFCFACE((#2074));
#2076=IFCPOLYLOOP((#599,#446,#336));
#2077=IFCFACEOUTERBOUND(#2076,.T.);
#2078=IFCFACE((#2077));
#2079=IFCPOLYLOOP((#599,#597,#439));
#2080=IFCFACEOUTERBOUND(#2079,.T.);
#2081=IFCFACE((#2080));
#2082=IFCPOLYLOOP((#340,#598,#577));
#2083=IFCFACEOUTERBOUND(#2082,.T.);
#2084=IFCFACE((#2083));
#2085=IFCPOLYLOOP((#340,#577,#580));
#2086=IFCFACEOUTERBOUND(#2085,.T.);
#2087=IFCFACE((#2086));
#2088=IFCPOLYLOOP((#600,#457,#449));
#2089=IFCFACEOUTERBOUND(#2088,.T.);
#2090=IFCFACE((#2089));
#2091=IFCPOLYLOOP((#600,#576,#457));
#2092=IFCFACEOUTERBOUND(#2091,.T.);
#2093=IFCFACE((#2092));
#2094=IFCPOLYLOOP((#361,#580,#554));
#2095=IFCFACEOUTERBOUND(#2094,.T.);
#2096=IFCFACE((#2095));
#2097=IFCPOLYLOOP((#601,#313,#321));
#2098=IFCFACEOUTERBOUND(#2097,.T.);
#2099=IFCFACE((#2098));
#2100=IFCPOLYLOOP((#601,#588,#313));
#2101=IFCFACEOUTERBOUND(#2100,.T.);
#2102=IFCFACE((#2101));
#2103=IFCPOLYLOOP((#601,#490,#492));
#2104=IFCFACEOUTERBOUND(#2103,.T.);
#2105=IFCFACE((#2104));
#2106=IFCPOLYLOOP((#601,#321,#490));
#2107=IFCFACEOUTERBOUND(#2106,.T.);
#2108=IFCFACE((#2107));
#2109=IFCPOLYLOOP((#361,#554,#528));
#2110=IFCFACEOUTERBOUND(#2109,.T.);
#2111=IFCFACE((#2110));
#2112=IFCPOLYLOOP((#602,#546,#594));
#2113=IFCFACEOUTERBOUND(#2112,.T.);
#2114=IFCFACE((#2113));
#2115=IFCPOLYLOOP((#602,#544,#543));
#2116=IFCFACEOUTERBOUND(#2115,.T.);
#2117=IFCFACE((#2116));
#2118=IFCPOLYLOOP((#602,#543,#546));
#2119=IFCFACEOUTERBOUND(#2118,.T.);
#2120=IFCFACE((#2119));
#2121=IFCPOLYLOOP((#344,#530,#596));
#2122=IFCFACEOUTERBOUND(#2121,.T.);
#2123=IFCFACE((#2122));
#2124=IFCPOLYLOOP((#344,#570,#530));
#2125=IFCFACEOUTERBOUND(#2124,.T.);
#2126=IFCFACE((#2125));
#2127=IFCPOLYLOOP((#603,#594,#593));
#2128=IFCFACEOUTERBOUND(#2127,.T.);
#2129=IFCFACE((#2128));
#2130=IFCPOLYLOOP((#603,#602,#594));
#2131=IFCFACEOUTERBOUND(#2130,.T.);
#2132=IFCFACE((#2131));
#2133=IFCPOLYLOOP((#604,#585,#574));
#2134=IFCFACEOUTERBOUND(#2133,.T.);
#2135=IFCFACE((#2134));
#2136=IFCPOLYLOOP((#603,#544,#602));
#2137=IFCFACEOUTERBOUND(#2136,.T.);
#2138=IFCFACE((#2137));
#2139=IFCPOLYLOOP((#603,#359,#544));
#2140=IFCFACEOUTERBOUND(#2139,.T.);
#2141=IFCFACE((#2140));
#2142=IFCPOLYLOOP((#604,#563,#372));
#2143=IFCFACEOUTERBOUND(#2142,.T.);
#2144=IFCFACE((#2143));
#2145=IFCPOLYLOOP((#603,#593,#359));
#2146=IFCFACEOUTERBOUND(#2145,.T.);
#2147=IFCFACE((#2146));
#2148=IFCPOLYLOOP((#605,#417,#424));
#2149=IFCFACEOUTERBOUND(#2148,.T.);
#2150=IFCFACE((#2149));
#2151=IFCPOLYLOOP((#604,#574,#563));
#2152=IFCFACEOUTERBOUND(#2151,.T.);
#2153=IFCFACE((#2152));
#2154=IFCPOLYLOOP((#605,#431,#576));
#2155=IFCFACEOUTERBOUND(#2154,.T.);
#2156=IFCFACE((#2155));
#2157=IFCPOLYLOOP((#358,#524,#572));
#2158=IFCFACEOUTERBOUND(#2157,.T.);
#2159=IFCFACE((#2158));
#2160=IFCPOLYLOOP((#605,#424,#431));
#2161=IFCFACEOUTERBOUND(#2160,.T.);
#2162=IFCFACE((#2161));
#2163=IFCPOLYLOOP((#605,#571,#417));
#2164=IFCFACEOUTERBOUND(#2163,.T.);
#2165=IFCFACE((#2164));
#2166=IFCPOLYLOOP((#358,#587,#524));
#2167=IFCFACEOUTERBOUND(#2166,.T.);
#2168=IFCFACE((#2167));
#2169=IFCPOLYLOOP((#605,#576,#600));
#2170=IFCFACEOUTERBOUND(#2169,.T.);
#2171=IFCFACE((#2170));
#2172=IFCPOLYLOOP((#606,#600,#449));
#2173=IFCFACEOUTERBOUND(#2172,.T.);
#2174=IFCFACE((#2173));
#2175=IFCPOLYLOOP((#349,#585,#604));
#2176=IFCFACEOUTERBOUND(#2175,.T.);
#2177=IFCFACE((#2176));
#2178=IFCPOLYLOOP((#606,#571,#605));
#2179=IFCFACEOUTERBOUND(#2178,.T.);
#2180=IFCFACE((#2179));
#2181=IFCPOLYLOOP((#606,#449,#443));
#2182=IFCFACEOUTERBOUND(#2181,.T.);
#2183=IFCFACE((#2182));
#2184=IFCPOLYLOOP((#606,#443,#571));
#2185=IFCFACEOUTERBOUND(#2184,.T.);
#2186=IFCFACE((#2185));
#2187=IFCPOLYLOOP((#606,#605,#600));
#2188=IFCFACEOUTERBOUND(#2187,.T.);
#2189=IFCFACE((#2188));
#2190=IFCPOLYLOOP((#607,#483,#588));
#2191=IFCFACEOUTERBOUND(#2190,.T.);
#2192=IFCFACE((#2191));
#2193=IFCPOLYLOOP((#607,#588,#601));
#2194=IFCFACEOUTERBOUND(#2193,.T.);
#2195=IFCFACE((#2194));
#2196=IFCPOLYLOOP((#607,#492,#491));
#2197=IFCFACEOUTERBOUND(#2196,.T.);
#2198=IFCFACE((#2197));
#2199=IFCPOLYLOOP((#607,#601,#492));
#2200=IFCFACEOUTERBOUND(#2199,.T.);
#2201=IFCFACE((#2200));
#2202=IFCPOLYLOOP((#608,#558,#587));
#2203=IFCFACEOUTERBOUND(#2202,.T.);
#2204=IFCFACE((#2203));
#2205=IFCPOLYLOOP((#609,#483,#607));
#2206=IFCFACEOUTERBOUND(#2205,.T.);
#2207=IFCFACE((#2206));
#2208=IFCPOLYLOOP((#608,#589,#558));
#2209=IFCFACEOUTERBOUND(#2208,.T.);
#2210=IFCFACE((#2209));
#2211=IFCPOLYLOOP((#609,#482,#483));
#2212=IFCFACEOUTERBOUND(#2211,.T.);
#2213=IFCFACE((#2212));
#2214=IFCPOLYLOOP((#346,#542,#570));
#2215=IFCFACEOUTERBOUND(#2214,.T.);
#2216=IFCFACE((#2215));
#2217=IFCPOLYLOOP((#346,#332,#542));
#2218=IFCFACEOUTERBOUND(#2217,.T.);
#2219=IFCFACE((#2218));
#2220=IFCPOLYLOOP((#346,#570,#344));
#2221=IFCFACEOUTERBOUND(#2220,.T.);
#2222=IFCFACE((#2221));
#2223=IFCPOLYLOOP((#610,#391,#482));
#2224=IFCFACEOUTERBOUND(#2223,.T.);
#2225=IFCFACE((#2224));
#2226=IFCPOLYLOOP((#348,#596,#585));
#2227=IFCFACEOUTERBOUND(#2226,.T.);
#2228=IFCFACE((#2227));
#2229=IFCPOLYLOOP((#610,#482,#609));
#2230=IFCFACEOUTERBOUND(#2229,.T.);
#2231=IFCFACE((#2230));
#2232=IFCPOLYLOOP((#348,#585,#349));
#2233=IFCFACEOUTERBOUND(#2232,.T.);
#2234=IFCFACE((#2233));
#2235=IFCPOLYLOOP((#610,#609,#607));
#2236=IFCFACEOUTERBOUND(#2235,.T.);
#2237=IFCFACE((#2236));
#2238=IFCPOLYLOOP((#611,#389,#391));
#2239=IFCFACEOUTERBOUND(#2238,.T.);
#2240=IFCFACE((#2239));
#2241=IFCPOLYLOOP((#611,#491,#493));
#2242=IFCFACEOUTERBOUND(#2241,.T.);
#2243=IFCFACE((#2242));
#2244=IFCPOLYLOOP((#352,#355,#589));
#2245=IFCFACEOUTERBOUND(#2244,.T.);
#2246=IFCFACE((#2245));
#2247=IFCPOLYLOOP((#611,#607,#491));
#2248=IFCFACEOUTERBOUND(#2247,.T.);
#2249=IFCFACE((#2248));
#2250=IFCPOLYLOOP((#611,#391,#610));
#2251=IFCFACEOUTERBOUND(#2250,.T.);
#2252=IFCFACE((#2251));
#2253=IFCPOLYLOOP((#611,#610,#607));
#2254=IFCFACEOUTERBOUND(#2253,.T.);
#2255=IFCFACE((#2254));
#2256=IFCPOLYLOOP((#352,#589,#608));
#2257=IFCFACEOUTERBOUND(#2256,.T.);
#2258=IFCFACE((#2257));
#2259=IFCPOLYLOOP((#611,#406,#389));
#2260=IFCFACEOUTERBOUND(#2259,.T.);
#2261=IFCFACE((#2260));
#2262=IFCPOLYLOOP((#611,#493,#406));
#2263=IFCFACEOUTERBOUND(#2262,.T.);
#2264=IFCFACE((#2263));
#2265=IFCPOLYLOOP((#359,#572,#544));
#2266=IFCFACEOUTERBOUND(#2265,.T.);
#2267=IFCFACE((#2266));
#2268=IFCPOLYLOOP((#359,#358,#572));
#2269=IFCFACEOUTERBOUND(#2268,.T.);
#2270=IFCFACE((#2269));
#2271=IFCPOLYLOOP((#354,#361,#528));
#2272=IFCFACEOUTERBOUND(#2271,.T.);
#2273=IFCFACE((#2272));
#2274=IFCPOLYLOOP((#354,#528,#355));
#2275=IFCFACEOUTERBOUND(#2274,.T.);
#2276=IFCFACE((#2275));
#2277=IFCPOLYLOOP((#345,#344,#596));
#2278=IFCFACEOUTERBOUND(#2277,.T.);
#2279=IFCFACE((#2278));
#2280=IFCPOLYLOOP((#345,#596,#348));
#2281=IFCFACEOUTERBOUND(#2280,.T.);
#2282=IFCFACE((#2281));
#2283=IFCPOLYLOOP((#371,#349,#604));
#2284=IFCFACEOUTERBOUND(#2283,.T.);
#2285=IFCFACE((#2284));
#2286=IFCPOLYLOOP((#371,#604,#372));
#2287=IFCFACEOUTERBOUND(#2286,.T.);
#2288=IFCFACE((#2287));
#2289=IFCPOLYLOOP((#341,#340,#580));
#2290=IFCFACEOUTERBOUND(#2289,.T.);
#2291=IFCFACE((#2290));
#2292=IFCPOLYLOOP((#341,#580,#361));
#2293=IFCFACEOUTERBOUND(#2292,.T.);
#2294=IFCFACE((#2293));
#2295=IFCPOLYLOOP((#373,#372,#548));
#2296=IFCFACEOUTERBOUND(#2295,.T.);
#2297=IFCFACE((#2296));
#2298=IFCPOLYLOOP((#373,#598,#340));
#2299=IFCFACEOUTERBOUND(#2298,.T.);
#2300=IFCFACE((#2299));
#2301=IFCPOLYLOOP((#373,#548,#598));
#2302=IFCFACEOUTERBOUND(#2301,.T.);
#2303=IFCFACE((#2302));
#2304=IFCPOLYLOOP((#342,#373,#340));
#2305=IFCFACEOUTERBOUND(#2304,.T.);
#2306=IFCFACE((#2305));
#2307=IFCPOLYLOOP((#350,#349,#371));
#2308=IFCFACEOUTERBOUND(#2307,.T.);
#2309=IFCFACE((#2308));
#2310=IFCPOLYLOOP((#357,#587,#358));
#2311=IFCFACEOUTERBOUND(#2310,.T.);
#2312=IFCFACE((#2311));
#2313=IFCPOLYLOOP((#357,#608,#587));
#2314=IFCFACEOUTERBOUND(#2313,.T.);
#2315=IFCFACE((#2314));
#2316=IFCPOLYLOOP((#353,#352,#608));
#2317=IFCFACEOUTERBOUND(#2316,.T.);
#2318=IFCFACE((#2317));
#2319=IFCPOLYLOOP((#353,#608,#357));
#2320=IFCFACEOUTERBOUND(#2319,.T.);
#2321=IFCFACE((#2320));
#2322=IFCOPENSHELL((#614,#617,#620,#623,#626,#629,#632,#635,#638,#641,#644,#647,#650,#653,#656,#659,#662,#665,#668,#671,#674,#677,#680,#683,#686,#689,#692,#695,#698,#701,#704,#707,#710,#713,#716,#719,#722,#725,#728,#731,#734,#737,#740,#743,#746,#749,#752,#755,#758,#761,#764,#767,#770,#773,#776,#779,#782,#785,#788,#791,#794,#797,#800,#803,#806,#809,#812,#815,#818,#821,#824,#827,#830,#833,#836,#839,#842,#845,#848,#851,#854,#857,#860,#863,#866,#869,#872,#875,#878,#881,#884,#887,#890,#893,#896,#899,#902,#905,#908,#911,#914,#917,#920,#923,#926,#929,#932,#935,#938,#941,#944,#947,#950,#953,#956,#959,#962,#965,#968,#971,#974,#977,#980,#983,#986,#989,#992,#995,#998,#1001,#1004,#1007,#1010,#1013,#1016,#1019,#1022,#1025,#1028,#1031,#1034,#1037,#1040,#1043,#1046,#1049,#1052,#1055,#1058,#1061,#1064,#1067,#1070,#1073,#1076,#1079,#1082,#1085,#1088,#1091,#1094,#1097,#1100,#1103,#1106,#1109,#1112,#1115,#1118,#1121,#1124,#1127,#1130,#1133,#1136,#1139,#1142,#1145,#1148,#1151,#1154,#1157,#1160,#1163,#1166,#1169,#1172,#1175,#1178,#1181,#1184,#1187,#1190,#1193,#1196,#1199,#1202,#1205,#1208,#1211,#1214,#1217,#1220,#1223,#1226,#1229,#1232,#1235,#1238,#1241,#1244,#1247,#1250,#1253,#1256,#1259,#1262,#1265,#1268,#1271,#1274,#1277,#1280,#1283,#1286,#1289,#1292,#1295,#1298,#1301,#1304,#1307,#1310,#1313,#1316,#1319,#1322,#1325,#1328,#1331,#1334,#1337,#1340,#1343,#1346,#1349,#1352,#1355,#1358,#1361,#1364,#1367,#1370,#1373,#1376,#1379,#1382,#1385,#1388,#1391,#1394,#1397,#1400,#1403,#1406,#1409,#1412,#1415,#1418,#1421,#1424,#1427,#1430,#1433,#1436,#1439,#1442,#1445,#1448,#1451,#1454,#1457,#1460,#1463,#1466,#1469,#1472,#1475,#1478,#1481,#1484,#1487,#1490,#1493,#1496,#1499,#1502,#1505,#1508,#1511,#1514,#1517,#1520,#1523,#1526,#1529,#1532,#1535,#1538,#1541,#1544,#1547,#1550,#1553,#1556,#1559,#1562,#1565,#1568,#1571,#1574,#1577,#1580,#1583,#1586,#1589,#1592,#1595,#1598,#1601,#1604,#1607,#1610,#1613,#1616,#1619,#1622,#1625,#1628,#1631,#1634,#1637,#1640,#1643,#1646,#1649,#1652,#1655,#1658,#1661,#1664,#1667,#1670,#1673,#1676,#1679,#1682,#1685,#1688,#1691,#1694,#1697,#1700,#1703,#1706,#1709,#1712,#1715,#1718,#1721,#1724,#1727,#1730,#1733,#1736,#1739,#1742,#1745,#1748,#1751,#1754,#1757,#1760,#1763,#1766,#1769,#1772,#1775,#1778,#1781,#1784,#1787,#1790,#1793,#1796,#1799,#1802,#1805,#1808,#1811,#1814,#1817,#1820,#1823,#1826,#1829,#1832,#1835,#1838,#1841,#1844,#1847,#1850,#1853,#1856,#1859,#1862,#1865,#1868,#1871,#1874,#1877,#1880,#1883,#1886,#1889,#1892,#1895,#1898,#1901,#1904,#1907,#1910,#1913,#1916,#1919,#1922,#1925,#1928,#1931,#1934,#1937,#1940,#1943,#1946,#1949,#1952,#1955,#1958,#1961,#1964,#1967,#1970,#1973,#1976,#1979,#1982,#1985,#1988,#1991,#1994,#1997,#2000,#2003,#2006,#2009,#2012,#2015,#2018,#2021,#2024,#2027,#2030,#2033,#2036,#2039,#2042,#2045,#2048,#2051,#2054,#2057,#2060,#2063,#2066,#2069,#2072,#2075,#2078,#2081,#2084,#2087,#2090,#2093,#2096,#2099,#2102,#2105,#2108,#2111,#2114,#2117,#2120,#2123,#2126,#2129,#2132,#2135,#2138,#2141,#2144,#2147,#2150,#2153,#2156,#2159,#2162,#2165,#2168,#2171,#2174,#2177,#2180,#2183,#2186,#2189,#2192,#2195,#2198,#2201,#2204,#2207,#2210,#2213,#2216,#2219,#2222,#2225,#2228,#2231,#2234,#2237,#2240,#2243,#2246,#2249,#2252,#2255,#2258,#2261,#2264,#2267,#2270,#2273,#2276,#2279,#2282,#2285,#2288,#2291,#2294,#2297,#2300,#2303,#2306,#2309,#2312,#2315,#2318,#2321));
#2323=IFCFACEBASEDSURFACEMODEL((#2322));
#2324=IFCSHAPEREPRESENTATION(#11,'Facetation','SurfaceModel',(#2323));
#2325=IFCPRODUCTDEFINITIONSHAPE($,$,(#2324));
#2326=IFCCOLOURRGB($,0.15,0.25,0.05);
#2327=IFCSURFACESTYLERENDERING(#2326,$,$,$,$,$,$,$,.FLAT.);
#2328=IFCSURFACESTYLE($,.BOTH.,(#2327));
#2329=IFCPRESENTATIONSTYLEASSIGNMENT((#2328));
#2330=IFCSTYLEDITEM(#2323,(#2329),$);
#2331=IFCMATERIAL('Brick',$,$);
#2332=IFCMATERIALLAYER(#2331,360.,$,$,$,$,$);
#2333=IFCMATERIALLAYERSET((#2332),'Wall',$);
#2334=IFCMATERIALLAYERSETUSAGE(#2333,.AXIS2.,.POSITIVE.,-180.,$);
#2335=IFCRELASSOCIATESMATERIAL('2rFU1yyePCieRIvFv9lVbU',#5,$,$,(#40,#221,#268,#281),#2334);
#2336=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2351));
#2337=IFCPRODUCTDEFINITIONSHAPE($,$,(#2336));
#2338=IFCCARTESIANPOINT((0.,0.));
#2339=IFCCARTESIANPOINT((250.,0.));
#2340=IFCCARTESIANPOINT((250.,200.));
#2341=IFCCARTESIANPOINT((500.,200.));
#2342=IFCCARTESIANPOINT((500.,400.));
#2343=IFCCARTESIANPOINT((0.,400.));
#2344=IFCDIRECTION((0.,0.,1.));
#2345=IFCDIRECTION((1.,0.,0.));
#2346=IFCDIRECTION((0.,0.,1.));
#2347=IFCCARTESIANPOINT((0.,0.,0.));
#2348=IFCAXIS2PLACEMENT3D(#2347,#2346,#2345);
#2349=IFCPOLYLINE((#2338,#2339,#2340,#2341,#2342,#2343,#2338));
#2350=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2349);
#2351=IFCEXTRUDEDAREASOLID(#2350,#2348,#2344,1200.);
#2352=IFCDIRECTION((1.,0.,0.));
#2353=IFCDIRECTION((0.,1.,0.));
#2354=IFCCARTESIANPOINT((5050.,1000.,0.));
#2355=IFCAXIS2PLACEMENT3D(#2354,#2353,#2352);
#2356=IFCLOCALPLACEMENT(#37,#2355);
#2357=IFCSTAIRFLIGHT('2Ab2jpYsXCGu4_o2TnJmpB',#5,$,$,$,#2356,#2337,$,2,2,0.2,0.25,.STRAIGHT.);
#2358=IFCRELCONTAINEDINSPATIALSTRUCTURE('1HAeYxqvL2lu5QQDtDh0Ba',#5,$,$,(#2357),#38);
#2359=IFCSTYLEDITEM(#2351,(#97),$);
#2360=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2374));
#2361=IFCPRODUCTDEFINITIONSHAPE($,$,(#2360));
#2362=IFCCARTESIANPOINT((-500.,-500.));
#2363=IFCCARTESIANPOINT((500.,-500.));
#2364=IFCCARTESIANPOINT((500.,500.));
#2365=IFCCARTESIANPOINT((-500.,500.));
#2366=IFCCARTESIANPOINT((-500.,-500.));
#2367=IFCDIRECTION((0.,0.,1.));
#2368=IFCDIRECTION((1.,0.,0.));
#2369=IFCDIRECTION((0.,0.,1.));
#2370=IFCCARTESIANPOINT((0.,0.,0.));
#2371=IFCAXIS2PLACEMENT3D(#2370,#2369,#2368);
#2372=IFCPOLYLINE((#2362,#2363,#2364,#2365,#2366,#2362));
#2373=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2372);
#2374=IFCEXTRUDEDAREASOLID(#2373,#2371,#2367,2200.);
#2375=IFCDIRECTION((1.,0.,0.));
#2376=IFCDIRECTION((0.,0.,1.));
#2377=IFCCARTESIANPOINT((4820.,1600.,0.));
#2378=IFCAXIS2PLACEMENT3D(#2377,#2376,#2375);
#2379=IFCLOCALPLACEMENT(#37,#2378);
#2380=IFCOPENINGELEMENT('2tj6XC4fPEOAaGWGahOG04',#5,$,$,$,#2379,#2361,$,.OPENING.);
#2381=IFCRELVOIDSELEMENT('1SuDtNOsn8kv7yk4YhXg1$',#5,$,$,#268,#2380);
#2382=IFCDIRECTION((0.,1.,0.));
#2383=IFCDIRECTION((0.,0.,1.));
#2384=IFCCARTESIANPOINT((4800.,1600.,0.));
#2385=IFCAXIS2PLACEMENT3D(#2384,#2383,#2382);
#2386=IFCLOCALPLACEMENT(#37,#2385);
#2387=IFCDIRECTION((1.,0.,0.));
#2388=IFCDIRECTION((0.,0.,1.));
#2389=IFCCARTESIANPOINT((460.,0.,0.));
#2390=IFCAXIS2PLACEMENT3D(#2389,#2388,#2387);
#2391=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2401,#2414,#2427,#2440));
#2392=IFCPRODUCTDEFINITIONSHAPE($,$,(#2391));
#2393=IFCCARTESIANPOINT((-40.,-40.));
#2394=IFCCARTESIANPOINT((40.,-40.));
#2395=IFCCARTESIANPOINT((40.,40.));
#2396=IFCCARTESIANPOINT((-40.,40.));
#2397=IFCCARTESIANPOINT((-40.,-40.));
#2398=IFCDIRECTION((0.,0.,1.));
#2399=IFCPOLYLINE((#2393,#2394,#2395,#2396,#2397,#2393));
#2400=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2399);
#2401=IFCEXTRUDEDAREASOLID(#2400,#2390,#2398,2120.);
#2402=IFCDIRECTION((1.,0.,0.));
#2403=IFCDIRECTION((0.,0.,1.));
#2404=IFCCARTESIANPOINT((-460.,0.,0.));
#2405=IFCAXIS2PLACEMENT3D(#2404,#2403,#2402);
#2406=IFCCARTESIANPOINT((-40.,-40.));
#2407=IFCCARTESIANPOINT((40.,-40.));
#2408=IFCCARTESIANPOINT((40.,40.));
#2409=IFCCARTESIANPOINT((-40.,40.));
#2410=IFCCARTESIANPOINT((-40.,-40.));
#2411=IFCDIRECTION((0.,0.,1.));
#2412=IFCPOLYLINE((#2406,#2407,#2408,#2409,#2410,#2406));
#2413=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2412);
#2414=IFCEXTRUDEDAREASOLID(#2413,#2405,#2411,2120.);
#2415=IFCDIRECTION((1.,0.,0.));
#2416=IFCDIRECTION((0.,0.,1.));
#2417=IFCCARTESIANPOINT((0.,0.,2120.));
#2418=IFCAXIS2PLACEMENT3D(#2417,#2416,#2415);
#2419=IFCCARTESIANPOINT((-500.,-40.));
#2420=IFCCARTESIANPOINT((500.,-40.));
#2421=IFCCARTESIANPOINT((500.,40.));
#2422=IFCCARTESIANPOINT((-500.,40.));
#2423=IFCCARTESIANPOINT((-500.,-40.));
#2424=IFCDIRECTION((0.,0.,1.));
#2425=IFCPOLYLINE((#2419,#2420,#2421,#2422,#2423,#2419));
#2426=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2425);
#2427=IFCEXTRUDEDAREASOLID(#2426,#2418,#2424,80.);
#2428=IFCCARTESIANPOINT((-430.,-15.));
#2429=IFCCARTESIANPOINT((430.,-15.));
#2430=IFCCARTESIANPOINT((430.,15.));
#2431=IFCCARTESIANPOINT((-430.,15.));
#2432=IFCCARTESIANPOINT((-430.,-15.));
#2433=IFCDIRECTION((0.,0.,1.));
#2434=IFCDIRECTION((1.,0.,0.));
#2435=IFCDIRECTION((0.,0.,1.));
#2436=IFCCARTESIANPOINT((0.,0.,0.));
#2437=IFCAXIS2PLACEMENT3D(#2436,#2435,#2434);
#2438=IFCPOLYLINE((#2428,#2429,#2430,#2431,#2432,#2428));
#2439=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2438);
#2440=IFCEXTRUDEDAREASOLID(#2439,#2437,#2433,2120.);
#2441=IFCDOOR('0Tif_$wI1FwAwq$OJt24I8',#5,$,$,$,#2386,#2392,$,2200.,1000.,.DOOR.,.SINGLE_SWING_LEFT.,$);
#2442=IFCRELCONTAINEDINSPATIALSTRUCTURE('29LApAqVf2EPy5IMpNkh87',#5,$,$,(#2441),#38);
#2443=IFCCOLOURRGB($,0.9,0.9,0.9);
#2444=IFCSURFACESTYLERENDERING(#2443,$,$,$,$,$,$,$,.FLAT.);
#2445=IFCSURFACESTYLE($,.BOTH.,(#2444));
#2446=IFCPRESENTATIONSTYLEASSIGNMENT((#2445));
#2447=IFCSTYLEDITEM(#2401,(#2446),$);
#2448=IFCSTYLEDITEM(#2414,(#2446),$);
#2449=IFCSTYLEDITEM(#2427,(#2446),$);
#2450=IFCSTYLEDITEM(#2440,(#2446),$);
#2451=IFCRELFILLSELEMENT('1xo_O0WKj3CwHLOhk$E2w$',#5,$,$,#2380,#2441);
#2452=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2466));
#2453=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2479));
#2454=IFCCARTESIANPOINT((-930.,-45.));
#2455=IFCCARTESIANPOINT((930.,-45.));
#2456=IFCCARTESIANPOINT((930.,45.));
#2457=IFCCARTESIANPOINT((-930.,45.));
#2458=IFCCARTESIANPOINT((-930.,-45.));
#2459=IFCDIRECTION((0.,0.,1.));
#2460=IFCDIRECTION((1.,0.,0.));
#2461=IFCDIRECTION((0.,0.,1.));
#2462=IFCCARTESIANPOINT((0.,0.,0.));
#2463=IFCAXIS2PLACEMENT3D(#2462,#2461,#2460);
#2464=IFCPOLYLINE((#2454,#2455,#2456,#2457,#2458,#2454));
#2465=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2464);
#2466=IFCEXTRUDEDAREASOLID(#2465,#2463,#2459,90.);
#2467=IFCCARTESIANPOINT((-45.,-45.));
#2468=IFCCARTESIANPOINT((45.,-45.));
#2469=IFCCARTESIANPOINT((45.,45.));
#2470=IFCCARTESIANPOINT((-45.,45.));
#2471=IFCCARTESIANPOINT((-45.,-45.));
#2472=IFCDIRECTION((0.,0.,1.));
#2473=IFCDIRECTION((1.,0.,0.));
#2474=IFCDIRECTION((0.,0.,1.));
#2475=IFCCARTESIANPOINT((0.,0.,0.));
#2476=IFCAXIS2PLACEMENT3D(#2475,#2474,#2473);
#2477=IFCPOLYLINE((#2467,#2468,#2469,#2470,#2471,#2467));
#2478=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2477);
#2479=IFCEXTRUDEDAREASOLID(#2478,#2476,#2472,1420.);
#2480=IFCCOLOURRGB($,0.5,0.4,0.3);
#2481=IFCSURFACESTYLERENDERING(#2480,$,$,$,$,$,$,$,.FLAT.);
#2482=IFCSURFACESTYLE($,.BOTH.,(#2481));
#2483=IFCPRESENTATIONSTYLEASSIGNMENT((#2482));
#2484=IFCSTYLEDITEM(#2466,(#2483),$);
#2485=IFCSTYLEDITEM(#2479,(#2483),$);
#2486=IFCDIRECTION((1.,0.,0.));
#2487=IFCDIRECTION((0.,0.,1.));
#2488=IFCCARTESIANPOINT((-4900.,-45.,400.));
#2489=IFCAXIS2PLACEMENT3D(#2488,#2487,#2486);
#2490=IFCLOCALPLACEMENT(#37,#2489);
#2491=IFCDIRECTION((1.,0.,0.));
#2492=IFCDIRECTION((0.,0.,1.));
#2493=IFCCARTESIANPOINT((-3130.,-45.,400.));
#2494=IFCAXIS2PLACEMENT3D(#2493,#2492,#2491);
#2495=IFCLOCALPLACEMENT(#37,#2494);
#2496=IFCDIRECTION((1.,0.,0.));
#2497=IFCDIRECTION((0.,0.,1.));
#2498=IFCCARTESIANPOINT((-1360.,-45.,400.));
#2499=IFCAXIS2PLACEMENT3D(#2498,#2497,#2496);
#2500=IFCLOCALPLACEMENT(#37,#2499);
#2501=IFCDIRECTION((1.,0.,0.));
#2502=IFCDIRECTION((0.,0.,1.));
#2503=IFCCARTESIANPOINT((2070.,-45.,400.));
#2504=IFCAXIS2PLACEMENT3D(#2503,#2502,#2501);
#2505=IFCLOCALPLACEMENT(#37,#2504);
#2506=IFCDIRECTION((0.,1.,0.));
#2507=IFCDIRECTION((0.,0.,1.));
#2508=IFCCARTESIANPOINT((-4810.,-45.,400.));
#2509=IFCAXIS2PLACEMENT3D(#2508,#2507,#2506);
#2510=IFCLOCALPLACEMENT(#37,#2509);
#2511=IFCWINDOW('3lY7vAqg1AOuHmhrmPSRNL',#5,$,$,$,#2490,$,$,1600.,1860.,.WINDOW.,.SINGLE_PANEL.,$);
#2512=IFCRELCONTAINEDINSPATIALSTRUCTURE('2EIDUEhhn4pAZIg2kaiJJq',#5,$,$,(#2511),#38);
#2513=IFCDIRECTION((1.,0.,0.));
#2514=IFCDIRECTION((0.,0.,1.));
#2515=IFCCARTESIANPOINT((930.,45.,0.));
#2516=IFCAXIS2PLACEMENT3D(#2515,#2514,#2513);
#2517=IFCLOCALPLACEMENT(#2490,#2516);
#2518=IFCDIRECTION((1.,0.,0.));
#2519=IFCDIRECTION((0.,0.,1.));
#2520=IFCCARTESIANPOINT((930.,45.,1510.));
#2521=IFCAXIS2PLACEMENT3D(#2520,#2519,#2518);
#2522=IFCLOCALPLACEMENT(#2490,#2521);
#2523=IFCDIRECTION((1.,0.,0.));
#2524=IFCDIRECTION((0.,0.,1.));
#2525=IFCCARTESIANPOINT((45.,45.,90.));
#2526=IFCAXIS2PLACEMENT3D(#2525,#2524,#2523);
#2527=IFCLOCALPLACEMENT(#2490,#2526);
#2528=IFCDIRECTION((1.,0.,0.));
#2529=IFCDIRECTION((0.,0.,1.));
#2530=IFCCARTESIANPOINT((1815.,45.,90.));
#2531=IFCAXIS2PLACEMENT3D(#2530,#2529,#2528);
#2532=IFCLOCALPLACEMENT(#2490,#2531);
#2533=IFCDIRECTION((1.,0.,0.));
#2534=IFCDIRECTION((0.,0.,1.));
#2535=IFCCARTESIANPOINT((0.,0.,0.));
#2536=IFCAXIS2PLACEMENT3D(#2535,#2534,#2533);
#2537=IFCREPRESENTATIONMAP(#2536,#2452);
#2538=IFCCARTESIANPOINT((0.,0.,0.));
#2539=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2538,$,$);
#2540=IFCMAPPEDITEM(#2537,#2539);
#2541=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2540));
#2542=IFCPRODUCTDEFINITIONSHAPE($,$,(#2541));
#2543=IFCMEMBER('1DDgU0hLvBH8oWeOKX3GI2',#5,$,$,$,#2517,#2542,$,.MULLION.);
#2544=IFCCARTESIANPOINT((0.,0.,0.));
#2545=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2544,$,$);
#2546=IFCMAPPEDITEM(#2537,#2545);
#2547=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2546));
#2548=IFCPRODUCTDEFINITIONSHAPE($,$,(#2547));
#2549=IFCMEMBER('2hlyLAdjfEGeOVzpaN2yyi',#5,$,$,$,#2522,#2548,$,.MULLION.);
#2550=IFCDIRECTION((1.,0.,0.));
#2551=IFCDIRECTION((0.,0.,1.));
#2552=IFCCARTESIANPOINT((0.,0.,0.));
#2553=IFCAXIS2PLACEMENT3D(#2552,#2551,#2550);
#2554=IFCREPRESENTATIONMAP(#2553,#2453);
#2555=IFCCARTESIANPOINT((0.,0.,0.));
#2556=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2555,$,$);
#2557=IFCMAPPEDITEM(#2554,#2556);
#2558=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2557));
#2559=IFCPRODUCTDEFINITIONSHAPE($,$,(#2558));
#2560=IFCMEMBER('2LgB4q_MXBK9HwUSNOlfaz',#5,$,$,$,#2527,#2559,$,.MULLION.);
#2561=IFCCARTESIANPOINT((0.,0.,0.));
#2562=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2561,$,$);
#2563=IFCMAPPEDITEM(#2554,#2562);
#2564=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2563));
#2565=IFCPRODUCTDEFINITIONSHAPE($,$,(#2564));
#2566=IFCMEMBER('2lzT5H31bAIPS3Ep0DTo7x',#5,$,$,$,#2532,#2565,$,.MULLION.);
#2567=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2581));
#2568=IFCPRODUCTDEFINITIONSHAPE($,$,(#2567));
#2569=IFCCARTESIANPOINT((-840.,-5.));
#2570=IFCCARTESIANPOINT((840.,-5.));
#2571=IFCCARTESIANPOINT((840.,5.));
#2572=IFCCARTESIANPOINT((-840.,5.));
#2573=IFCCARTESIANPOINT((-840.,-5.));
#2574=IFCDIRECTION((0.,0.,1.));
#2575=IFCDIRECTION((1.,0.,0.));
#2576=IFCDIRECTION((0.,0.,1.));
#2577=IFCCARTESIANPOINT((0.,0.,0.));
#2578=IFCAXIS2PLACEMENT3D(#2577,#2576,#2575);
#2579=IFCPOLYLINE((#2569,#2570,#2571,#2572,#2573,#2569));
#2580=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2579);
#2581=IFCEXTRUDEDAREASOLID(#2580,#2578,#2574,1420.);
#2582=IFCDIRECTION((1.,0.,0.));
#2583=IFCDIRECTION((0.,0.,1.));
#2584=IFCCARTESIANPOINT((930.,45.,90.));
#2585=IFCAXIS2PLACEMENT3D(#2584,#2583,#2582);
#2586=IFCLOCALPLACEMENT(#2490,#2585);
#2587=IFCPLATE('3wAl_gjCj9ZRWv12kmFW1U',#5,$,$,$,#2586,#2568,$,.SHEET.);
#2588=IFCCOLOURRGB($,0.6,0.7,0.75);
#2589=IFCSURFACESTYLERENDERING(#2588,0.9,$,$,$,$,$,$,.FLAT.);
#2590=IFCSURFACESTYLE($,.BOTH.,(#2589));
#2591=IFCPRESENTATIONSTYLEASSIGNMENT((#2590));
#2592=IFCSTYLEDITEM(#2581,(#2591),$);
#2593=IFCRELAGGREGATES('0R66uG_iv00vYyB5QyeQN2',#5,$,$,#2511,(#2543,#2549,#2560,#2566,#2587));
#2594=IFCWINDOW('1$ROshSLD8PxvcHXbFnnTi',#5,$,$,$,#2495,$,$,1600.,1860.,.WINDOW.,.SINGLE_PANEL.,$);
#2595=IFCRELCONTAINEDINSPATIALSTRUCTURE('1jyLim6t50yfQH9XEmvASJ',#5,$,$,(#2594),#38);
#2596=IFCDIRECTION((1.,0.,0.));
#2597=IFCDIRECTION((0.,0.,1.));
#2598=IFCCARTESIANPOINT((930.,45.,0.));
#2599=IFCAXIS2PLACEMENT3D(#2598,#2597,#2596);
#2600=IFCLOCALPLACEMENT(#2495,#2599);
#2601=IFCDIRECTION((1.,0.,0.));
#2602=IFCDIRECTION((0.,0.,1.));
#2603=IFCCARTESIANPOINT((930.,45.,1510.));
#2604=IFCAXIS2PLACEMENT3D(#2603,#2602,#2601);
#2605=IFCLOCALPLACEMENT(#2495,#2604);
#2606=IFCDIRECTION((1.,0.,0.));
#2607=IFCDIRECTION((0.,0.,1.));
#2608=IFCCARTESIANPOINT((45.,45.,90.));
#2609=IFCAXIS2PLACEMENT3D(#2608,#2607,#2606);
#2610=IFCLOCALPLACEMENT(#2495,#2609);
#2611=IFCDIRECTION((1.,0.,0.));
#2612=IFCDIRECTION((0.,0.,1.));
#2613=IFCCARTESIANPOINT((1815.,45.,90.));
#2614=IFCAXIS2PLACEMENT3D(#2613,#2612,#2611);
#2615=IFCLOCALPLACEMENT(#2495,#2614);
#2616=IFCCARTESIANPOINT((0.,0.,0.));
#2617=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2616,$,$);
#2618=IFCMAPPEDITEM(#2537,#2617);
#2619=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2618));
#2620=IFCPRODUCTDEFINITIONSHAPE($,$,(#2619));
#2621=IFCMEMBER('3h7VkxoBP65wjElx54zMBz',#5,$,$,$,#2600,#2620,$,.MULLION.);
#2622=IFCCARTESIANPOINT((0.,0.,0.));
#2623=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2622,$,$);
#2624=IFCMAPPEDITEM(#2537,#2623);
#2625=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2624));
#2626=IFCPRODUCTDEFINITIONSHAPE($,$,(#2625));
#2627=IFCMEMBER('1M2y_u9$vBTh7Yd5_usZ23',#5,$,$,$,#2605,#2626,$,.MULLION.);
#2628=IFCCARTESIANPOINT((0.,0.,0.));
#2629=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2628,$,$);
#2630=IFCMAPPEDITEM(#2554,#2629);
#2631=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2630));
#2632=IFCPRODUCTDEFINITIONSHAPE($,$,(#2631));
#2633=IFCMEMBER('2X4Md3tin09fSS3NN94BNj',#5,$,$,$,#2610,#2632,$,.MULLION.);
#2634=IFCCARTESIANPOINT((0.,0.,0.));
#2635=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2634,$,$);
#2636=IFCMAPPEDITEM(#2554,#2635);
#2637=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2636));
#2638=IFCPRODUCTDEFINITIONSHAPE($,$,(#2637));
#2639=IFCMEMBER('0vA0LIH5v9gxUg5A53p88K',#5,$,$,$,#2615,#2638,$,.MULLION.);
#2640=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2654));
#2641=IFCPRODUCTDEFINITIONSHAPE($,$,(#2640));
#2642=IFCCARTESIANPOINT((-840.,-5.));
#2643=IFCCARTESIANPOINT((840.,-5.));
#2644=IFCCARTESIANPOINT((840.,5.));
#2645=IFCCARTESIANPOINT((-840.,5.));
#2646=IFCCARTESIANPOINT((-840.,-5.));
#2647=IFCDIRECTION((0.,0.,1.));
#2648=IFCDIRECTION((1.,0.,0.));
#2649=IFCDIRECTION((0.,0.,1.));
#2650=IFCCARTESIANPOINT((0.,0.,0.));
#2651=IFCAXIS2PLACEMENT3D(#2650,#2649,#2648);
#2652=IFCPOLYLINE((#2642,#2643,#2644,#2645,#2646,#2642));
#2653=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2652);
#2654=IFCEXTRUDEDAREASOLID(#2653,#2651,#2647,1420.);
#2655=IFCDIRECTION((1.,0.,0.));
#2656=IFCDIRECTION((0.,0.,1.));
#2657=IFCCARTESIANPOINT((930.,45.,90.));
#2658=IFCAXIS2PLACEMENT3D(#2657,#2656,#2655);
#2659=IFCLOCALPLACEMENT(#2495,#2658);
#2660=IFCPLATE('0p8HXuwiD3yBz1RJLzKo7o',#5,$,$,$,#2659,#2641,$,.SHEET.);
#2661=IFCCOLOURRGB($,0.6,0.7,0.75);
#2662=IFCSURFACESTYLERENDERING(#2661,0.9,$,$,$,$,$,$,.FLAT.);
#2663=IFCSURFACESTYLE($,.BOTH.,(#2662));
#2664=IFCPRESENTATIONSTYLEASSIGNMENT((#2663));
#2665=IFCSTYLEDITEM(#2654,(#2664),$);
#2666=IFCRELAGGREGATES('1LmWuaJXv2kenJFq1BZfkC',#5,$,$,#2594,(#2621,#2627,#2633,#2639,#2660));
#2667=IFCWINDOW('1omMjrueL1RBg$X8j2Z2J5',#5,$,$,$,#2500,$,$,1600.,1860.,.WINDOW.,.SINGLE_PANEL.,$);
#2668=IFCRELCONTAINEDINSPATIALSTRUCTURE('1hYpL5k9HEturVWknCgHrZ',#5,$,$,(#2667),#38);
#2669=IFCDIRECTION((1.,0.,0.));
#2670=IFCDIRECTION((0.,0.,1.));
#2671=IFCCARTESIANPOINT((930.,45.,0.));
#2672=IFCAXIS2PLACEMENT3D(#2671,#2670,#2669);
#2673=IFCLOCALPLACEMENT(#2500,#2672);
#2674=IFCDIRECTION((1.,0.,0.));
#2675=IFCDIRECTION((0.,0.,1.));
#2676=IFCCARTESIANPOINT((930.,45.,1510.));
#2677=IFCAXIS2PLACEMENT3D(#2676,#2675,#2674);
#2678=IFCLOCALPLACEMENT(#2500,#2677);
#2679=IFCDIRECTION((1.,0.,0.));
#2680=IFCDIRECTION((0.,0.,1.));
#2681=IFCCARTESIANPOINT((45.,45.,90.));
#2682=IFCAXIS2PLACEMENT3D(#2681,#2680,#2679);
#2683=IFCLOCALPLACEMENT(#2500,#2682);
#2684=IFCDIRECTION((1.,0.,0.));
#2685=IFCDIRECTION((0.,0.,1.));
#2686=IFCCARTESIANPOINT((1815.,45.,90.));
#2687=IFCAXIS2PLACEMENT3D(#2686,#2685,#2684);
#2688=IFCLOCALPLACEMENT(#2500,#2687);
#2689=IFCCARTESIANPOINT((0.,0.,0.));
#2690=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2689,$,$);
#2691=IFCMAPPEDITEM(#2537,#2690);
#2692=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2691));
#2693=IFCPRODUCTDEFINITIONSHAPE($,$,(#2692));
#2694=IFCMEMBER('0lSDqOsFD7_QJ_J_bqBkIl',#5,$,$,$,#2673,#2693,$,.MULLION.);
#2695=IFCCARTESIANPOINT((0.,0.,0.));
#2696=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2695,$,$);
#2697=IFCMAPPEDITEM(#2537,#2696);
#2698=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2697));
#2699=IFCPRODUCTDEFINITIONSHAPE($,$,(#2698));
#2700=IFCMEMBER('2UFQMMz1jBgPS0zsW6zg5D',#5,$,$,$,#2678,#2699,$,.MULLION.);
#2701=IFCCARTESIANPOINT((0.,0.,0.));
#2702=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2701,$,$);
#2703=IFCMAPPEDITEM(#2554,#2702);
#2704=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2703));
#2705=IFCPRODUCTDEFINITIONSHAPE($,$,(#2704));
#2706=IFCMEMBER('2Ma0fbEZX0R97HX$kAWqQm',#5,$,$,$,#2683,#2705,$,.MULLION.);
#2707=IFCCARTESIANPOINT((0.,0.,0.));
#2708=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2707,$,$);
#2709=IFCMAPPEDITEM(#2554,#2708);
#2710=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2709));
#2711=IFCPRODUCTDEFINITIONSHAPE($,$,(#2710));
#2712=IFCMEMBER('24LmlyfDb0ZeEE4jvefsN5',#5,$,$,$,#2688,#2711,$,.MULLION.);
#2713=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2727));
#2714=IFCPRODUCTDEFINITIONSHAPE($,$,(#2713));
#2715=IFCCARTESIANPOINT((-840.,-5.));
#2716=IFCCARTESIANPOINT((840.,-5.));
#2717=IFCCARTESIANPOINT((840.,5.));
#2718=IFCCARTESIANPOINT((-840.,5.));
#2719=IFCCARTESIANPOINT((-840.,-5.));
#2720=IFCDIRECTION((0.,0.,1.));
#2721=IFCDIRECTION((1.,0.,0.));
#2722=IFCDIRECTION((0.,0.,1.));
#2723=IFCCARTESIANPOINT((0.,0.,0.));
#2724=IFCAXIS2PLACEMENT3D(#2723,#2722,#2721);
#2725=IFCPOLYLINE((#2715,#2716,#2717,#2718,#2719,#2715));
#2726=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2725);
#2727=IFCEXTRUDEDAREASOLID(#2726,#2724,#2720,1420.);
#2728=IFCDIRECTION((1.,0.,0.));
#2729=IFCDIRECTION((0.,0.,1.));
#2730=IFCCARTESIANPOINT((930.,45.,90.));
#2731=IFCAXIS2PLACEMENT3D(#2730,#2729,#2728);
#2732=IFCLOCALPLACEMENT(#2500,#2731);
#2733=IFCPLATE('23dVSDsxXCPhtqOSjMZ_H9',#5,$,$,$,#2732,#2714,$,.SHEET.);
#2734=IFCCOLOURRGB($,0.6,0.7,0.75);
#2735=IFCSURFACESTYLERENDERING(#2734,0.9,$,$,$,$,$,$,.FLAT.);
#2736=IFCSURFACESTYLE($,.BOTH.,(#2735));
#2737=IFCPRESENTATIONSTYLEASSIGNMENT((#2736));
#2738=IFCSTYLEDITEM(#2727,(#2737),$);
#2739=IFCRELAGGREGATES('3jRzHYqYbF2gnPKOpr2ZK9',#5,$,$,#2667,(#2694,#2700,#2706,#2712,#2733));
#2740=IFCWINDOW('2hoO8_fkD7T9eJPdlaToJT',#5,$,$,$,#2505,$,$,1600.,1860.,.WINDOW.,.SINGLE_PANEL.,$);
#2741=IFCRELCONTAINEDINSPATIALSTRUCTURE('2wYllJNCr5nezLPGfsewcd',#5,$,$,(#2740),#38);
#2742=IFCDIRECTION((1.,0.,0.));
#2743=IFCDIRECTION((0.,0.,1.));
#2744=IFCCARTESIANPOINT((930.,45.,0.));
#2745=IFCAXIS2PLACEMENT3D(#2744,#2743,#2742);
#2746=IFCLOCALPLACEMENT(#2505,#2745);
#2747=IFCDIRECTION((1.,0.,0.));
#2748=IFCDIRECTION((0.,0.,1.));
#2749=IFCCARTESIANPOINT((930.,45.,1510.));
#2750=IFCAXIS2PLACEMENT3D(#2749,#2748,#2747);
#2751=IFCLOCALPLACEMENT(#2505,#2750);
#2752=IFCDIRECTION((1.,0.,0.));
#2753=IFCDIRECTION((0.,0.,1.));
#2754=IFCCARTESIANPOINT((45.,45.,90.));
#2755=IFCAXIS2PLACEMENT3D(#2754,#2753,#2752);
#2756=IFCLOCALPLACEMENT(#2505,#2755);
#2757=IFCDIRECTION((1.,0.,0.));
#2758=IFCDIRECTION((0.,0.,1.));
#2759=IFCCARTESIANPOINT((1815.,45.,90.));
#2760=IFCAXIS2PLACEMENT3D(#2759,#2758,#2757);
#2761=IFCLOCALPLACEMENT(#2505,#2760);
#2762=IFCCARTESIANPOINT((0.,0.,0.));
#2763=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2762,$,$);
#2764=IFCMAPPEDITEM(#2537,#2763);
#2765=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2764));
#2766=IFCPRODUCTDEFINITIONSHAPE($,$,(#2765));
#2767=IFCMEMBER('1WrTfNTaPF2fXeGcePrGda',#5,$,$,$,#2746,#2766,$,.MULLION.);
#2768=IFCCARTESIANPOINT((0.,0.,0.));
#2769=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2768,$,$);
#2770=IFCMAPPEDITEM(#2537,#2769);
#2771=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2770));
#2772=IFCPRODUCTDEFINITIONSHAPE($,$,(#2771));
#2773=IFCMEMBER('1aI9bCsPnCXA4IRqZW9S74',#5,$,$,$,#2751,#2772,$,.MULLION.);
#2774=IFCCARTESIANPOINT((0.,0.,0.));
#2775=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2774,$,$);
#2776=IFCMAPPEDITEM(#2554,#2775);
#2777=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2776));
#2778=IFCPRODUCTDEFINITIONSHAPE($,$,(#2777));
#2779=IFCMEMBER('2n2gwY9rv8e9oHNGo8x2qW',#5,$,$,$,#2756,#2778,$,.MULLION.);
#2780=IFCCARTESIANPOINT((0.,0.,0.));
#2781=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2780,$,$);
#2782=IFCMAPPEDITEM(#2554,#2781);
#2783=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2782));
#2784=IFCPRODUCTDEFINITIONSHAPE($,$,(#2783));
#2785=IFCMEMBER('3J3RjR0$n8peLRZVKppSng',#5,$,$,$,#2761,#2784,$,.MULLION.);
#2786=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2800));
#2787=IFCPRODUCTDEFINITIONSHAPE($,$,(#2786));
#2788=IFCCARTESIANPOINT((-840.,-5.));
#2789=IFCCARTESIANPOINT((840.,-5.));
#2790=IFCCARTESIANPOINT((840.,5.));
#2791=IFCCARTESIANPOINT((-840.,5.));
#2792=IFCCARTESIANPOINT((-840.,-5.));
#2793=IFCDIRECTION((0.,0.,1.));
#2794=IFCDIRECTION((1.,0.,0.));
#2795=IFCDIRECTION((0.,0.,1.));
#2796=IFCCARTESIANPOINT((0.,0.,0.));
#2797=IFCAXIS2PLACEMENT3D(#2796,#2795,#2794);
#2798=IFCPOLYLINE((#2788,#2789,#2790,#2791,#2792,#2788));
#2799=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2798);
#2800=IFCEXTRUDEDAREASOLID(#2799,#2797,#2793,1420.);
#2801=IFCDIRECTION((1.,0.,0.));
#2802=IFCDIRECTION((0.,0.,1.));
#2803=IFCCARTESIANPOINT((930.,45.,90.));
#2804=IFCAXIS2PLACEMENT3D(#2803,#2802,#2801);
#2805=IFCLOCALPLACEMENT(#2505,#2804);
#2806=IFCPLATE('0JwnnKhZr3694nxJ1extNw',#5,$,$,$,#2805,#2787,$,.SHEET.);
#2807=IFCCOLOURRGB($,0.6,0.7,0.75);
#2808=IFCSURFACESTYLERENDERING(#2807,0.9,$,$,$,$,$,$,.FLAT.);
#2809=IFCSURFACESTYLE($,.BOTH.,(#2808));
#2810=IFCPRESENTATIONSTYLEASSIGNMENT((#2809));
#2811=IFCSTYLEDITEM(#2800,(#2810),$);
#2812=IFCRELAGGREGATES('1BuFofsH18Aw42Qca7DaEC',#5,$,$,#2740,(#2767,#2773,#2779,#2785,#2806));
#2813=IFCWINDOW('1f2A4_vjLEBRKe77yVKHtq',#5,$,$,$,#2510,$,$,1600.,1860.,.WINDOW.,.SINGLE_PANEL.,$);
#2814=IFCRELCONTAINEDINSPATIALSTRUCTURE('2_fhaS$BXCd9BLx1KbrUMX',#5,$,$,(#2813),#38);
#2815=IFCDIRECTION((1.,0.,0.));
#2816=IFCDIRECTION((0.,0.,1.));
#2817=IFCCARTESIANPOINT((930.,45.,0.));
#2818=IFCAXIS2PLACEMENT3D(#2817,#2816,#2815);
#2819=IFCLOCALPLACEMENT(#2510,#2818);
#2820=IFCDIRECTION((1.,0.,0.));
#2821=IFCDIRECTION((0.,0.,1.));
#2822=IFCCARTESIANPOINT((930.,45.,1510.));
#2823=IFCAXIS2PLACEMENT3D(#2822,#2821,#2820);
#2824=IFCLOCALPLACEMENT(#2510,#2823);
#2825=IFCDIRECTION((1.,0.,0.));
#2826=IFCDIRECTION((0.,0.,1.));
#2827=IFCCARTESIANPOINT((45.,45.,90.));
#2828=IFCAXIS2PLACEMENT3D(#2827,#2826,#2825);
#2829=IFCLOCALPLACEMENT(#2510,#2828);
#2830=IFCDIRECTION((1.,0.,0.));
#2831=IFCDIRECTION((0.,0.,1.));
#2832=IFCCARTESIANPOINT((1815.,45.,90.));
#2833=IFCAXIS2PLACEMENT3D(#2832,#2831,#2830);
#2834=IFCLOCALPLACEMENT(#2510,#2833);
#2835=IFCCARTESIANPOINT((0.,0.,0.));
#2836=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2835,$,$);
#2837=IFCMAPPEDITEM(#2537,#2836);
#2838=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2837));
#2839=IFCPRODUCTDEFINITIONSHAPE($,$,(#2838));
#2840=IFCMEMBER('279dlwph95$gPMCup$shvv',#5,$,$,$,#2819,#2839,$,.MULLION.);
#2841=IFCCARTESIANPOINT((0.,0.,0.));
#2842=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2841,$,$);
#2843=IFCMAPPEDITEM(#2537,#2842);
#2844=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2843));
#2845=IFCPRODUCTDEFINITIONSHAPE($,$,(#2844));
#2846=IFCMEMBER('2T0zJmuRDFx8VqbUOAsLII',#5,$,$,$,#2824,#2845,$,.MULLION.);
#2847=IFCCARTESIANPOINT((0.,0.,0.));
#2848=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2847,$,$);
#2849=IFCMAPPEDITEM(#2554,#2848);
#2850=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2849));
#2851=IFCPRODUCTDEFINITIONSHAPE($,$,(#2850));
#2852=IFCMEMBER('0nCuI0UWT9z9LUZfvNs8Bx',#5,$,$,$,#2829,#2851,$,.MULLION.);
#2853=IFCCARTESIANPOINT((0.,0.,0.));
#2854=IFCCARTESIANTRANSFORMATIONOPERATOR3D($,$,#2853,$,$);
#2855=IFCMAPPEDITEM(#2554,#2854);
#2856=IFCSHAPEREPRESENTATION(#11,'Body','MappedRepresentation',(#2855));
#2857=IFCPRODUCTDEFINITIONSHAPE($,$,(#2856));
#2858=IFCMEMBER('0s18lXjPD0suRoyNawI5Q_',#5,$,$,$,#2834,#2857,$,.MULLION.);
#2859=IFCSHAPEREPRESENTATION(#11,'Body','SweptSolid',(#2873));
#2860=IFCPRODUCTDEFINITIONSHAPE($,$,(#2859));
#2861=IFCCARTESIANPOINT((-840.,-5.));
#2862=IFCCARTESIANPOINT((840.,-5.));
#2863=IFCCARTESIANPOINT((840.,5.));
#2864=IFCCARTESIANPOINT((-840.,5.));
#2865=IFCCARTESIANPOINT((-840.,-5.));
#2866=IFCDIRECTION((0.,0.,1.));
#2867=IFCDIRECTION((1.,0.,0.));
#2868=IFCDIRECTION((0.,0.,1.));
#2869=IFCCARTESIANPOINT((0.,0.,0.));
#2870=IFCAXIS2PLACEMENT3D(#2869,#2868,#2867);
#2871=IFCPOLYLINE((#2861,#2862,#2863,#2864,#2865,#2861));
#2872=IFCARBITRARYCLOSEDPROFILEDEF(.AREA.,$,#2871);
#2873=IFCEXTRUDEDAREASOLID(#2872,#2870,#2866,1420.);
#2874=IFCDIRECTION((1.,0.,0.));
#2875=IFCDIRECTION((0.,0.,1.));
#2876=IFCCARTESIANPOINT((930.,45.,90.));
#2877=IFCAXIS2PLACEMENT3D(#2876,#2875,#2874);
#2878=IFCLOCALPLACEMENT(#2510,#2877);
#2879=IFCPLATE('3vhbFRmyXDouRftZl4AXgJ',#5,$,$,$,#2878,#2860,$,.SHEET.);
#2880=IFCCOLOURRGB($,0.6,0.7,0.75);
#2881=IFCSURFACESTYLERENDERING(#2880,0.9,$,$,$,$,$,$,.FLAT.);
#2882=IFCSURFACESTYLE($,.BOTH.,(#2881));
#2883=IFCPRESENTATIONSTYLEASSIGNMENT((#2882));
#2884=IFCSTYLEDITEM(#2873,(#2883),$);
#2885=IFCRELAGGREGATES('20yL1VssL4uPkX$0lbyk6D',#5,$,$,#2813,(#2840,#2846,#2852,#2858,#2879));
ENDSEC;
END-ISO-10303-21;
"""
