const tslib = require('tslib');
const utils = require('@sentry/utils');
const integrations = require('./integrations.js');
const core = require('@sentry/core');
const mode = require('./common/mode.js');

const Integrations = integrations.getIntegrations();
/** Fetches the SDK entry point for the current process */
function getEntryPoint() {
    try {
        return process.type === 'browser' ? utils.dynamicRequire(module, './main') : require('./renderer');
    }
    catch (e) {
        throw new Error(`Failed to automatically detect correct SDK entry point.

In the Electron main process you should import via:
import * as Sentry from '@sentry/electron/main';

In the Electron renderer process you should import via:
import * as Sentry from '@sentry/electron/renderer';`);
    }
}
/**
 * The Sentry Electron SDK Client.
 *
 * To use this SDK, call the {@link init} function as early as possible in the
 * entry modules. This applies to the main process as well as all renderer
 * processes or further sub processes you spawn. To set context information or
 * send manual events, use the provided methods.
 *
 * @example
 * const { init } = require('@sentry/electron');
 *
 * init({
 *   dsn: '__DSN__',
 *   // ...
 * });
 *
 * @example
 * import { configureScope } from '@sentry/electron';
 * configureScope((scope: Scope) => {
 *   scope.setExtra({ battery: 0.7 });
 *   scope.setTags({ user_mode: 'admin' });
 *   scope.setUser({ id: '4711' });
 * });
 *
 * @example
 * import { addBreadcrumb } from '@sentry/electron';
 * addBreadcrumb({
 *   message: 'My Breadcrumb',
 *   // ...
 * });
 *
 * @example
 * import * as Sentry from '@sentry/electron';
 * Sentry.captureMessage('Hello, world!');
 * Sentry.captureException(new Error('Good bye'));
 * Sentry.captureEvent({
 *   message: 'Manual',
 *   stacktrace: [
 *     // ...
 *   ],
 * });
 *
 * @see ElectronOptions for documentation on configuration options.
 */
function init(options) {
    // Filter out any EmptyIntegrations
    integrations.removeEmptyIntegrations(options);
    getEntryPoint().init(options);
}
/**
 * Call `close()` on the current client, if there is one. See {@link Client.close}.
 *
 * @param timeout Maximum time in ms the client should wait to flush its event queue before shutting down. Omitting this
 * parameter will cause the client to wait until all events are sent before disabling itself.
 * @returns A promise which resolves to `true` if the queue successfully drains before the timeout, or `false` if it
 * doesn't (or if there's no client defined).
 */
function close(timeout) {
    return tslib.__awaiter(this, void 0, void 0, function* () {
        const entryPoint = getEntryPoint();
        if (entryPoint.close) {
            return entryPoint.close(timeout);
        }
        throw new Error('The Electron SDK should be closed from the main process');
    });
}
/**
 * Call `flush()` on the current client, if there is one. See {@link Client.flush}.
 *
 * @param timeout Maximum time in ms the client should wait to flush its event queue. Omitting this parameter will cause
 * the client to wait until all events are sent before resolving the promise.
 * @returns A promise which resolves to `true` if the queue successfully drains before the timeout, or `false` if it
 * doesn't (or if there's no client defined).
 */
function flush(timeout) {
    return tslib.__awaiter(this, void 0, void 0, function* () {
        const entryPoint = getEntryPoint();
        if (entryPoint.flush) {
            return entryPoint.flush(timeout);
        }
        throw new Error('The Electron SDK should be flushed from the main process');
    });
}
/**
 * **Note** This feature is still in beta so there may be breaking changes in future releases.
 *
 * Starts a child process that detects Application Not Responding (ANR) errors.
 *
 * It's important to await on the returned promise before your app code to ensure this code does not run in the ANR
 * child process.
 *
 * ```js
 * import { init, enableMainProcessAnrDetection } from '@sentry/electron';
 *
 * init({ dsn: "__DSN__" });
 *
 * // with ESM + Electron v28+
 * await enableMainProcessAnrDetection({ captureStackTrace: true });
 * runApp();
 *
 * // with CJS
 * enableMainProcessAnrDetection({ captureStackTrace: true }).then(() => {
 *   runApp();
 * });
 * ```
 */
function enableMainProcessAnrDetection(options) {
    const entryPoint = getEntryPoint();
    if (entryPoint.enableMainProcessAnrDetection) {
        return entryPoint.enableMainProcessAnrDetection(options);
    }
    throw new Error('ANR detection should be started in the main process');
}

exports.FunctionToString = core.FunctionToString;
exports.Hub = core.Hub;
exports.InboundFilters = core.InboundFilters;
exports.Scope = core.Scope;
exports.addBreadcrumb = core.addBreadcrumb;
exports.addGlobalEventProcessor = core.addGlobalEventProcessor;
exports.captureEvent = core.captureEvent;
exports.captureException = core.captureException;
exports.captureMessage = core.captureMessage;
exports.configureScope = core.configureScope;
exports.createTransport = core.createTransport;
exports.getActiveTransaction = core.getActiveTransaction;
exports.getCurrentHub = core.getCurrentHub;
exports.getHubFromCarrier = core.getHubFromCarrier;
exports.makeMain = core.makeMain;
exports.setContext = core.setContext;
exports.setExtra = core.setExtra;
exports.setExtras = core.setExtras;
exports.setTag = core.setTag;
exports.setTags = core.setTags;
exports.setUser = core.setUser;
exports.startTransaction = core.startTransaction;
exports.trace = core.trace;
exports.withScope = core.withScope;
Object.defineProperty(exports, 'IPCMode', {
    enumerable: true,
    get: () => mode.IPCMode
});
exports.Integrations = Integrations;
exports.close = close;
exports.enableMainProcessAnrDetection = enableMainProcessAnrDetection;
exports.flush = flush;
exports.init = init;
//# sourceMappingURL=index.js.map
