import { __awaiter } from 'tslib';
import { app, screen } from 'electron';
import { cpus } from 'os';
import { whenAppReady } from '../electron-normalize.js';
import { mergeEvents } from '../../common/merge.js';

const DEFAULT_OPTIONS = {
    cpu: true,
    screen: true,
    memory: true,
    language: true,
};
/** Adds Electron context to events and normalises paths. */
class AdditionalContext {
    constructor(options = {}) {
        this._lazyDeviceContext = {};
        this.name = AdditionalContext.id;
        this._options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), options);
    }
    /** @inheritDoc */
    setupOnce(addGlobalEventProcessor) {
        addGlobalEventProcessor((event) => __awaiter(this, void 0, void 0, function* () { return this._addAdditionalContext(event); }));
        // Some metrics are only available after app ready so we lazily load them
        void whenAppReady.then(() => {
            const { language, screen: screen$1 } = this._options;
            if (language) {
                this._lazyDeviceContext.language = app.getLocale();
            }
            if (screen$1) {
                this._setPrimaryDisplayInfo();
                screen.on('display-metrics-changed', () => {
                    this._setPrimaryDisplayInfo();
                });
            }
        });
    }
    /** Adds additional context to event */
    _addAdditionalContext(event) {
        const device = this._lazyDeviceContext;
        const { memory, cpu } = this._options;
        if (memory) {
            const { total, free } = process.getSystemMemoryInfo();
            device.memory_size = total * 1024;
            device.free_memory = free * 1024;
        }
        if (cpu) {
            const cpuInfo = cpus();
            if (cpuInfo && cpuInfo.length) {
                const firstCpu = cpuInfo[0];
                device.processor_count = cpuInfo.length;
                device.cpu_description = firstCpu.model;
                device.processor_frequency = firstCpu.speed;
                if (app.runningUnderARM64Translation) {
                    device.machine_arch = 'arm64';
                }
            }
        }
        return mergeEvents(event, { contexts: { device } });
    }
    /** Sets the display info */
    _setPrimaryDisplayInfo() {
        const display = screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        this._lazyDeviceContext.screen_density = display.scaleFactor;
        this._lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
}
/** @inheritDoc */
AdditionalContext.id = 'AdditionalContext';

export { AdditionalContext };
//# sourceMappingURL=additional-context.js.map
