/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs.dir;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.cryptomator.cryptofs.VaultConfig;
import org.cryptomator.cryptofs.dir.DirectoryStreamScoped;
import org.cryptomator.cryptofs.dir.Node;
import org.cryptomator.cryptofs.event.ConflictResolutionFailedEvent;
import org.cryptomator.cryptofs.event.ConflictResolvedEvent;
import org.cryptomator.cryptofs.event.FilesystemEvent;
import org.cryptomator.cryptolib.api.Cryptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@DirectoryStreamScoped
class C9rConflictResolver {
    private static final Logger LOG = LoggerFactory.getLogger(C9rConflictResolver.class);
    private final Cryptor cryptor;
    private final byte[] dirId;
    private final int maxC9rFileNameLength;
    private final Path cleartextPath;
    private final int maxCleartextFileNameLength;
    private final Consumer<FilesystemEvent> eventConsumer;

    @Inject
    public C9rConflictResolver(Cryptor cryptor, @Named(value="dirId") String dirId, VaultConfig vaultConfig, Consumer<FilesystemEvent> eventConsumer, @Named(value="cleartextPath") Path cleartextPath) {
        this.cryptor = cryptor;
        this.dirId = dirId.getBytes(StandardCharsets.US_ASCII);
        this.maxC9rFileNameLength = vaultConfig.getShorteningThreshold();
        this.cleartextPath = cleartextPath;
        this.maxCleartextFileNameLength = (this.maxC9rFileNameLength - 4) / 4 * 3 - 16;
        this.eventConsumer = eventConsumer;
    }

    public Stream<Node> process(Node node) {
        Preconditions.checkArgument((node.extractedCiphertext != null ? 1 : 0) != 0, (Object)"Can only resolve conflicts if extractedCiphertext is set");
        Preconditions.checkArgument((node.cleartextName != null ? 1 : 0) != 0, (Object)"Can only resolve conflicts if cleartextName is set");
        String canonicalCiphertextFileName = node.extractedCiphertext + ".c9r";
        if (node.fullCiphertextFileName.equals(canonicalCiphertextFileName)) {
            return Stream.of(node);
        }
        if (node.fullCiphertextFileName.startsWith(".")) {
            LOG.debug("Ignoring hidden file {}", (Object)node.ciphertextPath);
            return Stream.empty();
        }
        try {
            Path canonicalPath = node.ciphertextPath.resolveSibling(canonicalCiphertextFileName);
            return this.resolveConflict(node, canonicalPath);
        }
        catch (IOException e) {
            this.eventConsumer.accept(new ConflictResolutionFailedEvent(this.cleartextPath.resolve(node.cleartextName), node.ciphertextPath, e));
            LOG.error("Failed to resolve conflict for {}", (Object)node.ciphertextPath, (Object)e);
            return Stream.empty();
        }
    }

    Stream<Node> resolveConflict(Node conflicting, Path canonicalPath) throws IOException {
        Path conflictingPath = conflicting.ciphertextPath;
        if (this.resolveConflictTrivially(canonicalPath, conflictingPath)) {
            Node resolved = new Node(canonicalPath);
            resolved.cleartextName = conflicting.cleartextName;
            resolved.extractedCiphertext = conflicting.extractedCiphertext;
            return Stream.of(resolved);
        }
        return this.renameConflictingFile(canonicalPath, conflicting);
    }

    private Stream<Node> renameConflictingFile(Path canonicalPath, Node conflicting) throws IOException {
        assert (Files.exists(canonicalPath, new LinkOption[0]));
        assert (conflicting.fullCiphertextFileName.endsWith(".c9r"));
        assert (conflicting.fullCiphertextFileName.contains(conflicting.extractedCiphertext));
        String cleartext = conflicting.cleartextName;
        int beginOfCleartextExt = cleartext.lastIndexOf(46);
        String cleartextFileExt = beginOfCleartextExt > 0 ? cleartext.substring(beginOfCleartextExt) : "";
        String cleartextBasename = beginOfCleartextExt > 0 ? cleartext.substring(0, beginOfCleartextExt) : cleartext;
        int endOfCiphertext = conflicting.fullCiphertextFileName.indexOf(conflicting.extractedCiphertext) + conflicting.extractedCiphertext.length();
        String originalConflictSuffix = conflicting.fullCiphertextFileName.substring(endOfCiphertext, conflicting.fullCiphertextFileName.length() - ".c9r".length());
        int netCleartext = this.maxCleartextFileNameLength - cleartextFileExt.length();
        String conflictSuffix = originalConflictSuffix.substring(0, Math.min(originalConflictSuffix.length(), netCleartext / 2));
        int conflictSuffixLen = Math.max(4, conflictSuffix.length());
        String lengthRestrictedBasename = cleartextBasename.substring(0, Math.min(cleartextBasename.length(), netCleartext - conflictSuffixLen));
        String alternativeCleartext = lengthRestrictedBasename + conflictSuffix + cleartextFileExt;
        String alternativeCiphertext = this.cryptor.fileNameCryptor().encryptFilename(BaseEncoding.base64Url(), alternativeCleartext, (byte[][])new byte[][]{this.dirId});
        String alternativeCiphertextName = alternativeCiphertext + ".c9r";
        Path alternativePath = canonicalPath.resolveSibling(alternativeCiphertextName);
        for (int i = 1; i < 10 && Files.exists(alternativePath, new LinkOption[0]); ++i) {
            alternativeCleartext = lengthRestrictedBasename + " (" + i + ")" + cleartextFileExt;
            alternativeCiphertext = this.cryptor.fileNameCryptor().encryptFilename(BaseEncoding.base64Url(), alternativeCleartext, (byte[][])new byte[][]{this.dirId});
            alternativeCiphertextName = alternativeCiphertext + ".c9r";
            alternativePath = canonicalPath.resolveSibling(alternativeCiphertextName);
        }
        assert (alternativeCiphertextName.length() <= this.maxC9rFileNameLength);
        if (Files.exists(alternativePath, new LinkOption[0])) {
            LOG.warn("Failed finding alternative name for {}. Keeping original name.", (Object)conflicting.ciphertextPath);
            return Stream.empty();
        }
        Files.move(conflicting.ciphertextPath, alternativePath, StandardCopyOption.ATOMIC_MOVE);
        LOG.info("Renamed conflicting file {} to {}...", (Object)conflicting.ciphertextPath, (Object)alternativePath);
        Node node = new Node(alternativePath);
        node.cleartextName = alternativeCleartext;
        node.extractedCiphertext = alternativeCiphertext;
        this.eventConsumer.accept(new ConflictResolvedEvent(this.cleartextPath.resolve(cleartext), conflicting.ciphertextPath, this.cleartextPath.resolve(alternativeCleartext), alternativePath));
        return Stream.of(node);
    }

    private boolean resolveConflictTrivially(Path canonicalPath, Path conflictingPath) throws IOException {
        if (!Files.exists(canonicalPath, new LinkOption[0])) {
            Files.move(conflictingPath, canonicalPath, new CopyOption[0]);
            return true;
        }
        if (this.hasSameFileContent(conflictingPath.resolve("dir.c9r"), canonicalPath.resolve("dir.c9r"), 36)) {
            LOG.info("Removing conflicting directory {} (identical to {})", (Object)conflictingPath, (Object)canonicalPath);
            MoreFiles.deleteRecursively((Path)conflictingPath, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            return true;
        }
        if (this.hasSameFileContent(conflictingPath.resolve("symlink.c9r"), canonicalPath.resolve("symlink.c9r"), Short.MAX_VALUE)) {
            LOG.info("Removing conflicting symlink {} (identical to {})", (Object)conflictingPath, (Object)canonicalPath);
            MoreFiles.deleteRecursively((Path)conflictingPath, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            return true;
        }
        return false;
    }

    private boolean hasSameFileContent(Path conflictingPath, Path canonicalPath, int numBytesToCompare) throws IOException {
        if (!Files.isDirectory(conflictingPath.getParent(), new LinkOption[0]) || !Files.isDirectory(canonicalPath.getParent(), new LinkOption[0])) {
            return false;
        }
        try {
            return -1L == Files.mismatch(conflictingPath, canonicalPath);
        }
        catch (NoSuchFileException e) {
            return false;
        }
    }
}

