/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.common.vaults;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.ResourceBundle;
import javafx.collections.ListChangeListener;
import javafx.collections.ObservableList;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.lang3.SystemUtils;
import org.cryptomator.common.recovery.BackupRestorer;
import org.cryptomator.common.settings.Settings;
import org.cryptomator.common.settings.VaultSettings;
import org.cryptomator.common.vaults.AutoLocker;
import org.cryptomator.common.vaults.Vault;
import org.cryptomator.common.vaults.VaultComponent;
import org.cryptomator.common.vaults.VaultConfigCache;
import org.cryptomator.common.vaults.VaultListChangeListener;
import org.cryptomator.common.vaults.VaultState;
import org.cryptomator.cryptofs.CryptoFileSystemProvider;
import org.cryptomator.cryptofs.DirStructure;
import org.cryptomator.cryptofs.migration.Migrators;
import org.cryptomator.integrations.mount.MountService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class VaultListManager {
    private static final Logger LOG = LoggerFactory.getLogger(VaultListManager.class);
    private final AutoLocker autoLocker;
    private final List<MountService> mountServices;
    private final VaultComponent.Factory vaultComponentFactory;
    private final ObservableList<Vault> vaultList;
    private final String defaultVaultName;

    @Inject
    public VaultListManager(ObservableList<Vault> vaultList, AutoLocker autoLocker, List<MountService> mountServices, VaultComponent.Factory vaultComponentFactory, ResourceBundle resourceBundle, Settings settings) {
        this.vaultList = vaultList;
        this.autoLocker = autoLocker;
        this.mountServices = mountServices;
        this.vaultComponentFactory = vaultComponentFactory;
        this.defaultVaultName = resourceBundle.getString("defaults.vault.vaultName");
        this.addAll((Collection<VaultSettings>)settings.directories);
        vaultList.addListener((ListChangeListener)new VaultListChangeListener(settings.directories));
        autoLocker.init();
    }

    public boolean isAlreadyAdded(Path vaultPath) {
        assert (vaultPath.isAbsolute());
        assert (vaultPath.normalize().equals(vaultPath));
        return this.vaultList.stream().anyMatch(v -> vaultPath.equals(v.getPath()));
    }

    public Vault add(Path pathToVault) throws IOException {
        Path normalizedPathToVault = pathToVault.normalize().toAbsolutePath();
        if (CryptoFileSystemProvider.checkDirStructureForVault((Path)normalizedPathToVault, (String)"vault.cryptomator", (String)"masterkey.cryptomator") == DirStructure.UNRELATED) {
            throw new NoSuchFileException(normalizedPathToVault.toString(), null, "Not a vault directory");
        }
        return this.get(normalizedPathToVault).orElseGet(() -> {
            Vault newVault = this.create(this.newVaultSettings(normalizedPathToVault));
            this.vaultList.add((Object)newVault);
            return newVault;
        });
    }

    private VaultSettings newVaultSettings(Path path) {
        VaultSettings vaultSettings = VaultSettings.withRandomId();
        vaultSettings.path.set((Object)path);
        if (path.getFileName() != null) {
            vaultSettings.displayName.set((Object)path.getFileName().toString());
        } else {
            vaultSettings.displayName.set((Object)this.defaultVaultName);
        }
        String nameOfWinfspLocalMounter = "org.cryptomator.frontend.fuse.mount.WinFspMountProvider";
        if (SystemUtils.IS_OS_WINDOWS && ((Path)vaultSettings.path.get()).toString().contains("Dropbox") && this.mountServices.stream().anyMatch(s -> s.getClass().getName().equals(nameOfWinfspLocalMounter))) {
            vaultSettings.mountService.setValue(nameOfWinfspLocalMounter);
        }
        return vaultSettings;
    }

    private void addAll(Collection<VaultSettings> vaultSettings) {
        List<Vault> vaults = vaultSettings.stream().map(this::create).toList();
        this.vaultList.addAll(vaults);
    }

    public Optional<Vault> get(Path vaultPath) {
        assert (vaultPath.isAbsolute());
        assert (vaultPath.normalize().equals(vaultPath));
        return this.vaultList.stream().filter(v -> vaultPath.equals(v.getPath())).findAny();
    }

    public void addVault(Vault vault) {
        Path path = vault.getPath().normalize().toAbsolutePath();
        if (!this.isAlreadyAdded(path)) {
            this.vaultList.add((Object)vault);
        }
    }

    private Vault create(VaultSettings vaultSettings) {
        VaultConfigCache wrapper = new VaultConfigCache(vaultSettings);
        try {
            VaultState.Value vaultState = VaultListManager.determineVaultState((Path)vaultSettings.path.get());
            this.initializeLastKnownKeyLoaderIfPossible(vaultSettings, vaultState, wrapper);
            return this.vaultComponentFactory.create(vaultSettings, wrapper, vaultState, null).vault();
        }
        catch (IOException e) {
            LOG.warn("Failed to determine vault state for {}", vaultSettings.path.get(), (Object)e);
            return this.vaultComponentFactory.create(vaultSettings, wrapper, VaultState.Value.ERROR, e).vault();
        }
    }

    private void initializeLastKnownKeyLoaderIfPossible(VaultSettings vaultSettings, VaultState.Value vaultState, VaultConfigCache wrapper) throws IOException {
        if (vaultSettings.lastKnownKeyLoader.get() != null) {
            return;
        }
        switch (vaultState) {
            case LOCKED: {
                wrapper.reloadConfig();
                vaultSettings.lastKnownKeyLoader.set((Object)wrapper.get().getKeyId().getScheme());
                break;
            }
            case NEEDS_MIGRATION: {
                vaultSettings.lastKnownKeyLoader.set((Object)"masterkeyfile");
                break;
            }
            case VAULT_CONFIG_MISSING: {
                break;
            }
            case MISSING: 
            case ALL_MISSING: 
            case ERROR: 
            case PROCESSING: {
                break;
            }
            default: {
                if (!Files.exists(((Path)vaultSettings.path.get()).resolve("vault.cryptomator"), new LinkOption[0])) break;
                try {
                    wrapper.reloadConfig();
                    vaultSettings.lastKnownKeyLoader.set((Object)wrapper.get().getKeyId().getScheme());
                    break;
                }
                catch (IOException e) {
                    LOG.debug("Unable to load config for {}", vaultSettings.path.get(), (Object)e);
                }
            }
        }
    }

    public static VaultState.Value redetermineVaultState(Vault vault) {
        VaultState state = vault.stateProperty();
        VaultState.Value previous = state.getValue();
        if (previous.equals((Object)VaultState.Value.UNLOCKED) || previous.equals((Object)VaultState.Value.PROCESSING)) {
            return previous;
        }
        try {
            VaultState.Value determined = VaultListManager.determineVaultState(vault.getPath());
            if (determined == VaultState.Value.LOCKED) {
                vault.getVaultConfigCache().reloadConfig();
            }
            state.set(determined);
            return determined;
        }
        catch (IOException e) {
            LOG.warn("Failed to (re)determine vault state for {}", (Object)vault.getPath(), (Object)e);
            vault.setLastKnownException(e);
            state.set(VaultState.Value.ERROR);
            return VaultState.Value.ERROR;
        }
    }

    public static VaultState.Value determineVaultState(Path pathToVault) throws IOException {
        boolean hasConfig;
        if (!Files.exists(pathToVault, new LinkOption[0])) {
            return VaultState.Value.MISSING;
        }
        VaultState.Value structureResult = VaultListManager.checkDirStructure(pathToVault);
        if (structureResult == VaultState.Value.LOCKED || structureResult == VaultState.Value.NEEDS_MIGRATION) {
            return structureResult;
        }
        Path pathToVaultConfig = pathToVault.resolve("vault.cryptomator");
        Path pathToMasterkey = pathToVault.resolve("masterkey.cryptomator");
        if (!Files.exists(pathToVaultConfig, new LinkOption[0])) {
            BackupRestorer.restoreIfBackupPresent(pathToVault, "vault.cryptomator");
        }
        if (!Files.exists(pathToMasterkey, new LinkOption[0])) {
            BackupRestorer.restoreIfBackupPresent(pathToVault, "masterkey.cryptomator");
        }
        if (!(hasConfig = Files.exists(pathToVaultConfig, new LinkOption[0])) && !Files.exists(pathToMasterkey, new LinkOption[0])) {
            return VaultState.Value.ALL_MISSING;
        }
        if (!hasConfig) {
            return VaultState.Value.VAULT_CONFIG_MISSING;
        }
        return VaultListManager.checkDirStructure(pathToVault);
    }

    private static VaultState.Value checkDirStructure(Path pathToVault) throws IOException {
        return switch (CryptoFileSystemProvider.checkDirStructureForVault((Path)pathToVault, (String)"vault.cryptomator", (String)"masterkey.cryptomator")) {
            default -> throw new MatchException(null, null);
            case DirStructure.VAULT -> VaultState.Value.LOCKED;
            case DirStructure.UNRELATED -> VaultState.Value.MISSING;
            case DirStructure.MAYBE_LEGACY -> Migrators.get().needsMigration(pathToVault, "vault.cryptomator", "masterkey.cryptomator") ? VaultState.Value.NEEDS_MIGRATION : VaultState.Value.MISSING;
        };
    }
}

