/*
    File: write_object.cc
*/

/*
Copyright (c) 2014, Christian E. Schafmeister

CLASP is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

See directory 'clasp/licenses' for full details.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
/* -^- */
/* This is copied from ECL write_object.c and modified for C++ */

// #define DEBUG_LEVEL_FULL

#include <clasp/core/foundation.h>
#include <clasp/core/object.h>
#include <clasp/core/cons.h>
#include <clasp/core/symbolTable.h>
#include <clasp/core/array.h>
#include <clasp/core/designators.h>
#include <clasp/core/predicates.h>
#include <clasp/core/lispStream.h>
#include <clasp/core/hashTableEq.h>
#include <clasp/core/arguments.h>
#include <clasp/core/evaluator.h>
#include <clasp/core/write_ugly.h>
#include <clasp/core/print.h>

#include <clasp/core/character.h>

#include <clasp/core/wrappers.h>

namespace core {

bool will_print_as_hash(T_sp x) {
  T_sp circle_counter = _sym_STARcircle_counterSTAR->symbolValue();
  HashTable_sp circle_stack = gc::As<HashTable_sp>(_sym_STARcircle_stackSTAR->symbolValue());
  T_sp code = circle_stack->gethash(x, unbound<T_O>());
  if (circle_counter.fixnump()) {
    return !(code.unboundp() || code.nilp());
  } else if (code.unboundp()) {
    /* Was not found before */
    circle_stack->hash_table_setf_gethash(x, nil<T_O>());
    return 0;
  } else {
    return 1;
  }
}

/* To print circular structures, we traverse the structure by adding
   a pair <element, flag> to the interpreter stack for each element visited.
   flag is initially NIL and becomes T if the element is visited again.
   After the visit we squeeze out all the non circular elements.
   The flags is used during printing to distinguish between the first visit
   to the element.
*/

Fixnum search_print_circle(T_sp x) {
  //        printf("%s:%d Entered search_print_circle with x.px=%p\n", __FILE__, __LINE__, x.raw_());
  T_sp circle_counter = _sym_STARcircle_counterSTAR->symbolValue();
  HashTable_sp circle_stack = gc::As<HashTable_sp>(_sym_STARcircle_stackSTAR->symbolValue());
  T_sp code;
  if (!circle_counter.fixnump()) {
    code = circle_stack->gethash(x, unbound<T_O>());
    if (code.unboundp()) {
      /* Was not found before */
      circle_stack->hash_table_setf_gethash(x, nil<T_O>());
      return 0;
    } else if (code.nilp()) {
      /* This object is referenced twice */
      circle_stack->hash_table_setf_gethash(x, _lisp->_true());
      return 1;
    } else {
      return 2;
    }
  } else {
    code = circle_stack->gethash(x, unbound<T_O>());
    if (code.unboundp() || code.nilp()) {
      /* Is not referenced or was not found before */
      /* _ecl__sethash(x, circle_stack, ECL_NIL); */
      return 0;
    } else if (code == _lisp->_true()) {
      /* This object is referenced twice, but has no code yet */
      ASSERT(circle_counter.fixnump());
      Fixnum new_code = circle_counter.unsafe_fixnum() + 1;
      circle_counter = gc::make_tagged_fixnum<T_O>(new_code);
      circle_stack->hash_table_setf_gethash(x, circle_counter);
      _sym_STARcircle_counterSTAR->setf_symbolValue(circle_counter);
      return -new_code;
    } else {
      return unbox_fixnum(gc::As<Fixnum_sp>(code));
    }
  }
}

T_sp do_write_object(T_sp x, T_sp stream) {
  return cl::_sym_printObject->fboundp() ? core::eval::funcall(cl::_sym_printObject, x, stream) : write_ugly_object(x, stream);
}

T_sp do_write_object_circle(T_sp x, T_sp stream) {
  T_sp circle_counter = _sym_STARcircle_counterSTAR->symbolValue();
  Fixnum code = search_print_circle(x);

  if (!circle_counter.fixnump()) {
    /* We are only inspecting the object to be printed. */
    /* Only run X if it was not referenced before */
    if (code != 0)
      return x;
  } else if (code == 0) {
    /* Object is not referenced twice */
  } else if (code < 0) {
    /* Object is referenced twice. We print its definition */
    stringstream ss;
    ss << '#' << -code << '=';
    cl__write_string(SimpleBaseString_O::make(ss.str()), stream);
  } else {
    /* Second reference to the object */
    stringstream ss;
    ss << '#' << code << '#';
    cl__write_string(SimpleBaseString_O::make(ss.str()), stream);
    return x;
  }

  return do_write_object(x, stream);
}

T_sp write_object(T_sp x, T_sp stream) {
  // With *print-pretty*, go immediately to the pretty printer, which does its own *print-circle* etc.
  if (!cl::_sym_STARprint_prettySTAR.unboundp() && cl::_sym_STARprint_prettySTAR->boundP() &&
      cl::_sym_STARprint_prettySTAR->symbolValue().notnilp()) {
    T_mv mv_f = eval::funcall(cl::_sym_pprint_dispatch, x);
    T_sp f0 = mv_f;
    MultipleValues& mvn = core::lisp_multipleValues();
    T_sp f1 = mvn.valueGet(1, mv_f.number_of_values());
    if (f1.notnilp()) {
      eval::funcall(f0, stream, x);
      return x;
    }
  }

  // Otherwise, check print circle stuff...
  bool circle = clasp_print_circle();
  // We only worry about *print-circle* for objects that aren't numbers, valists, characters,
  // or interned symbols.
  if (circle && (x) && !x.fixnump() && !x.valistp() && !x.characterp() && !cl__numberp(x) &&
      (!cl__symbolp(x) || gc::As<Symbol_sp>(x)->homePackage().nilp())) {
    T_sp circle_counter = _sym_STARcircle_counterSTAR->symbolValue();

    if (circle_counter.nilp()) {
      HashTable_sp hash = HashTable_O::createEq(1024);
      DynamicScopeManager scope(_sym_STARcircle_counterSTAR, _lisp->_true());
      DynamicScopeManager scope2(_sym_STARcircle_stackSTAR, hash);
      do_write_object_circle(x, _lisp->nullStream());
      _sym_STARcircle_counterSTAR->setf_symbolValue(gc::make_tagged_fixnum<core::Fixnum_I>(0));
      return do_write_object_circle(x, stream);
    }

    return do_write_object_circle(x, stream);
  }

  return do_write_object(x, stream);
}

CL_LAMBDA(obj &optional strm);
CL_DECLARE();
CL_DOCSTRING(R"dx(Identical to WRITE, but doesn't bind printer control variables.)dx");
DOCGROUP(clasp);
CL_DEFUN T_sp core__write_object(T_sp obj, T_sp ostrm) {
  T_sp strm = coerce::outputStreamDesignator(ostrm);
  return write_object(obj, strm);
};
}; // namespace core
