/*
 * Decompiled with CFR 0.152.
 */
package libcore.util;

import android.system.ErrnoException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import libcore.io.BufferIterator;
import libcore.io.MemoryMappedFile;
import libcore.util.BasicLruCache;
import libcore.util.TimeZoneDataFiles;
import libcore.util.ZoneInfo;

public final class ZoneInfoDB {
    public static final String TZDATA_FILE = "tzdata";
    private static final TzData DATA = TzData.loadTzDataWithFallback(TimeZoneDataFiles.getTimeZoneFilePaths("tzdata"));

    private ZoneInfoDB() {
    }

    public static TzData getInstance() {
        return DATA;
    }

    public static class TzData {
        private static final int SIZEOF_TZNAME = 40;
        private static final int SIZEOF_TZINT = 4;
        public static final int SIZEOF_INDEX_ENTRY = 52;
        private boolean closed;
        private MemoryMappedFile mappedFile;
        private String version;
        private String zoneTab;
        private String[] ids;
        private int[] byteOffsets;
        private int[] rawUtcOffsetsCache;
        private static final int CACHE_SIZE = 1;
        private final BasicLruCache<String, ZoneInfo> cache = new BasicLruCache<String, ZoneInfo>(1){

            @Override
            protected ZoneInfo create(String id) {
                try {
                    return this.makeTimeZoneUncached(id);
                }
                catch (IOException e) {
                    throw new IllegalStateException("Unable to load timezone for ID=" + id, e);
                }
            }
        };

        public static TzData loadTzDataWithFallback(String ... paths) {
            String[] stringArray = paths;
            int n = stringArray.length;
            for (int i = 0; i < n; ++i) {
                TzData tzData = new TzData();
                String path = stringArray[i];
                if (!tzData.loadData(path)) continue;
                return tzData;
            }
            System.logE("Couldn't find any tzdata file!");
            return TzData.createFallback();
        }

        public static TzData loadTzData(String path) {
            TzData tzData = new TzData();
            if (tzData.loadData(path)) {
                return tzData;
            }
            return null;
        }

        private static TzData createFallback() {
            TzData tzData = new TzData();
            tzData.populateFallback();
            return tzData;
        }

        private TzData() {
        }

        public BufferIterator getBufferIterator(String id) {
            this.checkNotClosed();
            int index = Arrays.binarySearch(this.ids, id);
            if (index < 0) {
                return null;
            }
            int byteOffset = this.byteOffsets[index];
            BufferIterator it = this.mappedFile.bigEndianIterator();
            it.skip(byteOffset);
            return it;
        }

        private void populateFallback() {
            this.version = "missing";
            this.zoneTab = "# Emergency fallback data.\n";
            this.ids = new String[]{"GMT"};
            this.rawUtcOffsetsCache = new int[1];
            this.byteOffsets = this.rawUtcOffsetsCache;
        }

        private boolean loadData(String path) {
            try {
                this.mappedFile = MemoryMappedFile.mmapRO(path);
            }
            catch (ErrnoException errnoException) {
                return false;
            }
            try {
                this.readHeader();
                return true;
            }
            catch (Exception ex) {
                this.close();
                System.logE("tzdata file \"" + path + "\" was present but invalid!", ex);
                return false;
            }
        }

        private void readHeader() throws IOException {
            BufferIterator it = this.mappedFile.bigEndianIterator();
            try {
                byte[] tzdata_version = new byte[12];
                it.readByteArray(tzdata_version, 0, tzdata_version.length);
                String magic = new String(tzdata_version, 0, 6, StandardCharsets.US_ASCII);
                if (!magic.equals(ZoneInfoDB.TZDATA_FILE) || tzdata_version[11] != 0) {
                    throw new IOException("bad tzdata magic: " + Arrays.toString(tzdata_version));
                }
                this.version = new String(tzdata_version, 6, 5, StandardCharsets.US_ASCII);
                int fileSize = this.mappedFile.size();
                int index_offset = it.readInt();
                TzData.validateOffset(index_offset, fileSize);
                int data_offset = it.readInt();
                TzData.validateOffset(data_offset, fileSize);
                int zonetab_offset = it.readInt();
                TzData.validateOffset(zonetab_offset, fileSize);
                if (index_offset >= data_offset || data_offset >= zonetab_offset) {
                    throw new IOException("Invalid offset: index_offset=" + index_offset + ", data_offset=" + data_offset + ", zonetab_offset=" + zonetab_offset + ", fileSize=" + fileSize);
                }
                this.readIndex(it, index_offset, data_offset);
                this.readZoneTab(it, zonetab_offset, fileSize - zonetab_offset);
            }
            catch (IndexOutOfBoundsException e) {
                throw new IOException("Invalid read from data file", e);
            }
        }

        private static void validateOffset(int offset, int size) throws IOException {
            if (offset < 0 || offset >= size) {
                throw new IOException("Invalid offset=" + offset + ", size=" + size);
            }
        }

        private void readZoneTab(BufferIterator it, int zoneTabOffset, int zoneTabSize) {
            byte[] bytes = new byte[zoneTabSize];
            it.seek(zoneTabOffset);
            it.readByteArray(bytes, 0, bytes.length);
            this.zoneTab = new String(bytes, 0, bytes.length, StandardCharsets.US_ASCII);
        }

        private void readIndex(BufferIterator it, int indexOffset, int dataOffset) throws IOException {
            it.seek(indexOffset);
            byte[] idBytes = new byte[40];
            int indexSize = dataOffset - indexOffset;
            if (indexSize % 52 != 0) {
                throw new IOException("Index size is not divisible by 52, indexSize=" + indexSize);
            }
            int entryCount = indexSize / 52;
            this.byteOffsets = new int[entryCount];
            this.ids = new String[entryCount];
            for (int i = 0; i < entryCount; ++i) {
                int len;
                it.readByteArray(idBytes, 0, idBytes.length);
                this.byteOffsets[i] = it.readInt();
                int n = i;
                this.byteOffsets[n] = this.byteOffsets[n] + dataOffset;
                int length = it.readInt();
                if (length < 44) {
                    throw new IOException("length in index file < sizeof(tzhead)");
                }
                it.skip(4);
                for (len = 0; idBytes[len] != 0 && len < idBytes.length; ++len) {
                }
                if (len == 0) {
                    throw new IOException("Invalid ID at index=" + i);
                }
                this.ids[i] = new String(idBytes, 0, len, StandardCharsets.US_ASCII);
                if (i <= 0 || this.ids[i].compareTo(this.ids[i - 1]) > 0) continue;
                throw new IOException("Index not sorted or contains multiple entries with the same ID, index=" + i + ", ids[i]=" + this.ids[i] + ", ids[i - 1]=" + this.ids[i - 1]);
            }
        }

        public void validate() throws IOException {
            this.checkNotClosed();
            for (String id : this.getAvailableIDs()) {
                ZoneInfo zoneInfo = this.makeTimeZoneUncached(id);
                if (zoneInfo != null) continue;
                throw new IOException("Unable to find data for ID=" + id);
            }
        }

        ZoneInfo makeTimeZoneUncached(String id) throws IOException {
            BufferIterator it = this.getBufferIterator(id);
            if (it == null) {
                return null;
            }
            return ZoneInfo.readTimeZone(id, it, System.currentTimeMillis());
        }

        public String[] getAvailableIDs() {
            this.checkNotClosed();
            return (String[])this.ids.clone();
        }

        public String[] getAvailableIDs(int rawUtcOffset) {
            this.checkNotClosed();
            ArrayList<String> matches = new ArrayList<String>();
            int[] rawUtcOffsets = this.getRawUtcOffsets();
            for (int i = 0; i < rawUtcOffsets.length; ++i) {
                if (rawUtcOffsets[i] != rawUtcOffset) continue;
                matches.add(this.ids[i]);
            }
            return matches.toArray(new String[matches.size()]);
        }

        private synchronized int[] getRawUtcOffsets() {
            if (this.rawUtcOffsetsCache != null) {
                return this.rawUtcOffsetsCache;
            }
            this.rawUtcOffsetsCache = new int[this.ids.length];
            for (int i = 0; i < this.ids.length; ++i) {
                this.rawUtcOffsetsCache[i] = this.cache.get(this.ids[i]).getRawOffset();
            }
            return this.rawUtcOffsetsCache;
        }

        public String getVersion() {
            this.checkNotClosed();
            return this.version;
        }

        public String getZoneTab() {
            this.checkNotClosed();
            return this.zoneTab;
        }

        public ZoneInfo makeTimeZone(String id) throws IOException {
            this.checkNotClosed();
            ZoneInfo zoneInfo = this.cache.get(id);
            return zoneInfo == null ? null : (ZoneInfo)zoneInfo.clone();
        }

        public boolean hasTimeZone(String id) throws IOException {
            this.checkNotClosed();
            return this.cache.get(id) != null;
        }

        public void close() {
            if (!this.closed) {
                this.closed = true;
                this.ids = null;
                this.byteOffsets = null;
                this.rawUtcOffsetsCache = null;
                this.mappedFile = null;
                this.cache.evictAll();
                if (this.mappedFile != null) {
                    try {
                        this.mappedFile.close();
                    }
                    catch (ErrnoException errnoException) {
                        // empty catch block
                    }
                }
            }
        }

        private void checkNotClosed() throws IllegalStateException {
            if (this.closed) {
                throw new IllegalStateException("TzData is closed");
            }
        }

        protected void finalize() throws Throwable {
            try {
                this.close();
            }
            finally {
                super.finalize();
            }
        }

        public static String getRulesVersion(File tzDataFile) throws IOException {
            try (FileInputStream is = new FileInputStream(tzDataFile);){
                int bytesToRead = 12;
                byte[] tzdataVersion = new byte[12];
                int bytesRead = is.read(tzdataVersion, 0, 12);
                if (bytesRead != 12) {
                    throw new IOException("File too short: only able to read " + bytesRead + " bytes.");
                }
                String magic = new String(tzdataVersion, 0, 6, StandardCharsets.US_ASCII);
                if (!magic.equals(ZoneInfoDB.TZDATA_FILE) || tzdataVersion[11] != 0) {
                    throw new IOException("bad tzdata magic: " + Arrays.toString(tzdataVersion));
                }
                String string = new String(tzdataVersion, 6, 5, StandardCharsets.US_ASCII);
                return string;
            }
        }
    }
}

