/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Arrays;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.MathUtil;
import org.apache.lucene.util.bkd.BKDConfig;
import org.apache.lucene.util.bkd.DocIdsWriter;

public final class BKDReader
extends PointValues {
    final int leafNodeOffset;
    final BKDConfig config;
    final int numLeaves;
    final IndexInput in;
    final byte[] minPackedValue;
    final byte[] maxPackedValue;
    final long pointCount;
    final int docCount;
    final int version;
    final long minLeafBlockFP;
    final IndexInput packedIndex;

    public BKDReader(IndexInput metaIn, IndexInput indexIn, IndexInput dataIn) throws IOException {
        long indexStartPointer;
        this.version = CodecUtil.checkHeader(metaIn, "BKD", 4, 9);
        int numDims = metaIn.readVInt();
        int numIndexDims = this.version >= 6 ? metaIn.readVInt() : numDims;
        int maxPointsInLeafNode = metaIn.readVInt();
        int bytesPerDim = metaIn.readVInt();
        this.config = new BKDConfig(numDims, numIndexDims, bytesPerDim, maxPointsInLeafNode);
        this.numLeaves = metaIn.readVInt();
        assert (this.numLeaves > 0);
        this.leafNodeOffset = this.numLeaves;
        this.minPackedValue = new byte[this.config.packedIndexBytesLength];
        this.maxPackedValue = new byte[this.config.packedIndexBytesLength];
        metaIn.readBytes(this.minPackedValue, 0, this.config.packedIndexBytesLength);
        metaIn.readBytes(this.maxPackedValue, 0, this.config.packedIndexBytesLength);
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            if (Arrays.compareUnsigned(this.minPackedValue, dim * this.config.bytesPerDim, dim * this.config.bytesPerDim + this.config.bytesPerDim, this.maxPackedValue, dim * this.config.bytesPerDim, dim * this.config.bytesPerDim + this.config.bytesPerDim) <= 0) continue;
            throw new CorruptIndexException("minPackedValue " + new BytesRef(this.minPackedValue) + " is > maxPackedValue " + new BytesRef(this.maxPackedValue) + " for dim=" + dim, metaIn);
        }
        this.pointCount = metaIn.readVLong();
        this.docCount = metaIn.readVInt();
        int numIndexBytes = metaIn.readVInt();
        if (this.version >= 9) {
            this.minLeafBlockFP = metaIn.readLong();
            indexStartPointer = metaIn.readLong();
        } else {
            indexStartPointer = indexIn.getFilePointer();
            this.minLeafBlockFP = indexIn.readVLong();
            indexIn.seek(indexStartPointer);
        }
        this.packedIndex = indexIn.slice("packedIndex", indexStartPointer, numIndexBytes);
        this.in = dataIn;
    }

    long getMinLeafBlockFP() {
        return this.minLeafBlockFP;
    }

    private int getTreeDepth() {
        return MathUtil.log(this.numLeaves, 2) + 2;
    }

    @Override
    public void intersect(PointValues.IntersectVisitor visitor) throws IOException {
        this.intersect(this.getIntersectState(visitor), this.minPackedValue, this.maxPackedValue);
    }

    @Override
    public long estimatePointCount(PointValues.IntersectVisitor visitor) {
        return this.estimatePointCount(this.getIntersectState(visitor), this.minPackedValue, this.maxPackedValue);
    }

    private void addAll(IntersectState state, boolean grown) throws IOException {
        long maxPointCount;
        if (!grown && (maxPointCount = (long)this.config.maxPointsInLeafNode * (long)state.index.getNumLeaves()) <= Integer.MAX_VALUE) {
            state.visitor.grow((int)maxPointCount);
            grown = true;
        }
        if (state.index.isLeafNode()) {
            assert (grown);
            if (state.index.nodeExists()) {
                this.visitDocIDs(state.in, state.index.getLeafBlockFP(), state.visitor);
            }
        } else {
            state.index.pushLeft();
            this.addAll(state, grown);
            state.index.pop();
            state.index.pushRight();
            this.addAll(state, grown);
            state.index.pop();
        }
    }

    public IntersectState getIntersectState(PointValues.IntersectVisitor visitor) {
        IndexTree index = new IndexTree();
        return new IntersectState(this.in.clone(), this.config, visitor, index);
    }

    public void visitLeafBlockValues(IndexTree index, IntersectState state) throws IOException {
        int count = this.readDocIDs(state.in, index.getLeafBlockFP(), state.scratchIterator);
        this.visitDocValues(state.commonPrefixLengths, state.scratchDataPackedValue, state.scratchMinIndexPackedValue, state.scratchMaxIndexPackedValue, state.in, state.scratchIterator, count, state.visitor);
    }

    private void visitDocIDs(IndexInput in, long blockFP, PointValues.IntersectVisitor visitor) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        DocIdsWriter.readInts(in, count, visitor);
    }

    int readDocIDs(IndexInput in, long blockFP, BKDReaderDocIDSetIterator iterator) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        DocIdsWriter.readInts(in, count, iterator.docIDs);
        return count;
    }

    void visitDocValues(int[] commonPrefixLengths, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
        if (this.version >= 7) {
            this.visitDocValuesWithCardinality(commonPrefixLengths, scratchDataPackedValue, scratchMinIndexPackedValue, scratchMaxIndexPackedValue, in, scratchIterator, count, visitor);
        } else {
            this.visitDocValuesNoCardinality(commonPrefixLengths, scratchDataPackedValue, scratchMinIndexPackedValue, scratchMaxIndexPackedValue, in, scratchIterator, count, visitor);
        }
    }

    void visitDocValuesNoCardinality(int[] commonPrefixLengths, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
        this.readCommonPrefixes(commonPrefixLengths, scratchDataPackedValue, in);
        if (this.config.numIndexDims != 1 && this.version >= 5) {
            byte[] minPackedValue = scratchMinIndexPackedValue;
            System.arraycopy(scratchDataPackedValue, 0, minPackedValue, 0, this.config.packedIndexBytesLength);
            byte[] maxPackedValue = scratchMaxIndexPackedValue;
            System.arraycopy(minPackedValue, 0, maxPackedValue, 0, this.config.packedIndexBytesLength);
            this.readMinMax(commonPrefixLengths, minPackedValue, maxPackedValue, in);
            PointValues.Relation r = visitor.compare(minPackedValue, maxPackedValue);
            if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
                return;
            }
            visitor.grow(count);
            if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                for (int i = 0; i < count; ++i) {
                    visitor.visit(scratchIterator.docIDs[i]);
                }
                return;
            }
        } else {
            visitor.grow(count);
        }
        int compressedDim = this.readCompressedDim(in);
        if (compressedDim == -1) {
            this.visitUniqueRawDocValues(scratchDataPackedValue, scratchIterator, count, visitor);
        } else {
            this.visitCompressedDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor, compressedDim);
        }
    }

    void visitDocValuesWithCardinality(int[] commonPrefixLengths, byte[] scratchDataPackedValue, byte[] scratchMinIndexPackedValue, byte[] scratchMaxIndexPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
        this.readCommonPrefixes(commonPrefixLengths, scratchDataPackedValue, in);
        int compressedDim = this.readCompressedDim(in);
        if (compressedDim == -1) {
            visitor.grow(count);
            this.visitUniqueRawDocValues(scratchDataPackedValue, scratchIterator, count, visitor);
        } else {
            if (this.config.numIndexDims != 1) {
                byte[] minPackedValue = scratchMinIndexPackedValue;
                System.arraycopy(scratchDataPackedValue, 0, minPackedValue, 0, this.config.packedIndexBytesLength);
                byte[] maxPackedValue = scratchMaxIndexPackedValue;
                System.arraycopy(minPackedValue, 0, maxPackedValue, 0, this.config.packedIndexBytesLength);
                this.readMinMax(commonPrefixLengths, minPackedValue, maxPackedValue, in);
                PointValues.Relation r = visitor.compare(minPackedValue, maxPackedValue);
                if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
                    return;
                }
                visitor.grow(count);
                if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                    for (int i = 0; i < count; ++i) {
                        visitor.visit(scratchIterator.docIDs[i]);
                    }
                    return;
                }
            } else {
                visitor.grow(count);
            }
            if (compressedDim == -2) {
                this.visitSparseRawDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor);
            } else {
                this.visitCompressedDocValues(commonPrefixLengths, scratchDataPackedValue, in, scratchIterator, count, visitor, compressedDim);
            }
        }
    }

    private void readMinMax(int[] commonPrefixLengths, byte[] minPackedValue, byte[] maxPackedValue, IndexInput in) throws IOException {
        for (int dim = 0; dim < this.config.numIndexDims; ++dim) {
            int prefix = commonPrefixLengths[dim];
            in.readBytes(minPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
            in.readBytes(maxPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
        }
    }

    private void visitSparseRawDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
        int i;
        int length;
        for (i = 0; i < count; i += length) {
            length = in.readVInt();
            for (int dim = 0; dim < this.config.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
            }
            scratchIterator.reset(i, length);
            visitor.visit(scratchIterator, scratchPackedValue);
        }
        if (i != count) {
            throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
        }
    }

    private void visitUniqueRawDocValues(byte[] scratchPackedValue, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor) throws IOException {
        scratchIterator.reset(0, count);
        visitor.visit(scratchIterator, scratchPackedValue);
    }

    private void visitCompressedDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, BKDReaderDocIDSetIterator scratchIterator, int count, PointValues.IntersectVisitor visitor, int compressedDim) throws IOException {
        int i;
        int runLen;
        int compressedByteOffset = compressedDim * this.config.bytesPerDim + commonPrefixLengths[compressedDim];
        int n = compressedDim;
        commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
        for (i = 0; i < count; i += runLen) {
            scratchPackedValue[compressedByteOffset] = in.readByte();
            runLen = Byte.toUnsignedInt(in.readByte());
            for (int j = 0; j < runLen; ++j) {
                for (int dim = 0; dim < this.config.numDims; ++dim) {
                    int prefix = commonPrefixLengths[dim];
                    in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim + prefix, this.config.bytesPerDim - prefix);
                }
                visitor.visit(scratchIterator.docIDs[i + j], scratchPackedValue);
            }
        }
        if (i != count) {
            throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
        }
    }

    private int readCompressedDim(IndexInput in) throws IOException {
        byte compressedDim = in.readByte();
        if (compressedDim < -2 || compressedDim >= this.config.numDims || this.version < 7 && compressedDim == -2) {
            throw new CorruptIndexException("Got compressedDim=" + compressedDim, in);
        }
        return compressedDim;
    }

    private void readCommonPrefixes(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in) throws IOException {
        for (int dim = 0; dim < this.config.numDims; ++dim) {
            int prefix;
            commonPrefixLengths[dim] = prefix = in.readVInt();
            if (prefix <= 0) continue;
            in.readBytes(scratchPackedValue, dim * this.config.bytesPerDim, prefix);
        }
    }

    private void intersect(IntersectState state, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r != PointValues.Relation.CELL_OUTSIDE_QUERY) {
            if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                this.addAll(state, false);
            } else if (state.index.isLeafNode()) {
                if (state.index.nodeExists()) {
                    int count = this.readDocIDs(state.in, state.index.getLeafBlockFP(), state.scratchIterator);
                    this.visitDocValues(state.commonPrefixLengths, state.scratchDataPackedValue, state.scratchMinIndexPackedValue, state.scratchMaxIndexPackedValue, state.in, state.scratchIterator, count, state.visitor);
                }
            } else {
                int splitDim = state.index.getSplitDim();
                assert (splitDim >= 0) : "splitDim=" + splitDim + ", config.numIndexDims=" + this.config.numIndexDims;
                assert (splitDim < this.config.numIndexDims) : "splitDim=" + splitDim + ", config.numIndexDims=" + this.config.numIndexDims;
                byte[] splitPackedValue = state.index.getSplitPackedValue();
                BytesRef splitDimValue = state.index.getSplitDimValue();
                assert (splitDimValue.length == this.config.bytesPerDim);
                assert (Arrays.compareUnsigned(cellMinPacked, splitDim * this.config.bytesPerDim, splitDim * this.config.bytesPerDim + this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.config.bytesPerDim) <= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDim=" + splitDim + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
                assert (Arrays.compareUnsigned(cellMaxPacked, splitDim * this.config.bytesPerDim, splitDim * this.config.bytesPerDim + this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.config.bytesPerDim) >= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDim=" + splitDim + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
                System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.config.packedIndexBytesLength);
                System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
                state.index.pushLeft();
                this.intersect(state, cellMinPacked, splitPackedValue);
                state.index.pop();
                System.arraycopy(splitPackedValue, splitDim * this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, this.config.bytesPerDim);
                System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.config.packedIndexBytesLength);
                System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
                state.index.pushRight();
                this.intersect(state, splitPackedValue, cellMaxPacked);
                state.index.pop();
            }
        }
    }

    private long estimatePointCount(IntersectState state, byte[] cellMinPacked, byte[] cellMaxPacked) {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
            return 0L;
        }
        if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
            return (long)this.config.maxPointsInLeafNode * (long)state.index.getNumLeaves();
        }
        if (state.index.isLeafNode()) {
            return (this.config.maxPointsInLeafNode + 1) / 2;
        }
        int splitDim = state.index.getSplitDim();
        assert (splitDim >= 0) : "splitDim=" + splitDim + ", config.numIndexDims=" + this.config.numIndexDims;
        assert (splitDim < this.config.numIndexDims) : "splitDim=" + splitDim + ", config.numIndexDims=" + this.config.numIndexDims;
        byte[] splitPackedValue = state.index.getSplitPackedValue();
        BytesRef splitDimValue = state.index.getSplitDimValue();
        assert (splitDimValue.length == this.config.bytesPerDim);
        assert (Arrays.compareUnsigned(cellMinPacked, splitDim * this.config.bytesPerDim, splitDim * this.config.bytesPerDim + this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.config.bytesPerDim) <= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDim=" + splitDim + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
        assert (Arrays.compareUnsigned(cellMaxPacked, splitDim * this.config.bytesPerDim, splitDim * this.config.bytesPerDim + this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.config.bytesPerDim) >= 0) : "config.bytesPerDim=" + this.config.bytesPerDim + " splitDim=" + splitDim + " config.numIndexDims=" + this.config.numIndexDims + " config.numDims=" + this.config.numDims;
        System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.config.packedIndexBytesLength);
        System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
        state.index.pushLeft();
        long leftCost = this.estimatePointCount(state, cellMinPacked, splitPackedValue);
        state.index.pop();
        System.arraycopy(splitPackedValue, splitDim * this.config.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, this.config.bytesPerDim);
        System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.config.packedIndexBytesLength);
        System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.config.bytesPerDim, this.config.bytesPerDim);
        state.index.pushRight();
        long rightCost = this.estimatePointCount(state, splitPackedValue, cellMaxPacked);
        state.index.pop();
        return leftCost + rightCost;
    }

    @Override
    public byte[] getMinPackedValue() {
        return (byte[])this.minPackedValue.clone();
    }

    @Override
    public byte[] getMaxPackedValue() {
        return (byte[])this.maxPackedValue.clone();
    }

    @Override
    public int getNumDimensions() {
        return this.config.numDims;
    }

    @Override
    public int getNumIndexDimensions() {
        return this.config.numIndexDims;
    }

    @Override
    public int getBytesPerDimension() {
        return this.config.bytesPerDim;
    }

    @Override
    public long size() {
        return this.pointCount;
    }

    @Override
    public int getDocCount() {
        return this.docCount;
    }

    public boolean isLeafNode(int nodeID) {
        return nodeID >= this.leafNodeOffset;
    }

    protected static class BKDReaderDocIDSetIterator
    extends DocIdSetIterator {
        private int idx;
        private int length;
        private int offset;
        private int docID;
        final int[] docIDs;

        public BKDReaderDocIDSetIterator(int maxPointsInLeafNode) {
            this.docIDs = new int[maxPointsInLeafNode];
        }

        @Override
        public int docID() {
            return this.docID;
        }

        private void reset(int offset, int length) {
            this.offset = offset;
            this.length = length;
            assert (offset + length <= this.docIDs.length);
            this.docID = -1;
            this.idx = 0;
        }

        @Override
        public int nextDoc() throws IOException {
            if (this.idx == this.length) {
                this.docID = Integer.MAX_VALUE;
            } else {
                this.docID = this.docIDs[this.offset + this.idx];
                ++this.idx;
            }
            return this.docID;
        }

        @Override
        public int advance(int target) throws IOException {
            return this.slowAdvance(target);
        }

        @Override
        public long cost() {
            return this.length;
        }
    }

    public static final class IntersectState {
        final IndexInput in;
        final BKDReaderDocIDSetIterator scratchIterator;
        final byte[] scratchDataPackedValue;
        final byte[] scratchMinIndexPackedValue;
        final byte[] scratchMaxIndexPackedValue;
        final int[] commonPrefixLengths;
        final PointValues.IntersectVisitor visitor;
        public final IndexTree index;

        public IntersectState(IndexInput in, BKDConfig config, PointValues.IntersectVisitor visitor, IndexTree indexVisitor) {
            this.in = in;
            this.visitor = visitor;
            this.commonPrefixLengths = new int[config.numDims];
            this.scratchIterator = new BKDReaderDocIDSetIterator(config.maxPointsInLeafNode);
            this.scratchDataPackedValue = new byte[config.packedBytesLength];
            this.scratchMinIndexPackedValue = new byte[config.packedIndexBytesLength];
            this.scratchMaxIndexPackedValue = new byte[config.packedIndexBytesLength];
            this.index = indexVisitor;
        }
    }

    public class IndexTree
    implements Cloneable {
        private int nodeID;
        private int level;
        private int splitDim;
        private final byte[][] splitPackedValueStack;
        private final IndexInput in;
        private final long[] leafBlockFPStack;
        private final int[] rightNodePositions;
        private final int[] splitDims;
        private final boolean[] negativeDeltas;
        private final byte[][] splitValuesStack;
        private final BytesRef scratch;

        IndexTree() {
            this(this$0.packedIndex.clone(), 1, 1);
            this.readNodeData(false);
        }

        private IndexTree(IndexInput in, int nodeID, int level) {
            int treeDepth = BKDReader.this.getTreeDepth();
            this.splitPackedValueStack = new byte[treeDepth + 1][];
            this.nodeID = nodeID;
            this.level = level;
            this.splitPackedValueStack[level] = new byte[BKDReader.this.config.packedIndexBytesLength];
            this.leafBlockFPStack = new long[treeDepth + 1];
            this.rightNodePositions = new int[treeDepth + 1];
            this.splitValuesStack = new byte[treeDepth + 1][];
            this.splitDims = new int[treeDepth + 1];
            this.negativeDeltas = new boolean[BKDReader.this.config.numIndexDims * (treeDepth + 1)];
            this.in = in;
            this.splitValuesStack[0] = new byte[BKDReader.this.config.packedIndexBytesLength];
            this.scratch = new BytesRef();
            this.scratch.length = BKDReader.this.config.bytesPerDim;
        }

        public void pushLeft() {
            this.nodeID *= 2;
            ++this.level;
            this.readNodeData(true);
        }

        public IndexTree clone() {
            IndexTree index = new IndexTree(this.in.clone(), this.nodeID, this.level);
            index.splitDim = this.splitDim;
            index.leafBlockFPStack[this.level] = this.leafBlockFPStack[this.level];
            index.rightNodePositions[this.level] = this.rightNodePositions[this.level];
            index.splitValuesStack[index.level] = (byte[])this.splitValuesStack[index.level].clone();
            System.arraycopy(this.negativeDeltas, this.level * BKDReader.this.config.numIndexDims, index.negativeDeltas, this.level * BKDReader.this.config.numIndexDims, BKDReader.this.config.numIndexDims);
            index.splitDims[this.level] = this.splitDims[this.level];
            return index;
        }

        public void pushRight() {
            int nodePosition = this.rightNodePositions[this.level];
            assert ((long)nodePosition >= this.in.getFilePointer()) : "nodePosition = " + nodePosition + " < currentPosition=" + this.in.getFilePointer();
            this.nodeID = this.nodeID * 2 + 1;
            ++this.level;
            try {
                this.in.seek(nodePosition);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            this.readNodeData(false);
        }

        public void pop() {
            this.nodeID /= 2;
            --this.level;
            this.splitDim = this.splitDims[this.level];
        }

        public boolean isLeafNode() {
            return this.nodeID >= BKDReader.this.leafNodeOffset;
        }

        public boolean nodeExists() {
            return this.nodeID - BKDReader.this.leafNodeOffset < BKDReader.this.leafNodeOffset;
        }

        public int getNodeID() {
            return this.nodeID;
        }

        public byte[] getSplitPackedValue() {
            assert (!this.isLeafNode());
            assert (this.splitPackedValueStack[this.level] != null) : "level=" + this.level;
            return this.splitPackedValueStack[this.level];
        }

        public int getSplitDim() {
            assert (!this.isLeafNode());
            return this.splitDim;
        }

        public BytesRef getSplitDimValue() {
            assert (!this.isLeafNode());
            this.scratch.bytes = this.splitValuesStack[this.level];
            this.scratch.offset = this.splitDim * BKDReader.this.config.bytesPerDim;
            return this.scratch;
        }

        public long getLeafBlockFP() {
            assert (this.isLeafNode()) : "nodeID=" + this.nodeID + " is not a leaf";
            return this.leafBlockFPStack[this.level];
        }

        public int getNumLeaves() {
            int leftMostLeafNode;
            for (leftMostLeafNode = this.nodeID; leftMostLeafNode < BKDReader.this.leafNodeOffset; leftMostLeafNode *= 2) {
            }
            int rightMostLeafNode = this.nodeID;
            while (rightMostLeafNode < BKDReader.this.leafNodeOffset) {
                rightMostLeafNode = rightMostLeafNode * 2 + 1;
            }
            int numLeaves = rightMostLeafNode >= leftMostLeafNode ? rightMostLeafNode - leftMostLeafNode + 1 : rightMostLeafNode - leftMostLeafNode + 1 + BKDReader.this.leafNodeOffset;
            assert (numLeaves == this.getNumLeavesSlow(this.nodeID)) : numLeaves + " " + this.getNumLeavesSlow(this.nodeID);
            return numLeaves;
        }

        private int getNumLeavesSlow(int node) {
            if (node >= 2 * BKDReader.this.leafNodeOffset) {
                return 0;
            }
            if (node >= BKDReader.this.leafNodeOffset) {
                return 1;
            }
            int leftCount = this.getNumLeavesSlow(node * 2);
            int rightCount = this.getNumLeavesSlow(node * 2 + 1);
            return leftCount + rightCount;
        }

        private void readNodeData(boolean isLeft) {
            if (this.splitPackedValueStack[this.level] == null) {
                this.splitPackedValueStack[this.level] = new byte[BKDReader.this.config.packedIndexBytesLength];
            }
            System.arraycopy(this.negativeDeltas, (this.level - 1) * BKDReader.this.config.numIndexDims, this.negativeDeltas, this.level * BKDReader.this.config.numIndexDims, BKDReader.this.config.numIndexDims);
            assert (this.splitDim != -1);
            this.negativeDeltas[this.level * BKDReader.this.config.numIndexDims + this.splitDim] = isLeft;
            try {
                this.leafBlockFPStack[this.level] = this.leafBlockFPStack[this.level - 1];
                if (!isLeft) {
                    int n = this.level;
                    this.leafBlockFPStack[n] = this.leafBlockFPStack[n] + this.in.readVLong();
                }
                if (this.isLeafNode()) {
                    this.splitDim = -1;
                } else {
                    int code = this.in.readVInt();
                    this.splitDims[this.level] = this.splitDim = code % BKDReader.this.config.numIndexDims;
                    int prefix = (code /= BKDReader.this.config.numIndexDims) % (1 + BKDReader.this.config.bytesPerDim);
                    int suffix = BKDReader.this.config.bytesPerDim - prefix;
                    if (this.splitValuesStack[this.level] == null) {
                        this.splitValuesStack[this.level] = new byte[BKDReader.this.config.packedIndexBytesLength];
                    }
                    System.arraycopy(this.splitValuesStack[this.level - 1], 0, this.splitValuesStack[this.level], 0, BKDReader.this.config.packedIndexBytesLength);
                    if (suffix > 0) {
                        int firstDiffByteDelta = code / (1 + BKDReader.this.config.bytesPerDim);
                        if (this.negativeDeltas[this.level * BKDReader.this.config.numIndexDims + this.splitDim]) {
                            firstDiffByteDelta = -firstDiffByteDelta;
                        }
                        int oldByte = this.splitValuesStack[this.level][this.splitDim * BKDReader.this.config.bytesPerDim + prefix] & 0xFF;
                        this.splitValuesStack[this.level][this.splitDim * BKDReader.this.config.bytesPerDim + prefix] = (byte)(oldByte + firstDiffByteDelta);
                        this.in.readBytes(this.splitValuesStack[this.level], this.splitDim * BKDReader.this.config.bytesPerDim + prefix + 1, suffix - 1);
                    }
                    int leftNumBytes = this.nodeID * 2 < BKDReader.this.leafNodeOffset ? this.in.readVInt() : 0;
                    this.rightNodePositions[this.level] = Math.toIntExact(this.in.getFilePointer()) + leftNumBytes;
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }
}

