// Copyright ©2025 blackshirt.
// Use of this source code is governed by an MIT license
// that can be found in the LICENSE file.
//
module ascon

import arrays
import encoding.hex

struct HashTest {
	count int
	msg   string
	md    string
}

// This test material mostly taken and adapted from Known-Answer-Test (KAT) of reference implementation.
// See at https://github.com/ascon/ascon-c/blob/main/crypto_hash/asconhash256/LWC_HASH_KAT_128_256.txt
fn test_hash256_sum_chunked() ! {
	item := HashTest{
		count: 500
		msg:   '000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2'
		md:    '8174F5E8DFC9441DBB7A183D56F431B1AEF9513E747A3878BB806BAE27655E3E'
	}
	msg := hex.decode(item.msg)!
	md := hex.decode(item.md)!
	// One shoot sum256 function
	md0 := sum256(msg)
	assert md0 == md

	// Digest based
	mut h := new_hash256()
	expected_md := h.sum(msg)
	assert expected_md == md

	// with multistep
	h.reset()
	_ := h.Digest.absorb(msg)
	h.Digest.finish()
	mut dst := []u8{len: hash256_size}
	h.Digest.squeeze(mut dst)
	assert dst == md

	// with splitted message
	msg0 := msg[0..200]
	msg1 := msg[200..400]
	msg2 := msg[400..]
	h.reset()
	_ := h.Digest.absorb(msg0)
	_ := h.Digest.absorb(msg1)
	_ := h.Digest.absorb(msg2)
	h.Digest.finish()
	h.Digest.squeeze(mut dst)
	assert dst == md

	// with arrays chunk
	h.reset()
	chunks := arrays.chunk[u8](msg, 200)
	mut n := 0
	for chunk in chunks {
		n += h.Digest.absorb(chunk)
	}
	assert n == msg.len
	h.Digest.finish()
	h.Digest.squeeze(mut dst)
	assert dst == md

	// with sum
	h.reset()
	for chunk in chunks {
		n += h.write(chunk)!
	}
	chunked_md := h.sum([]u8{})
	assert chunked_md == md
}

fn test_hash256_sum_kat() ! {
	for item in ascon_hash256_test_data {
		msg := hex.decode(item.msg)!
		md := hex.decode(item.md)!
		out := sum256(msg)
		assert out == md

		// work with Digest opaque
		mut h := new_hash256()
		exp_md := h.sum(msg)
		assert exp_md == md

		// Lets work in streaming-way
		chunks := arrays.chunk[u8](msg, 7)
		h.reset()
		mut tot := 0
		for chunk in chunks {
			n := h.write(chunk)!
			tot += n
		}
		assert msg.len == tot

		chunked_md := h.sum([]u8{})
		assert chunked_md == md
	}
}

const ascon_hash256_test_data = [
	HashTest{
		count: 1
		msg:   ''
		md:    '0b3be5850f2f6b98caf29f8fdea89b64a1fa70aa249b8f839bd53baa304d92b2'
	},
	HashTest{
		count: 2
		msg:   '00'
		md:    '0728621035af3ed2bca03bf6fde900f9456f5330e4b5ee23e7f6a1e70291bc80'
	},
	HashTest{
		count: 3
		msg:   '0001'
		md:    '6115e7c9c4081c2797fc8fe1bc57a836afa1c5381e556dd583860ca2dfb48dd2'
	},
	HashTest{
		count: 4
		msg:   '000102'
		md:    '265ab89a609f5a05dca57e83fbba700f9a2d2c4211ba4cc9f0a1a369e17b915c'
	},
	HashTest{
		count: 5
		msg:   '00010203'
		md:    'd7e4c7ed9b8a325cd08b9ef259f8877054ecd8304fe1b2d7fd847137df6727ee'
	},
	HashTest{
		count: 6
		msg:   '0001020304'
		md:    'c7b28962d4f5c2211f466f83d3c57ae1504387e2a326949747a8376447a6bb51'
	},
	HashTest{
		count: 7
		msg:   '000102030405'
		md:    'dc0c6748af8ffe63e1084aa3e5786a194685c88c21348b29e184fb50409703bc'
	},
	HashTest{
		count: 8
		msg:   '00010203040506'
		md:    '3e4d273ba69b3b9c53216107e88b75cdbeedbcbf8faf0219c3928ab62b116577'
	},
	HashTest{
		count: 9
		msg:   '0001020304050607'
		md:    'b88e497ae8e6fb641b87ef622eb8f2fca0ed95383f7ffebe167acf1099ba764f'
	},
	HashTest{
		count: 10
		msg:   '000102030405060708'
		md:    '94269C30E0296E1EC86655041841823EFA1927F520FD58C8E9BCE6197878C1A6'
	},
	HashTest{
		count: 500
		msg:   '000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2'
		md:    '8174F5E8DFC9441DBB7A183D56F431B1AEF9513E747A3878BB806BAE27655E3E'
	},
	HashTest{
		count: 501
		msg:   '000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3'
		md:    'E73D4DDB9D248BF2C0F8D49892D7455A4C3053153DE7F79BA4487C7D823F605C'
	},
	HashTest{
		count: 502
		msg:   '000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4'
		md:    'B73FA04079263F6A733B67466552784B436138F41F80B72C4D5D03934B72207D'
	},
]
